/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CamelServiceTest.java 20667 2011-01-03 17:07:21Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.service.test;

import java.io.InputStream;

import javax.jms.ConnectionFactory;

import junit.framework.Assert;
import junit.framework.TestCase;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.camel.CamelContext;
import org.apache.camel.Endpoint;
import org.apache.camel.component.jms.JmsComponent;
import org.apache.camel.impl.ServiceSupport;
import org.osgi.framework.BundleContext;
import org.ow2.jonas.camel.service.impl.CamelService;

/**
 * @author Guillaume Porcher
 */
public class CamelServiceTest extends TestCase {

    /**
     * Bundle context instance used by this test.
     */
    private BundleContext bundleContext;

    /**
     * Camel service instance used by this test.
     */
    private CamelService camelService;

    /**
     * Create an instance of this.camelService.
     */
    @Override
    protected void setUp() throws Exception {
        super.setUp();
        this.bundleContext = new MockBundleContext();
        this.camelService = new CamelService(this.bundleContext);
        this.camelService.start();
    }

    /**
     * @see junit.framework.TestCase#tearDown()
     */
    @Override
    protected void tearDown() throws Exception {
        super.tearDown();
        this.camelService.stop();
    }

    /**
     * Test method for
     * {@link org.ow2.jonas.camel.service.api.ICamelService#getContextNames()}.
     * 
     * @throws Exception
     */
    public void testGetContextNames() throws Exception {
        // Check context names is empty
        Assert.assertNotNull(this.camelService.getContextNames());
        Assert.assertTrue(this.camelService.getContextNames().isEmpty());

        // start new context
        final String contextName = this.camelService.startNewContext(this.bundleContext);

        // check context names contains new context
        Assert.assertEquals(1, this.camelService.getContextNames().size());
        Assert.assertEquals(contextName, this.camelService.getContextNames().get(0));

        // stop context
        this.camelService.stop(contextName);

        // context is removed
        Assert.assertTrue(this.camelService.getContextNames().isEmpty());
    }

    /**
     * Test method for
     * {@link org.ow2.jonas.camel.service.api.ICamelService#getCamelContext(java.lang.String)}
     * .
     */
    public void testGetCamelContext() throws Exception {
        // Check context names is empty
        Assert.assertNotNull(this.camelService.getContextNames());
        Assert.assertTrue(this.camelService.getContextNames().isEmpty());

        // start new context
        final String contextName = this.camelService.startNewContext(this.bundleContext);

        // get camelContext:
        final CamelContext camelContext = this.camelService.getCamelContext(contextName);

        Assert.assertNotNull(camelContext);
        Assert.assertEquals(contextName, camelContext.getName());

        final CamelContext invalidCamelContext = this.camelService.getCamelContext("invalid");
        Assert.assertNull(invalidCamelContext);
    }

    /**
     * Test method for
     * {@link org.ow2.jonas.camel.service.api.ICamelService#startNewContext()}.
     * 
     * @throws Exception
     */
    public void testStartNewContext() throws Exception {
        // start new context
        final String contextName = this.camelService.startNewContext(this.bundleContext);

        Assert.assertNotNull(contextName);

        // get camelContext:
        final CamelContext camelContext = this.camelService.getCamelContext(contextName);

        Assert.assertNotNull(camelContext);
        Assert.assertTrue(((ServiceSupport) camelContext).isStarted());
    }

    /**
     * Test method for
     * {@link org.ow2.jonas.camel.service.api.ICamelService#startAndGetNewContext()}
     * .
     * 
     * @throws Exception
     */
    public void testStartAndGetNewContext() throws Exception {
        // start new context
        final CamelContext camelContext = this.camelService.startAndGetNewContext(this.bundleContext);

        Assert.assertNotNull(camelContext);

        // get camelContext:
        Assert.assertEquals(camelContext, this.camelService.getCamelContext(camelContext.getName()));
    }

    /**
     * Test method for
     * {@link org.ow2.jonas.camel.service.api.ICamelService#stop(java.lang.String)}
     * .
     * 
     * @throws Exception
     */
    public void testStop() throws Exception {
        // start new context
        final String contextName = this.camelService.startNewContext(this.bundleContext);

        Assert.assertNotNull(contextName);

        // get camelContext:
        final CamelContext camelContext = this.camelService.getCamelContext(contextName);

        Assert.assertNotNull(camelContext);
        Assert.assertTrue(((ServiceSupport) camelContext).isStarted());

        // stop context
        this.camelService.stop(contextName);
        Assert.assertTrue(((ServiceSupport) camelContext).isStopped());
    }

    /**
     * Test method for
     * {@link org.ow2.jonas.camel.service.api.ICamelService#addRegistry(java.io.InputStream, java.lang.String)}
     * .
     * 
     * @throws Exception
     */
    public void testAddRegistry() throws Exception {
        InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry.xml");

        // start new context
        final String contextName = this.camelService.startNewContext(this.bundleContext);

        Assert.assertNotNull(contextName);

        CamelContext ctxt = this.camelService.getCamelContext(contextName);

        ConnectionFactory connectionFactory = new ActiveMQConnectionFactory("vm://localhost?broker.persistent=false");
        ctxt.addComponent("test-jms", JmsComponent.jmsComponentAutoAcknowledge(connectionFactory));

        this.camelService.addRegistry(stream, contextName);

        Endpoint endpoint = (Endpoint) ctxt.getRegistry().lookup("queue");

        Assert.assertNotNull(endpoint);

        Assert.assertEquals("test-jms://queue:sampleQueue", endpoint.getEndpointUri());
    }

    /**
     * Test method for
     * {@link org.ow2.jonas.camel.service.api.ICamelService#removeRegistry(java.io.InputStream, java.lang.String)}
     * .
     * 
     * @throws Exception
     */
    public void testRemoveRegistry() throws Exception {
        InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry.xml");

        // start new context
        final String contextName = this.camelService.startNewContext(this.bundleContext);

        Assert.assertNotNull(contextName);

        CamelContext ctxt = this.camelService.getCamelContext(contextName);

        ConnectionFactory connectionFactory = new ActiveMQConnectionFactory("vm://localhost?broker.persistent=false");
        ctxt.addComponent("test-jms", JmsComponent.jmsComponentAutoAcknowledge(connectionFactory));

        this.camelService.addRegistry(stream, contextName);

        stream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry.xml");

        this.camelService.removeRegistry(stream, contextName);

        Object endpoint = ctxt.getRegistry().lookup("queue");

        Assert.assertNull(endpoint);
    }

    /**
     * Test method for
     * {@link org.ow2.jonas.camel.service.api.ICamelService#addComponent(java.lang.String, org.apache.camel.Component, java.lang.String)}
     * .
     */
    /*
     * public void testAddComponent() { Assert.fail("Not yet implemented"); }
     */

}
