/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2009-2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ExampleCamelServlet.java 21289 2011-05-19 22:10:27Z renaultgu $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.example.servlet;

import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.http.HttpContext;
import org.osgi.service.http.HttpService;
import org.ow2.jonas.camel.component.CamelServlet;
import org.ow2.jonas.camel.component.RouteBuilderComponent;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import javax.servlet.ServletRequest;
import java.util.Dictionary;
import java.util.Hashtable;

/**
 * Example to show how to declare a servlet to handle HTTP requests
 */
public class ExampleCamelServlet extends RouteBuilderComponent {

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(ExampleCamelServlet.class);

    /**
     * Servlet path.
     */
    public static final String SERVLET_PATH = "/camelServlet";

    /**
     * Camel internal endpoint on which the http request is forwarded from
     * the servlet.
     */
    public static final String SERVLET_ENDPOINT = "direct:fromServlet";

    public ExampleCamelServlet(BundleContext bundleContext) {
        super(bundleContext);
    }

    /**
     * Register the Consumer servlet.
     */
    @Override
    public void start() throws Throwable {
        super.start();

        Dictionary<String, String> initParams;
        final HttpService httpService = getService(HttpService.class);

        // create a default context to share between registrations
        final HttpContext httpContext = httpService.createDefaultHttpContext();

        // register the getDocument servlet
        initParams = new Hashtable<String, String>();
        initParams.put("matchOnUriPrefix", "false");
        initParams.put("servlet-name", ExampleCamelServlet.SERVLET_PATH);

        httpService.registerServlet(ExampleCamelServlet.SERVLET_PATH,
                new CamelServlet(this.getContext(), ExampleCamelServlet.SERVLET_ENDPOINT),
                initParams,
                httpContext);

    }

    @Override
    public void configure() throws Exception {
        super.configure();

        // Handle the servlet request
        this.from(ExampleCamelServlet.SERVLET_ENDPOINT)
            .process(new Processor() {
                public void process(Exchange exchange) throws Exception {
                    ServletRequest request = exchange.getProperty(CamelServlet.REQUEST_PROPERTY_NAME,
                            ServletRequest.class);
                    logger.info("The request content length property of the servlet request is {0}",
                            request.getContentLength());
                    exchange.getOut()
                            .setBody("Hello from the CamelServlet endpoint... Thanks for using JOnAS/Camel");
                }
            });

    }

    /**
     * Get an OSGi service
     *
     * @param serviceType
     * @return the expected OSGi service
     */
    private <T> T getService(Class<T> serviceType) {
        final String serviceName = serviceType.getName();
        ServiceReference serviceReference = this.bundleContext.getServiceReference(serviceName);

        if (serviceReference != null) {
            Object service = this.bundleContext.getService(serviceReference);
            if (service != null) {
                if (!serviceType.isInstance(service)) {
                    throw new IllegalStateException("OSGi service " + service + " not instance of " + serviceName);
                }

                return (T) service;
            }
        }

        throw new IllegalStateException("Cannot find OSGi service " + serviceName);
    }

}
