/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: RegistryWrapperTest.java 21372 2011-06-07 08:58:24Z ardaaydin $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.registry.xml.test;

import java.io.InputStream;

import javax.jms.ConnectionFactory;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.camel.CamelContext;
import org.apache.camel.component.jms.JmsComponent;
import org.apache.camel.impl.DefaultCamelContext;
import org.apache.camel.spi.Registry;
import org.ow2.jonas.camel.registry.xml.RegistryWrapper;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.testng.Assert;
import org.testng.annotations.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

public class RegistryWrapperTest {

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(this.getClass());

    /**
     * The registry wrapper.
     */
    RegistryWrapper registryWrapper;

    /**
     * The camel context.
     */

    CamelContext ctxt;

    @Test
    public void registryPackingTest() throws Exception {
        // Create the registry wrapper.
        this.registryWrapper = new RegistryWrapper();
        // Get a Camel context.
        this.ctxt = new DefaultCamelContext();
        // Assign the context to the registry.
        this.registryWrapper.createRegistryAndAssignToContext((DefaultCamelContext) this.ctxt);
        // Create JMS connection factory.
        ConnectionFactory connectionFactory = new ActiveMQConnectionFactory("vm://localhost?broker.persistent=false");
        // register JMS component that will handle technical name starting with
        // test-jms:
        this.ctxt.addComponent("test-jms", JmsComponent.jmsComponentAutoAcknowledge(connectionFactory));
        // Start the context
        this.ctxt.start();
        // Load registry file.
        InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry.xml");
        // Assign the registry to the context.
        this.registryWrapper.addToTheRegistry(stream);
        this.checkEntries("registry.xml", this.registryWrapper);
    }

    private void checkEntries(final String file, final RegistryWrapper registryWrapper) throws Exception {
        InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream(file);
        Document dom = this.parseXmlFile(stream);
        this.validate(registryWrapper, dom);
    }

    private Document parseXmlFile(final InputStream fileInstream) throws Exception {
        // get the factory
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

        // Using factory get an instance of document builder
        DocumentBuilder db = dbf.newDocumentBuilder();

        // parse using builder to get DOM representation of the XML file
        Document dom = db.parse(fileInstream);
        return dom;
    }

    private void validate(final RegistryWrapper registryWrapper, final Document dom) {
        // get the root element
        Element docEle = dom.getDocumentElement();

        // get a nodelist of elements
        NodeList nl = docEle.getElementsByTagName("entry");
        if (nl != null && nl.getLength() > 0) {
            for (int i = 0; i < nl.getLength(); i++) {

                // get the employee element
                Element el = (Element) nl.item(i);

                this.checkEntry(registryWrapper, el);

            }
        }
    }

    private void checkEntry(final RegistryWrapper registryWrapper, final Element el) {
        // for each <employee> element get text or int values of
        // name ,id, age and name
        String logicalName = this.getTextValue(el, "logicalName");
        String technicalName = this.getTextValue(el, "technicalName");
        Object regvalue = (Object) registryWrapper.getRegistry().lookup(logicalName);
        this.logger.info("logicalName: {0} .\n\t Technical name: {1} .\n\t Registered as: \n\t", logicalName, technicalName,
            regvalue);
        Assert.assertNotNull(regvalue, "Check that the registered entry is the same as expected");

    }

    /**
     * I take a xml element and the tag name, look for the tag and get the text
     * content i.e for <employee><name>John</name></employee> xml snippet if the
     * Element points to employee node and tagName is 'name' I will return John
     */
    private String getTextValue(final Element ele, final String tagName) {
        String textVal = null;
        NodeList nl = ele.getElementsByTagName(tagName);
        if (nl != null && nl.getLength() > 0) {
            Element el = (Element) nl.item(0);
            textVal = el.getFirstChild().getNodeValue();
        }

        return textVal;
    }

    @Test(dependsOnMethods = {"registryPackingTest"})
    public void removeValuesTest() throws Exception {
        this.logger.info("Remove all entries of a xml file from the warpped registry");
        // Load registry file.
        InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry.xml");
        // Unregister entries.
        this.registryWrapper.removeFromTheRegistry(stream);
        Assert.assertEquals(this.registryWrapper.getRegistry().isEmpty(), true, "The registry is empty.");
    }

    @Test(dependsOnMethods = {"removeValuesTest"})
    public void duplicateEntryTest() throws Exception {
        this.logger.info("fill the registry + try to duplicate entry");
        InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry.xml");
        this.registryWrapper.addToTheRegistry(stream);
        stream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry_2.xml");
        try {
            this.registryWrapper.addToTheRegistry(stream);
        } catch (IllegalStateException e) {
            Assert.fail("Adding duplicate entries should not be sucessful.");
        } catch (Exception e) {
            Assert.fail("Unexpected failure when filling the registry " + e);
        }
        // Verify that the first entry is in the registry but not the second one.
        Assert.assertEquals(this.registryWrapper.getRegistry().get("queue"), "test-jms:queue:sampleQueue", "First registry value is acknowledged");

    }

    @Test(dependsOnMethods = {"duplicateEntryTest"})
    public void createAndPackRegistryTest() throws Exception {
        this.logger.info("Init registry wrapper " + "+ create registry using a context + " + "fill the registry + " + "tests ");
        // Get the Camel context.
        this.ctxt = new DefaultCamelContext();
        // Create the registry wrapper
        this.registryWrapper = new RegistryWrapper();
        // Create the registry, and assign the context to it.
        Registry reg = this.registryWrapper.createRegistry(this.ctxt);
        // Assign the registry to the context.
        ((DefaultCamelContext) this.ctxt).setRegistry(reg);
        // Create JMS connection factory.
        ConnectionFactory connectionFactory = new ActiveMQConnectionFactory("vm://localhost?broker.persistent=false");
        // register JMS component that will handle technical name starting with
        // test-jms:
        this.ctxt.addComponent("test-jms", JmsComponent.jmsComponentAutoAcknowledge(connectionFactory));
        // Start the context.
        this.ctxt.start();
        // Load the registry file.
        InputStream stream = Thread.currentThread().getContextClassLoader().getResourceAsStream("registry.xml");
        // Register entries.
        this.registryWrapper.addToTheRegistry(stream);
        this.checkEntries("registry.xml", this.registryWrapper);
        this.removeValuesTest();
    }
}
