/**
 * Copyright 2009-2012 Serli
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

importPackage(org.osgi.framework);

/**
 * BundleInfo acts as a facade against a real Bundle object.
 * @param identifier bundle identifier (ID or symbolicName + version couple)
 * @constructor
 */
function BundleInfo(identifier) {

    this.bundle = this.findBundle(identifier);

    if (this.bundle != null) {
        this.headers = Converters.convertDictionaryToObject(this.bundle.headers);
        this.bundleId = this.bundle.bundleId;
        this.state = this.bundle.state;
        this.location = this.bundle.location;
        this.symbolicName = this.bundle.symbolicName;
        this.lastModified = this.bundle.lastModified;
        if (this.bundle.version != null) {
            this.version = this.bundle.version.toString();
        }
    }

    this.exists = (this.bundle != null);
    this.uninstalled = this.checkState([Bundle.UNINSTALLED]);
    this.installed = this.checkState([Bundle.INSTALLED, Bundle.RESOLVED, Bundle.STARTING, Bundle.STOPPING, Bundle.ACTIVE]);
    this.resolved = this.checkState([Bundle.RESOLVED, Bundle.STARTING, Bundle.STOPPING, Bundle.ACTIVE]);
    this.starting = this.checkState([Bundle.STARTING]);
    this.stopping = this.checkState([Bundle.STOPPING]);
    this.active = this.checkState([Bundle.ACTIVE]);

}

BundleInfo.prototype.findBundle = function(identifier) {

    // Is the identifier the BundleID ?
    if (typeof identifier == 'number') {
        var b = bundleContext.getBundle(Number(identifier));
        if (b) {
            return b;
        }
        // The given numerical ID do nto matches an installed Bundle
        return null;
    }

    // Otherwise it is something of the format:
    // symbolicName[:bundleVersion]

    var fragments = String(identifier).split(":");
    var name = fragments[0];
    var version = fragments[1];

    var bundles = bundleContext.bundles;
    for (var i = 0; i < bundles.length ; i++) {
        var b = bundles[i];
        if (name == b.symbolicName) {
            if (version != undefined) {
                // Version matching is required
                if (version == b.version) {
                    return b;
                }
            } else {
                // No constraining version, so matching only use the symbolic name
                return b;
            }
        }
    }

    return null;

}

BundleInfo.prototype.checkState = function(expected) {
    if (this.bundle == null) {
        return false;
    }

    for (var i = 0; i < expected.length; i++) {
        if (this.bundle.state == expected[i]) {
            return true;
        }
    }
    return false;
}
