/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: AbstractRegistryComponent.java 17429 2009-05-20 12:31:30Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.camel.registry.api;

import java.io.InputStream;

import org.apache.camel.CamelContext;
import org.apache.camel.Component;
import org.apache.camel.Endpoint;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public abstract class AbstractRegistryComponent implements Component {

    /**
     * The logger.
     */
    private static Log logger = LogFactory.getLog(AbstractRegistryComponent.class);

    /**
     * the camel context.
     */
    private CamelContext context;

    /**
     * Return the camel context.
     * 
     * @return the camel context.
     */
    public CamelContext getCamelContext() {
        return this.context;
    }

    /**
     * Set the camel context.
     * 
     * @param context the camel context to set.
     */
    public void setCamelContext(final CamelContext context) {
        this.context = context;
    }

    /**
     * Create the AbstractRegistryComponent endpoint.
     */
    public Endpoint createEndpoint(final String name) throws Exception {
        final String logicalAddress = name.substring(name.indexOf(':') + 1);
        final String technicalAddress = this.lookup(logicalAddress);

        final String scheme;
        final int schemeStart = technicalAddress.indexOf(':');
        if (schemeStart > 0) {
            scheme = technicalAddress.substring(0, schemeStart);
        } else {
            scheme = technicalAddress;
        }

        AbstractRegistryComponent.logger.debug("AbstractRegistryComponent creating endpoint: name is " + name
            + ", logicalAddress is " + logicalAddress + ", technicalAddress is " + technicalAddress + ", scheme is " + scheme);

        return this.context.getComponent(scheme).createEndpoint(technicalAddress);
    }

    /**
     * Look up a definition from the registry.
     * 
     * @param name the name of the definition to lookup.
     * @return the technical definition.
     */
    protected abstract String lookup(String name) throws Exception;

    /**
     * Add values to the registry.
     * 
     * @param input InputStream containing the values to add.
     */
    public abstract void addValues(final InputStream input) throws Exception;

    /**
     * Remove values from the registry.
     * 
     * @param input InputStream containing the values to remove.
     */
    public abstract void removeValues(final InputStream input) throws Exception;
}
