/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2004-2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Carol.java 12645 2008-01-22 14:39:16Z sauthieg $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.ant.jonasbase;

import java.io.File;

import org.ow2.jonas.ant.JOnASBaseTask;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.taskdefs.Echo;

/**
 * Defines properties for carol.properties.
 *
 * @author Florent Benoit
 */
public class Carol extends Tasks {
    /**
     * Inner class allowing to echo message concerning deprecated jeremie.
     * Did try to use JEcho but didn't manage (see setJermiePort method).
     */
    public class MyEcho extends Task implements BaseTaskItf {

        /**
         * {@inheritDoc}
         */
        @Override
        public void setTaskName(final String arg0) {
            super.setTaskName(arg0);
            echo.setTaskName(arg0);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public void setProject(final Project arg0) {
            super.setProject(arg0);
            echo.setProject(arg0);
        }

        /**
         * Wrapped {@link Echo} task.
         */
        private Echo echo = new Echo();

        /**
         * {@inheritDoc}
         */
        @Override
        public void execute() {
            echo.setMessage(INFO + "Attribute 'jeremie' is deprecated."
                    + " This protocol is no more supported by JOnAS.");
            echo.execute();
        }

        public String getLogInfo() {
            return null;
        }

        public void setConfigurationFile(final String configurationFile) {
        }

        public void setDestDir(final File destDir) {
        }

        public void setJonasRoot(final File jonasRoot) {
        }

        public void setLogInfo(final String logInfo) {
        }

    }

    /**
     * Info for the logger.
     */
    private static final String INFO = "[Carol] ";

    /**
     * Protocols property.
     */
    private static final String PROTOCOLS_PROPERTY = "carol.protocols";

    /**
     * Default protocols.
     */
    private static final String DEFAULT_PROTOCOLS = "jrmp";

    /**
     * Default Host.
     */
    private static final String DEFAULT_HOST = "localhost";

    /**
     * Default protocol header.
     */
    private static final String PROTOCOL_HEADER = "://";

    /**
     * Default jrmp port number.
     */
    private static final String DEFAULT_JRMP_PORT = "1099";

    /**
     * Default iiop port number.
     */
    private static final String DEFAULT_IIOP_PORT = "2001";

    /**
     * Default irmi port number.
     */
    private static final String DEFAULT_IRMI_PORT = "1098";

    /**
     * Optimization.
     */
    private static final String OPTIMIZATION_PROPERTY = "carol.jvm.rmi.local.call=";

    /**
     * Optimization (off).
     */
    private static final String OPTIMIZATION_OFF = OPTIMIZATION_PROPERTY
            + "false";

    /**
     * Optimization (on).
     */
    private static final String OPTIMIZATION_ON = OPTIMIZATION_PROPERTY
            + "true";

    /**
     * Local Registry.
     */
    private static final String LOCALREGISTRY_PROPERTY = "carol.jvm.rmi.local.registry=";

    /**
     * Local Registry(off).
     */
    private static final String LOCALREGISTRY_OFF = LOCALREGISTRY_PROPERTY
            + "false";

    /**
     * Local Registry (on).
     */
    private static final String LOCALREGISTRY_ON = LOCALREGISTRY_PROPERTY
            + "true";

    /**
     * Start CMI.
     */
    public static final String CMI_STARTED = "carol.start.cmi=";

    /**
     * Start CMI (on).
     */
    public static final String CMI_STARTED_ON = CMI_STARTED + "true";

    /**
     * Start CMI (off).
     */
    public static final String CMI_STARTED_OFF = CMI_STARTED + "false";

    /**
     * Sequence of associations between a protocol and a list of provider URLs.
     */
    public static final String SEQ_PROTOCOLS_TO_PROVIDER_URLS = "cmi.server.provider.urls=";

    /**
     * Default constructor.
     */
    public Carol() {
        super();
    }

    /**
     * Set the port for a protocol.
     *
     * @param protocolName
     *            name of the protocol
     * @param hostName
     *            for the protocol
     * @param portNumber
     *            port for a given protocol
     * @param protocolUrl
     *            URL for the protocol (if any)
     * @param tokenToReplace
     *            of the protocol to replace
     */
    private void setURL(final String protocolName, final String hostName,
            final String portNumber, final String protocolUrl, final String tokenToReplace) {
        String url = null;

        // Token to replace
        String token = protocolName + PROTOCOL_HEADER + DEFAULT_HOST + ":" + tokenToReplace;

        if (protocolUrl != null) {
            url = protocolUrl;
        } else {
            url = protocolName + PROTOCOL_HEADER + hostName + ":" + portNumber;
        }
        JReplace propertyReplace = new JReplace();
        propertyReplace.setLogInfo(INFO + "Setting URL for protocol '"
                + protocolName + "' to : " + url);
        propertyReplace.setConfigurationFile(JOnASBaseTask.CAROL_CONF_FILE);
        propertyReplace.setToken(token);
        propertyReplace.setValue(url);
        addTask(propertyReplace);
    }

    /**
     * Set the port for JRMP.
     *
     * @param portNumber
     *            port for JRMP
     */
    public void setJrmpPort(final String portNumber) {
        setURL("rmi", DEFAULT_HOST, portNumber, null, DEFAULT_JRMP_PORT);
    }

    /**
     * Set the port for JEREMIE.
     *
     * @param portNumber
     *            port for JEREMIE
     * @deprecated
     */
    public void setJeremiePort(final String portNumber) {
        this.addTask(new MyEcho());
    }

    /**
     * Set the port for IIOP.
     *
     * @param portNumber
     *            port for IIOP
     */
    public void setIiopPort(final String portNumber) {
        setURL("iiop", DEFAULT_HOST, portNumber, null, DEFAULT_IIOP_PORT);
    }

    /**
     * Set the port for IRMI.
     *
     * @param portNumber
     *            port for IRMI
     */
    public void setIrmiPort(final String portNumber) {
        setURL("rmi", DEFAULT_HOST, portNumber, null, DEFAULT_IRMI_PORT);
    }

    /**
     * Set the port for all protocols.
     *
     * @param portNumber
     *            port for all protocols
     */
    public void setDefaultPort(final String portNumber) {
        setJrmpPort(portNumber);
        setIiopPort(portNumber);
        setIrmiPort(portNumber);
    }

    /**
     * Set the initial protocols when JOnAS start.
     *
     * @param protocols
     *            comma separated list of protocols
     */
    public void setProtocols(final String protocols) {
        JReplace propertyReplace = new JReplace();
        propertyReplace.setConfigurationFile(JOnASBaseTask.CAROL_CONF_FILE);
        propertyReplace.setToken(PROTOCOLS_PROPERTY + "=" + DEFAULT_PROTOCOLS);
        propertyReplace.setValue(PROTOCOLS_PROPERTY + "=" + protocols);
        propertyReplace
                .setLogInfo(INFO + "Setting protocols to : " + protocols);
        addTask(propertyReplace);
    }

    /**
     * Enable or disable optimization.
     *
     * @param enabled
     *            (true or false)
     */
    public void setJrmpOptimization(final boolean enabled) {
        // Change only if needed
        if (!enabled) {
            return;
        }
        JReplace propertyReplace = new JReplace();
        propertyReplace.setConfigurationFile(JOnASBaseTask.CAROL_CONF_FILE);
        propertyReplace.setToken(OPTIMIZATION_OFF);
        propertyReplace.setValue(OPTIMIZATION_ON);
        propertyReplace.setLogInfo(INFO + "Enable JRMP optimization: "
                + enabled);
        addTask(propertyReplace);
    }

    /**
     * Enable or disable local registry.
     *
     * @param enabled
     *            (true or false)
     */
    public void setLocalRegistry(final boolean enabled) {
        // Change only if needed
        if (!enabled) {
            return;
        }
        JReplace propertyReplace = new JReplace();
        propertyReplace.setConfigurationFile(JOnASBaseTask.CAROL_CONF_FILE);
        propertyReplace.setToken(LOCALREGISTRY_OFF);
        propertyReplace.setValue(LOCALREGISTRY_ON);
        propertyReplace.setLogInfo(INFO + "Enable local registry  setting: "
                + enabled);
        addTask(propertyReplace);
    }

    /**
     * Enable or disable the use of CMI.
     * @param enabled (true or false)
     */
    public void setCmiStarted(final boolean enabled) {
        // Change only if needed
        if (!enabled) {
            return;
        }
        JReplace propertyReplace = new JReplace();
        propertyReplace.setConfigurationFile(JOnASBaseTask.CAROL_CONF_FILE);
        propertyReplace.setToken(CMI_STARTED_OFF);
        propertyReplace.setValue(CMI_STARTED_ON);
        propertyReplace.setLogInfo(INFO + "Enable start cmi  setting: "
                + enabled);
        addTask(propertyReplace);
    }

    /**
     * Set the sequence of protocols and provider urls.
     * @param seqProtocol2ProviderURLs a sequence of protocols and provider urls
     */
    public void setClusterViewProviderUrls(final String seqProtocol2ProviderURLs) {
        JReplace propertyReplace = new JReplace();
        propertyReplace.setConfigurationFile(JOnASBaseTask.CAROL_CONF_FILE);
        propertyReplace.setToken(SEQ_PROTOCOLS_TO_PROVIDER_URLS);
        propertyReplace.setValue(SEQ_PROTOCOLS_TO_PROVIDER_URLS + seqProtocol2ProviderURLs);
        propertyReplace
                .setLogInfo(INFO + "Setting sequence of protocols and provider urls to : " + seqProtocol2ProviderURLs);
        addTask(propertyReplace);
    }

}
