/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DefaultWeldService.java 19316 2010-03-02 16:24:45Z sauthieg $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.cdi.weld.internal;

import java.net.URL;
import java.util.Collection;

import org.jboss.weld.bootstrap.WeldBootstrap;
import org.jboss.weld.bootstrap.api.Bootstrap;
import org.jboss.weld.bootstrap.api.Environments;
import org.jboss.weld.bootstrap.spi.BeanDeploymentArchive;
import org.jboss.weld.bootstrap.spi.Deployment;
import org.jboss.weld.context.api.BeanStore;
import org.jboss.weld.context.api.helpers.ConcurrentHashMapBeanStore;
import org.ow2.jonas.cdi.weld.IWeldService;
import org.ow2.jonas.cdi.weld.internal.deployment.DefaultDeploymentBuilder;
import org.ow2.jonas.lib.service.AbsServiceImpl;
import org.ow2.jonas.service.ServiceException;
import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.archive.api.IArchive;

/**
 * Weld CDI service implementation.
 * @author Guillaume Sauthier
 */
public class DefaultWeldService extends AbsServiceImpl implements IWeldService {

    /**
     * Abstract start-up method to be implemented by sub-classes.
     *
     * @throws org.ow2.jonas.service.ServiceException
     *          service start-up failed
     */
    @Override
    protected void doStart() throws ServiceException {
        // Nothing to do
    }

    /**
     * Abstract method for service stopping to be implemented by sub-classes.
     *
     * @throws org.ow2.jonas.service.ServiceException
     *          service stopping failed
     */
    @Override
    protected void doStop() throws ServiceException {
        // Nothing to do
    }

    /**
     * Detect if the given {@code IArchive} contains a {@code beans.xml}.
     * Checked location includes:
     * <ul>
     * <li>{@code WEB-INF/beans.xml}</li>
     * <li>{@code META-INF/beans.xml}</li>
     * </ul>
     *
     * @param archive explored archive
     * @return {@code true} if the given archive is CDI enabled, {@code false} otherwise.
     */
    public boolean isCdiEnabled(IArchive archive) {

        try {

            // Search in usual jar place ...
            URL resource = archive.getResource(DefaultDeploymentBuilder.METAINF_BEANS_XML);
            if (resource != null) {
                // CDI enabled archive (Jar)
                return true;
            }

            // ... and in the war usual location
            resource = archive.getResource(DefaultDeploymentBuilder.WEBINF_BEANS_XML);
            if (resource != null) {
                // CDI enabled archive (Web)
                return true;
            }

        } catch (ArchiveException e) {
            // Missing entry, not CDI enabled
        }

        return false;
    }

    /**
     * Connects all the BDA in the first Collection to each BDA in the second Collection.
     * That makes beans from the second Collection (aka parent) accessible from the beans
     * defined in the first Collection (childs).
     * @param childs Collection of BDAs
     * @param parents Collection of BDAs that will be accessibles from childs
     */
    public void connectBeanDeploymentArchives(final Collection<BeanDeploymentArchive> childs,
                                              final Collection<BeanDeploymentArchive> parents) {

        // Connect all archives together so that they are accessibles to each others
        // TODO Check that libs can access WEB-INF/classes
        for (BeanDeploymentArchive parent : parents) {
            for (BeanDeploymentArchive child : childs) {
                // Prevent recursion (archive loop)
                if (!parent.equals(child)) {
                    // Add the parent in the accessible list of archives of the given archive
                    child.getBeanDeploymentArchives().add(parent);
                }
            }
        }

    }

    /**
     * Starts a Weld container from a given Weld {@code Deployment}.
     * @param deployment Weld deployment interface
     * @return a started Weld Bootstrap
     */
    public Bootstrap startWeldContainer(Deployment deployment) {


        // Provides a BeanStore supporting concurrent access
        BeanStore beanStore = new ConcurrentHashMapBeanStore();

        // Start the Weld Container
        WeldBootstrap boot = new WeldBootstrap();
        // TODO Maybe the Servlet Env is not well suited for our case
        // Consider built up our own Env with a minimal required set of services
        boot.startContainer(Environments.SERVLET,
                            deployment,
                            beanStore);

        // Initialization process
        boot.startInitialization();
        boot.deployBeans();
        boot.validateBeans();
        boot.endInitialization();

        return boot;
    }
}
