/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DefaultBeanDeploymentArchive.java 19316 2010-03-02 16:24:45Z sauthieg $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.cdi.weld.internal.deployment;

import java.net.URL;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;

import org.jboss.weld.bootstrap.api.ServiceRegistry;
import org.jboss.weld.bootstrap.spi.BeanDeploymentArchive;
import org.jboss.weld.ejb.spi.EjbDescriptor;
import org.jboss.weld.resources.spi.ResourceLoader;
import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.archive.api.IArchive;

/**
 * A {@code DefaultBeanDeploymentArchive} is a structure that represents a BDA.
 * @author Guillaume Sauthier
 */
public class DefaultBeanDeploymentArchive implements BeanDeploymentArchive {

    /**
     * WEB-INF/classes location.
     */
    private static final String WEBINF_CLASSES = "WEB-INF/classes/";

    /**
     * Class file suffix.
     */
    private static final String DOT_CLASS = ".class";

    /**
     *
     */
    private enum ResolveStatus {
        UNRESOLVED, RESOLVED
    }

    /**
     * BDA unique ID in this Deployment.
     */
    private String id;

    private Collection<BeanDeploymentArchive> accessibleBdas;

    private Collection<Class<?>> beanClasses;

    private Collection<URL> beansXmlUrls;

    private Collection<EjbDescriptor<?>> ejbDescriptors;

    private ServiceRegistry registry;

    private ResolveStatus status = ResolveStatus.UNRESOLVED;

    private IArchive archive;

    /**
     * Constructs a Default BDA.
     * @param id BDA identity (unique in the Deployment)
     * @param archive the archive to be analyzed
     * @param registry the parent {@code Deployment} registry.
     */
    public DefaultBeanDeploymentArchive(final String id,
                                        final IArchive archive,
                                        final ServiceRegistry registry) {
        this.id = id;
        this.archive = archive;
        this.registry = registry;
        this.accessibleBdas = new HashSet<BeanDeploymentArchive>();
        this.beanClasses = new HashSet<Class<?>>();
        this.beansXmlUrls = new HashSet<URL>();
        this.ejbDescriptors = new HashSet<EjbDescriptor<?>>();
    }

    /**
     * Resolve the current BDA (ie load Class objects)
     * @throws IllegalStateException if an error occurs during class-loading.
     */
    private void resolve() {
        // Do resolve stuff using the provided ResourceLoader
        // That avoids relying on the TCCL for ClassLoading
        ResourceLoader loader = registry.get(ResourceLoader.class);

        try {
            Iterator<String> entries = archive.getEntries();
            while (entries.hasNext()) {
                String entry = entries.next();
                // Only handle classes
                if (entry.endsWith(DOT_CLASS)) {
                    String classname = entry.substring(0, entry.lastIndexOf('.'));

                    // Support classes defined in WEB-INF/lib
                    if (entry.startsWith(WEBINF_CLASSES)) {
                        classname = classname.substring(WEBINF_CLASSES.length());
                    }

                    // Fix classname
                    classname = classname.replace('/', '.');

                    // Try class loading
                    beanClasses.add(loader.classForName(classname));
                }
            }

            // change status
            status = ResolveStatus.RESOLVED;
        } catch (ArchiveException e) {
            throw new IllegalStateException("Unable to get resource from archive " + archive, e);
        }
    }

    /**
     * Get the bean deployment archives which are accessible to this bean
     * deployment archive and adjacent to it in the deployment archive graph.
     * <p/>
     * Cycles in the accessible BeanDeploymentArchive graph are allowed. If a
     * cycle is detected by Weld, it will be automatically removed by Web
     * Beans. This means any implementor of this interface don't need to worry
     * about circularities.
     *
     * @return the accessible bean deployment archives
     */
    public Collection<BeanDeploymentArchive> getBeanDeploymentArchives() {
        return accessibleBdas;
    }

    /**
     * Gets all classes in the bean deployment archive
     *
     * @return the classes, empty if no classes are present
     */
    public Collection<Class<?>> getBeanClasses() {

        // Maybe we have to resolve the Class first
        if (ResolveStatus.UNRESOLVED.equals(status)) {
            resolve();
        }

        return beanClasses;
    }

    /**
     * Get any deployment descriptors in the bean deployment archive.
     * <p/>
     * The container will normally return a single deployment descriptor per bean
     * deployment archive (the physical META-INF/beans.xml or WEB-INF/beans.xml),
     * however it is permitted to return other deployment descriptors defined
     * using other methods.
     *
     * @return the URLs pointing to the deployment descriptor,
     *         or an empty set if none are present
     */
    public Collection<URL> getBeansXml() {
        return beansXmlUrls;
    }

    /**
     * Get all the EJBs in the deployment archive
     *
     * @return the EJBs, or empty if no EJBs are present or if
     *         this is not an EJB archive
     */
    public Collection<EjbDescriptor<?>> getEjbs() {
        return ejbDescriptors;
    }

    /**
     * Get the Bean Deployment Archive scoped services
     *
     * @return
     */
    public ServiceRegistry getServices() {
        return registry;
    }

    /**
     * Get a string which uniquely identifies the {@link org.jboss.weld.bootstrap.spi.BeanDeploymentArchive} within
     * the {@link org.jboss.weld.bootstrap.spi.Deployment}. The identifier must be consistent between multiple
     * occurrences of this deployment.
     *
     * @return
     */
    public String getId() {
        return id;
    }
}
