/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2007,2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: CmiServiceImpl.java 12729 2008-01-28 13:45:24Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.cmi.internal;


import java.net.URL;

import javax.ejb.EJBObject;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Logger;
import org.ow2.carol.cmi.config.CMIConfig;
import org.ow2.carol.cmi.controller.server.AbsServerClusterViewManager;
import org.ow2.carol.cmi.info.CMIInfoExtractor;
import org.ow2.carol.cmi.info.CMIInfoExtractorException;
import org.ow2.carol.cmi.info.CMIInfoRepository;
import org.ow2.carol.cmi.info.ClusteredObjectInfo;
import org.ow2.carol.util.configuration.ConfigurationRepository;
import org.ow2.jonas.cmi.CmiService;
import org.ow2.jonas.lib.bootstrap.JProp;
import org.ow2.jonas.lib.execution.ExecutionResult;
import org.ow2.jonas.lib.execution.IExecution;
import org.ow2.jonas.lib.execution.RunnableHelper;
import org.ow2.jonas.lib.service.AbsServiceImpl;
import org.ow2.jonas.lib.util.Log;
import org.ow2.jonas.service.ServiceException;

/**
 * CMI Service interface.
 * It provides a way to use CMI with JOnAS.
 * @author Loris Bouzonnet
 */
public class CmiServiceImpl extends AbsServiceImpl implements CmiService {

    /**
     * Logger for traces
     */
    protected static Logger cmilogger = Log.getLogger("org.ow2.carol.cmi");

    /**
     * Start the service.
     */
    @Override
    protected void doStart() throws ServiceException {
        // Enable bindings into the cluster
        try {
            ConfigurationRepository.getServerConfiguration().setStartCMI(true);
        } catch (Exception e) {
            cmilogger.log(BasicLevel.ERROR, "Cannot configure Carol to use CMI", e);
            throw new ServiceException("Cannot configure Carol to use CMI", e);
        }

        CMIConfig.setConfDir(JProp.getConfDir());

        // Start the CMI server in an execution block
        IExecution<Void> startExec = new IExecution<Void>() {
            public Void execute() throws Exception {
                AbsServerClusterViewManager.start();
                return null;
            }
        };

        // Execute
        ExecutionResult<Void> startExecResult = RunnableHelper.execute(getClass().getClassLoader(), startExec);

        if(startExecResult.hasException()) {
            cmilogger.log(BasicLevel.ERROR, "Cannot start the server-side manager", startExecResult.getException());
            throw new ServiceException("Cannot start the server-side manager", startExecResult.getException());
        }

    }

    /**
     * Stop the service.
     */
    @Override
    protected void doStop() throws ServiceException {
        try {
            ConfigurationRepository.getServerConfiguration().setStartCMI(false);
        } catch (Exception e) {
            cmilogger.log(BasicLevel.ERROR, "Cannot disable cmi in Carol", e);
        }
        try {
            AbsServerClusterViewManager.stop();
        } catch (Exception e) {
            cmilogger.log(BasicLevel.ERROR, "Cannot stop the server-side manager", e);
            throw new ServiceException("Cannot stop the server-side manager", e);
        }
    }

    /**
     * Add a bean to the cluster.
     * @param jndiName name of the bean
     * @param clusterPolicyName filename
     * @param homeClass class of the home interface
     * @param remoteClass class of the remote interface
     * @param classLoader the classloader used by the container of this bean
     * @param stateful true if the bean has a state
     * @param clusterReplicated true if the bean is replicated (ha service is required)
     * @throws Exception if the provided policy of load-balancing is not valid
     */
    public void addClusteredObject(
            final String jndiName,
            final String clusterPolicyName,
            final Class<?> homeClass,
            final Class<? extends EJBObject> remoteClass,
            final ClassLoader classLoader,
            final boolean stateful,
            final boolean clusterReplicated)
            throws Exception {
        // An object is clustered if and only if a policy of load-balancing is associated to this
        if(clusterPolicyName == null) {
            cmilogger.log(BasicLevel.DEBUG, "The object with name " + jndiName + " is not clustered.");
            return;
        }
        URL url = classLoader.getResource(clusterPolicyName);
        if(url != null) {
            try {
                ClusteredObjectInfo clusteredObjectInfo =
                    CMIInfoExtractor.extractClusteringInfoFromDD(
                       homeClass, remoteClass,
                        url, stateful, clusterReplicated, null);
                CMIInfoRepository.addClusteredObjectInfo(jndiName, clusteredObjectInfo);
                cmilogger.log(BasicLevel.INFO, "The object with name " + jndiName + " is clustered.");
            } catch (CMIInfoExtractorException e) {
                cmilogger.log(BasicLevel.ERROR,
                        "Error when extracting infos about clustering for the object with name " + jndiName, e);
                throw new Exception(
                        "Error when extracting infos about clustering for the object with name " + jndiName, e);
            }
        } else {
            cmilogger.log(BasicLevel.ERROR, "Cannot find the policy of load-balancing with name " + clusterPolicyName);
            throw new Exception("Cannot find the policy of load-balancing with name " + clusterPolicyName);
        }

    }

}
