/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 1999-2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: CheckEnv.java 13510 2008-04-04 08:43:45Z fornacif $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.commands.check;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

/**
 * This class allows to check if the environment is correct for JOnAS.
 * @author Francois Fornaciari
 */
public final class CheckEnv {

    /**
     * Constant result of JOnAS's conf check [OK].
     */
    public static final int ENV_OK = 0;

    /**
     * Constant result of the JOnAS configuration check [Error].
     */
    public static final int ENV_ERROR = 1;

    /**
     * Status of the the JOnAS configuration check.
     */
    private static int envStatus = ENV_OK;

    /**
     * Constant JONAS_ROOT.
     */
    private static final String JONAS_ROOT = "jonas.root";

    /**
     * Constant JONAS_BASE.
     */
    private static final String JONAS_BASE = "jonas.base";

    /**
     * Constant JORAM rar name.
     */
    private static final String JORAM_RAR = "joram_ra_for_jonas.rar";

    /**
     * JORAM admin configuration file.
     */
    private static final String JORAM_ADMIN_FILE = "joramAdmin.xml";

    /**
     * Files to check.
     */
    private static List<String> filesToCheck = new ArrayList<String>();

    /**
     * Initialize files to check.
     */
    static {
        filesToCheck.add("jonas.properties");
        filesToCheck.add("trace.properties");
        filesToCheck.add("carol.properties");
        filesToCheck.add("jonas-realm.xml");
    }

    /**
     * Private constructor.
     */
    private CheckEnv() {
    }

    /**
     * Main function, creates a new {@link CheckEnv} instance.
     * @param args See {@link CheckEnv#ClientAdmin(String[])}.
     */
    public static void main(final String[] args) {
        // Get command arguments
        for (String arg : args) {
            if (arg.equals("-help") || arg.equals("-?")) {
                help();
                System.exit(0);
            }
        }

        // Perform the check
        envStatus = performCheck();
        if (envStatus == ENV_OK) {
            System.out.println("\nThe JOnAS environment seems correct.");
            System.exit(0);
        } else if (envStatus == ENV_ERROR) {
            System.out.println("\nERROR : The JOnAS environment is NOT correct.");
            System.exit(2);
        }
    }

    /**
     * Check the JOnAS's configuration.
     * @return check status
     */
    public static int performCheck() {
        envStatus = ENV_OK;

        // Check the $JONAS_ROOT system property
        try {
            String jonasRoot = System.getProperty(JONAS_ROOT);
            if (jonasRoot == null) {
                System.out.println("ERROR : JONAS_ROOT system property not defined");
                return ENV_ERROR;
            } else {
                if (!new File(jonasRoot).isDirectory()) {
                    System.out.println("ERROR : " + jonasRoot + " directory does not exist");
                    return ENV_ERROR;
                } else {
                    System.out.println("- JONAS_ROOT value : " + jonasRoot);
                }
            }
        } catch (SecurityException e) {
            System.out.println("ERROR : Cannot get the '" + JONAS_ROOT + "' system property ");
            return ENV_ERROR;
        }

        // Check the $JONAS_BASE system property
        try {
            String jonasBase = System.getProperty(JONAS_BASE);
            if (jonasBase == null) {
                System.out.println("ERROR : JONAS_BASE system property not defined");
                return ENV_ERROR;
            } else {
                if (!new File(jonasBase).isDirectory()) {
                    System.out.println("ERROR : " + jonasBase + " directory does not exist");
                    return ENV_ERROR;
                } else {
                    System.out.println("- JONAS_BASE value : " + jonasBase);
                }
            }
        } catch (SecurityException e) {
            System.out.println("ERROR : Cannot get the '" + JONAS_BASE + "' system property (" + e + ")");
            return ENV_ERROR;
        }

        // Check declared property files
        for (String fileName : filesToCheck) {
            if (!fileExists(fileName)) {
                System.out.println("ERROR : '" + fileName + "' not accessible in JONAS_BASE/conf directory");
                envStatus = ENV_ERROR;
            }
        }

        // Check JORAM RA configuration
        checkJORAMRar();

        return envStatus;
    }

    /**
     * Check is a file is accessible.
     * @param fileName The file name
     * @return True is the given file is accessible
     */
    private static boolean fileExists(final String fileName) {
        String jonasBase = System.getProperty(JONAS_BASE);
        File file = new File(jonasBase, "conf" + File.separator + fileName);
        return file.exists();
    }

    /**
     * Print help message.
     */
    private static void help() {
        System.out.println("This comand allows to check if the JOnAS environment is correct or not.");
    }

    /**
     * Check JORAM RA configuration.
     */
    private static void checkJORAMRar() {
        System.out.println("\nChecking JORAM configuration...");

        String jonasBase = System.getProperty(JONAS_BASE);

        File deployDirectory = new File(jonasBase, "deploy");
        if (!new File(deployDirectory, JORAM_RAR).exists()) {
            System.out.println("ERROR : the file '" + JORAM_RAR + "' was not found in JONAS_BASE/deploy directory");
            envStatus = ENV_ERROR;
        }

        // Check presence of joramAdmin.xml
        File joramAdminFile = new File(jonasBase + File.separator + "conf" + File.separator + JORAM_ADMIN_FILE);
        if (!joramAdminFile.exists()) {
            System.out.println("ERROR : the file " + JORAM_ADMIN_FILE + " was not found in JONAS_BASE/conf directory");
            envStatus = ENV_ERROR;
        }
    }
}
