/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 1999-2007 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: EarCleanTask.java 12246 2007-12-09 21:42:38Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.ear.internal;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Vector;

import org.objectweb.util.monolog.api.BasicLevel;
import org.ow2.jonas.ear.EarService;
import org.ow2.jonas.lib.work.AbsCleanTask;
import org.ow2.jonas.lib.work.CleanerException;
import org.ow2.jonas.lib.work.DeployerLog;
import org.ow2.jonas.lib.work.DeployerLogException;
import org.ow2.jonas.lib.work.FileManager;
import org.ow2.jonas.lib.work.FileManagerException;
import org.ow2.jonas.lib.work.LogEntry;

/**
 * JOnAS Ear unused directory clean task class. This class provides a way for
 * removing directories wich are unconsistent directories for ear files.
 * @author Florent BENOIT
 * @author Benoit PELLETIER
 */
public class EarCleanTask extends AbsCleanTask {

    /**
     * reference to the deployerLog.
     */
    private static DeployerLog earDeployerLog = null;

    /**
     * Url of the jonas_ROOT/apps directory.
     */
    private static URL earAppsUrl = null;

    /**
     * {@link EarService} instance.
     */
    private EarService earService;

    /**
     * Default constructor : Construct a new cleaner.
     * @param earService {@link EarService} instance
     * @param earApps the jonasroot apps directory
     * @param earDeployerLog the deployer logger
     */
    public EarCleanTask(final EarService earService,
                        final URL earApps,
                        final DeployerLog earDeployerLog) {
        super();
        this.earService = earService;
        EarCleanTask.earAppsUrl = earApps;
        EarCleanTask.earDeployerLog = earDeployerLog;
    }

    /**
     * Return true if the work copy exists and is up to date.
     * @param logEntry entry in a deploy log
     * @return true if the work copy exists and is up to date
     * @throws CleanerException if it fails
     */
    protected boolean isValidLogEntry(final LogEntry logEntry) throws CleanerException {
        String fTimeStamp = null;
        File earLogEntryFile = logEntry.getOriginal();
        String earLogEntryUnpackedDir = logEntry.getCopy().getName();
        //File dirEar = new File(earAppsUrl.getFile() + File.separator + earLogEntryUnpackedDir);
        if (getLogger().isLoggable(BasicLevel.DEBUG)) {
            getLogger().log(BasicLevel.DEBUG,
                    "LogEntry <" + earLogEntryFile.getName() + "> exist :" + earLogEntryFile.exists());
        }

        // if the file doesn't exist, return
        if (!earLogEntryFile.exists()) {
            return false;
        }

        //get the timestamp
        try {
            fTimeStamp = FileManager.fileToTimeStampDir(earLogEntryFile.toURL());
        } catch (FileManagerException efme) {
            throw new CleanerException("Can't get the timestamp of the file " + earLogEntryFile + " : "
                    + efme.getMessage());
        } catch (MalformedURLException mue) {
            throw new CleanerException("Can't get the timestamp of the file " + earLogEntryFile + " : "
                    + mue.getMessage());
        }

        if (getLogger().isLoggable(BasicLevel.DEBUG)) {
            getLogger().log(BasicLevel.DEBUG, "LogEntry fTimeStamp :" + fTimeStamp);
            getLogger().log(BasicLevel.DEBUG, "LogEntry isValid :" + fTimeStamp.equalsIgnoreCase(earLogEntryUnpackedDir));
        }

        //compare
        return (fTimeStamp.equalsIgnoreCase(earLogEntryUnpackedDir));

    }

    /**
     * Remove the work copy specified in the log entry and the log entry.
     * @param logEntry entry in a deploy log
     * @throws CleanerException if it fails
     */
    protected void removeLogEntry(final LogEntry logEntry) throws CleanerException {
        String earLogEntryUnpackedDir = logEntry.getCopy().getName();

        File dirEar = new File(earAppsUrl.getFile() + File.separator + earLogEntryUnpackedDir);

        removeRecursiveDirectory(dirEar);

        try {
            earDeployerLog.removeEntry(logEntry);
        } catch (DeployerLogException edle) {
            throw new CleanerException("Can't remove an entry" + edle.getMessage());
        }

    }

    /**
     * Gets the log entries.
     * @return the log entries
     */
    protected Vector getLogEntries() {
        return earDeployerLog.getEntries();
    }

    /**
     * Check if the package pointed by the log entry is currently deploy.
     * @param logEntry entry in a deploy log
     * @return true if the package pointed by the log entry is currently deployed
     * @throws CleanerException if it fails
     */
    protected boolean isDeployLogEntry(final LogEntry logEntry) throws CleanerException {
        // check if the ear file is deployed
        return earService.isEarDeployedByUnpackName(logEntry.getCopy().getName());
    }
}
