/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 1999-2007 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JOnASEARService.java 12348 2007-12-13 13:57:35Z fornacif $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.ear.internal;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.security.Policy;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.naming.Context;
import javax.naming.NamingException;
import javax.security.jacc.PolicyConfiguration;
import javax.security.jacc.PolicyConfigurationFactory;
import javax.security.jacc.PolicyContextException;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Logger;
import org.ow2.jonas.deployment.client.ClientContainerDeploymentDesc;
import org.ow2.jonas.deployment.client.ClientContainerDeploymentDescException;
import org.ow2.jonas.deployment.client.wrapper.ClientManagerWrapper;
import org.ow2.jonas.deployment.ear.EarDeploymentDesc;
import org.ow2.jonas.deployment.ear.EarDeploymentDescException;
import org.ow2.jonas.deployment.ear.wrapper.EarManagerWrapper;
import org.ow2.jonas.deployment.ear.xml.Web;
import org.ow2.jonas.deployment.ejb.wrapper.EjbManagerWrapper;
import org.ow2.jonas.deployment.web.wrapper.WebManagerWrapper;
import org.ow2.jonas.ear.EarService;
import org.ow2.jonas.ear.EarServiceException;
import org.ow2.jonas.ear.internal.mbean.AppClientModule;
import org.ow2.jonas.ear.internal.mbean.Ear;
import org.ow2.jonas.ejb.EJBService;
import org.ow2.jonas.ejb.easybeans.IEasyBeansService;
import org.ow2.jonas.jmx.JmxService;
import org.ow2.jonas.lib.bootstrap.JClassLoader;
import org.ow2.jonas.lib.bootstrap.JProp;
import org.ow2.jonas.lib.bootstrap.LoaderManager;
import org.ow2.jonas.lib.cpmanager.EarClassPathManager;
import org.ow2.jonas.lib.cpmanager.EarClassPathManagerException;
import org.ow2.jonas.lib.cpmanager.JarList;
import org.ow2.jonas.lib.cpmanager.JarListException;
import org.ow2.jonas.lib.loader.ClientClassLoader;
import org.ow2.jonas.lib.loader.EjbJarClassLoader;
import org.ow2.jonas.lib.management.javaee.J2eeObjectName;
import org.ow2.jonas.lib.naming.ComponentContext;
import org.ow2.jonas.lib.security.mapping.JPolicyUserRoleMapping;
import org.ow2.jonas.lib.service.AbsServiceImpl;
import org.ow2.jonas.lib.util.JModule;
import org.ow2.jonas.lib.util.JonasObjectName;
import org.ow2.jonas.lib.util.Log;
import org.ow2.jonas.lib.util.ModuleNamingUtils;
import org.ow2.jonas.lib.work.CleanerException;
import org.ow2.jonas.lib.work.DeployerLog;
import org.ow2.jonas.lib.work.DeployerLogException;
import org.ow2.jonas.lib.work.EarFileManager;
import org.ow2.jonas.lib.work.FileManagerException;
import org.ow2.jonas.lib.work.WorkCleaner;
import org.ow2.jonas.lib.wsgen.CheckerException;
import org.ow2.jonas.lib.wsgen.WsGenChecker;
import org.ow2.jonas.resource.ResourceService;
import org.ow2.jonas.resource.ResourceServiceException;
import org.ow2.jonas.service.ServiceException;
import org.ow2.jonas.web.JWebContainerService;
import org.ow2.jonas.web.JWebContainerServiceException;
import org.ow2.jonas.ws.WSServiceException;
import org.ow2.jonas.ws.WebServicesService;
import org.ow2.util.ee.deploy.api.archive.IArchive;
import org.ow2.util.ee.deploy.api.deployable.EARDeployable;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployer.IDeployerManager;
import org.ow2.util.ee.deploy.impl.archive.ArchiveManager;
import org.ow2.util.ee.deploy.impl.helper.DeployableHelper;
import org.ow2.util.ee.deploy.impl.helper.DeployableHelperException;
import org.ow2.util.url.URLUtils;

/**
 * JOnAS EAR Service Implementation class. This class provides an implementation
 * of the ear service.
 * @author Florent Benoit
 * @author Ludovic Bert Contributor(s): Adriana Danes: highlight configuration
 *         properties Eric Hardesty: added ability to include rar files in an
 *         ear Michel-Ange Anton : new JSR77 MBean
 */
public class JOnASEARService extends AbsServiceImpl implements EarService, JOnASEARServiceMBean {

    /**
     * The name of the JONAS_BASE directory.
     */
    protected static final String JONAS_BASE = JProp.getJonasBase();

    /**
     * The name of the apps directory.
     */
    protected static final String APPS_DIR = JONAS_BASE + File.separator + "apps";

    /**
     * The name of the working directory.
     */
    protected static final String WORK_DIR = JProp.getWorkDir();

    /**
     * The name of the working apps directory.
     */
    protected static final String WORK_APPS_DIR = WORK_DIR + File.separator + "apps";

    /**
     * Logger for this service.
     */
    private static Logger logger = Log.getLogger(Log.JONAS_EAR_PREFIX);

    /**
     * Reference to the cleaner.
     */
    private static WorkCleaner workCleaner = null;

    /** JMX Service. */
    private JmxService jmxService = null;

    /**
     * List of the ear names to load when starting the EarService.
     */
    private List earNames = new Vector();

    /**
     * Reference to the Ejb service.
     */
    private EJBService ejbService = null;

    /**
     * Reference to the WebContainer service.
     */
    private JWebContainerService webContainerService = null;

    /**
     * Reference to the WebServices service.
     */
    private WebServicesService wsService = null;

    /**
     * Reference to the Resource service.
     */
    private ResourceService resourceService = null;

    /**
     * EasyBeans service.
     */
    private IEasyBeansService easyBeansService = null;

    /**
     * List of loaded ear.
     */
    private Hashtable ears = null;

    /**
     * List of autoloaded directories.
     */
    private ArrayList autoloadDirectories = new ArrayList();

    /**
     * Reference on the DeployerLog which is the class that manage the
     * accesses to the log file (to remove the ear).
     */
    private DeployerLog earDeployerLog = null;

    /**
     * Application Classloader.
     */
    private ClassLoader appsClassLoader;

    /**
     * Link to the Ear Deployer.
     */
    private EarDeployer earDeployer = null;

    /**
     * DeployerManager Service.
     */
    private IDeployerManager deployerManager;

    /**
     * Initialize some Maps at instantiation time.
     */
    public JOnASEARService() {
        //create hashtable of the ears ( Url of an ear ---> Ear structure)
        ears = new Hashtable();

        // Create the EAR deployer
        earDeployer = new EarDeployer();
    }

    /**
     * @param validate Use a validating parser ?
     */
    public void setParsingwithvalidation(final boolean validate) {
        EarManagerWrapper.setParsingWithValidation(validate);
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            if (!validate) {
                logger.log(BasicLevel.DEBUG, "EAR XML parsing without validation");
            } else {
                logger.log(BasicLevel.DEBUG, "EAR XML parsing with validation");
            }
        }
    }

    /**
     * @param descriptors EARs to be deployed at startup.
     */
    public void setDescriptors(final String descriptors) {
        this.earNames = convertToList(descriptors);
    }

    /**
     * @param directoriesValue Autoload directories
     */
    public void setAutoloaddir(final String directoriesValue) {

        List<String> directories = convertToList(directoriesValue);
        for(String directory : directories) {
            addEars(directory);
            File oFile = new File(APPS_DIR, directory);
            if (!oFile.exists()) {
                oFile = new File(directory);
            }
            if (oFile.exists()) {
                try {
                    autoloadDirectories.add(oFile.getCanonicalPath());
                } catch (IOException e) {
                    String err = "Error when trying to verify Application EAR autoload directory : " + directory;
                    logger.log(BasicLevel.ERROR, err, e);
                }
            }
        }
    }

    /**
     * {@inheritDoc}
     * @see org.ow2.jonas.lib.service.AbsServiceImpl#checkRequirements()
     */
    @Override
    public void checkRequirements() throws ServiceException {
        if (jmxService == null) {
            throwRequirementException("Missing reference on " + JmxService.class);
        }
    }

    /**
     * Stop the EAR service.
     * @throws ServiceException if the stop failed.
     */
    protected void doStop() throws ServiceException {
        //undeploy the loaded ears files
        URL earFileName = null;
        //For each ear file, we delegate the operation to the
        //unDeployEar method
        for (Enumeration earEntries = ears.keys(); earEntries.hasMoreElements();) {
            earFileName = (URL) earEntries.nextElement();
            //Try to undeploy the ear
            try {
                Context ctx = new ComponentContext(earFileName.getFile());
                ctx.rebind("filename", earFileName);
                unDeployEar(ctx);
            } catch (Exception e) {
                //We don't stop the process of undeploying the ears.
                //We only display an error in the logger.
                String err = "Error when undeploying the ear :" + earFileName;
                logger.log(BasicLevel.ERROR, err, e);
            }
        }

        if (deployerManager != null) {
            // Unregister deployer
            deployerManager.unregister(earDeployer);
        }

        // Admin code
        if (jmxService != null) {
            unregisterEarServiceMBean(getDomainName());
        }
        // TODO Remove ModelMBean from the commons modeler registry

        //The service is stopped
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "EarService stopped");
        }
    }

    /**
     * Deploy an EAR file with sending JAR file to the EJB container and WAR
     * file to the WEB container and RAR file to the resource service.
     * @param ctx the context which contains the configuration in order to
     *        deploy an EAR.
     * @return The OBJECT_NAME of the J2EE Application MBean associated to the
     *         deployed EAR the Application Container
     * @throws EarServiceException if the deployment of the EAR failed.
     */
    public String deployEar(final Context ctx) throws EarServiceException {

        //We have one of the ejb or web services launched
        if ((webContainerService == null) && (ejbService == null)) {
            throw new EarServiceException(
                    "The ear service requires that at least the service ejb or web is launched for deploying an ear file.");
        }

        //We get the fileName
        // Get the file name and check if file exists
        String fileName;
        try {
            fileName = (String) ctx.lookup("filename");
        } catch (NamingException e) {
            throw new EarServiceException("Error during performing lookup a fileName", e);
        }

        File f = null;
        try {
            //We search the file
            f = new File(fileName).getCanonicalFile();

            if (!f.exists()) {
                boolean found = false;
                String earFileName = null;
                if (fileName.toLowerCase().endsWith(".ear")) {
                    // In case of the name is a ear file name, check also in
                    // the JONAS_BASE/apps directory
                    earFileName = APPS_DIR + File.separator + fileName;
                    f = new File(earFileName).getCanonicalFile();
                    found = f.exists();
                }
                if (found) {
                    fileName = earFileName;
                } else {
                    String err = "deployEar: The file " + fileName
                            + " was not found neither in the current directory nor in the " + APPS_DIR + " directory";
                    logger.log(BasicLevel.ERROR, err);
                    throw new EarServiceException(err);
                }
            }
        } catch (IOException e) {
            String err = "Error when trying to get the canonical file from " + fileName;
            logger.log(BasicLevel.ERROR, err + " " + e.getMessage());
            throw new EarServiceException(err, e);
        }

        // Apply WsGen if needed
        // Before creating URLs, ...
        // -------------------------------
        try {
            ObjectName j2eeServer = J2eeObjectName.J2EEServer(this.getDomainName(), this.getJonasServerName());
            WsGenChecker wsgc = new WsGenChecker(fileName, j2eeServer, jmxService.getJmxServer());
            wsgc.checkWsGen(getDomainName());
        } catch (CheckerException e) {
            throw new EarServiceException("Cannot apply WsGen on the ear : " + fileName, e);
        } catch (Exception ue) {
            throw new EarServiceException("Cannot apply WsGen on the ear (unexpected exception) : " + fileName, ue);
        }

        //--------------------------------

        //Make the url for the specified fileName and the earRootUrl
        URL[] earUrl = new URL[1];
        URL earRootUrl = null;

        try {
            earUrl[0] = f.toURL();
            earRootUrl = (new File(WORK_APPS_DIR + File.separator + getJonasServerName())).toURL();
        } catch (MalformedURLException e) {
            String err = "Invalid ear file name '" + fileName;
            logger.log(BasicLevel.ERROR, err + "': " + e.getMessage());
            throw new EarServiceException(err, e);
        }

        //Check if the ear is already deployed or not
        if (ears.get(earUrl[0]) != null) {
            String err = "The ear file : " + f.getName() + " is already deployed ('" + earUrl[0].getFile()
                    + "'). You must undeploy the application " + "before a new deployment.";
            logger.log(BasicLevel.ERROR, err);
            throw new EarServiceException(err);
        }

        //Create classLoader
        //parent classloader is the current classloader
        URLClassLoader loaderCls = new URLClassLoader(earUrl, appsClassLoader);

        EarDeploymentDesc earDD = null;
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "Getting the deployment descriptor of the file" + f.getName());
        }

        // TODO : Remove the use of this file ?
        File earFile = URLUtils.urlToFile(earUrl[0]);
        IArchive archive = ArchiveManager.getInstance().getArchive(earFile);

        IDeployable<?> deployable;
        try {
            deployable = DeployableHelper.getDeployable(archive);
        } catch (DeployableHelperException e) {
            throw new EarServiceException("Cannot get a deployable for the archive '"
                    + archive + "'", e);
        }
        EARDeployable earDeployable = null;
        if (deployable instanceof EARDeployable) {
            earDeployable = (EARDeployable) deployable;
        } else {
            throw new EarServiceException("The Deployable '" + deployable + "' is not an EAR Deployable");
        }

        try {
            earDD = EarManagerWrapper.getDeploymentDesc(earDeployable, loaderCls);
        } catch (EarDeploymentDescException e) {
            String err = "Error in the Deployment descriptor '" + fileName + "': " + e;
            logger.log(BasicLevel.ERROR, err);
            throw new EarServiceException(err, e);
        }

        //We get the tags from the Deployment descriptor
        Web[] webs = earDD.getWebTags();
        String[] webTags = new String[webs.length];
        String[] ejbTags = earDD.getEjbTags();
        String[] connectorTags = earDD.getConnectorTags();
        String[] clientTags = earDD.getClientTags();
        String[] altDDClients = earDD.getAltDDClients();
        String[] altDDEjbs = earDD.getAltDDEjbs();
        String[] altDDWebs = earDD.getAltDDWebs();
        String[] altDDConnectors = earDD.getAltDDConnectors();
        String[] securityRoles = earDD.getSecurityRolesNames();

        //Check if all modules are inside the EAR file
        //no relatives mode like ../../file1.jar

        File fEar = new File(earUrl[0].getFile());
        File tmpFile = null;

        try {
            File fCanonicEar = fEar.getCanonicalFile();
            for (int i = 0; i < ejbTags.length; i++) {
                tmpFile = new File(fEar, ejbTags[i]);
                tmpFile = tmpFile.getCanonicalFile();
                if (!tmpFile.getPath().startsWith(fCanonicEar.getPath())) {
                    String err = "Error : The ejb-jar file " + ejbTags[i] + " is not inside the ear file " + fEar;
                    logger.log(BasicLevel.ERROR, err);
                    throw new EarServiceException(err);
                }
            }

            for (int i = 0; i < webs.length; i++) {
                tmpFile = new File(fEar, webs[i].getWebUri());
                tmpFile = tmpFile.getCanonicalFile();
                if (!tmpFile.getPath().startsWith(fCanonicEar.getPath())) {
                    String err = "Error : The war file " + webs[i].getWebUri() + " is not inside the ear file "
                            + fEar;
                    logger.log(BasicLevel.ERROR, err);
                    throw new EarServiceException(err);
                }
                webTags[i] = webs[i].getWebUri();
            }

            for (int i = 0; i < connectorTags.length; i++) {
                tmpFile = new File(fEar, connectorTags[i]);
                tmpFile = tmpFile.getCanonicalFile();
                if (!tmpFile.getPath().startsWith(fCanonicEar.getPath())) {
                    String err = "Error : The rar file " + connectorTags[i] + " is not inside the ear file " + fEar;
                    logger.log(BasicLevel.ERROR, err);
                    throw new EarServiceException(err);
                }
            }


            for (int i = 0; i < clientTags.length; i++) {
                tmpFile = new File(fEar, clientTags[i]);
                tmpFile = tmpFile.getCanonicalFile();

                if (!tmpFile.getPath().startsWith(fCanonicEar.getPath())) {
                    String err = "Error : The client jar file " + clientTags[i] + " is not inside the ear file " + fEar;
                    throw new EarServiceException(err);
                }
            }
        } catch (IOException ioe) {
            String err = "Error while trying to get the canonical file of " + tmpFile;
            logger.log(BasicLevel.ERROR, err + " : " + ioe.getMessage());
            throw new EarServiceException(err, ioe);
        }

        //Changing array into JarList
        JarList ejbsList = new JarList(ejbTags);
        JarList warsList = new JarList(webTags);
        JarList connectorsList = new JarList(connectorTags);
        JarList clientsList = new JarList(clientTags);

        //Unpack the ear file and get the unpacked dir
        URL dirUnpackURL = null;
        try {
            dirUnpackURL = EarFileManager.unpackEar(earUrl[0], earRootUrl);
        } catch (FileManagerException e) {
            String err = "Error while unpacking the file '" + earUrl[0] + "'";
            logger.log(BasicLevel.ERROR, err + " : " + e.getMessage());
            throw new EarServiceException(err, e);
        }

        //the file is unpacked. Log it only if the ear is a file and not a
        // directory
        if (new File(earUrl[0].getFile()).isFile()) {
            try {
                earDeployerLog.addEntry(new File(earUrl[0].getFile()), new File(dirUnpackURL.getFile()));
            } catch (DeployerLogException e) {
                String err = "Error while adding the " + earUrl[0] + " entry in the log file";
                logger.log(BasicLevel.ERROR, err + " : " + e.getMessage());
                throw new EarServiceException(err, e);
            }
        }

        //We have successfully unpacked the ear file, now we analyze manifest
        // Class-path:
        EarClassPathManager earCPManager = null;
        try {
            earCPManager = new EarClassPathManager(ejbsList, warsList, dirUnpackURL);
        } catch (EarClassPathManagerException e) {
            String err = "Error while creating the Ear class path manager of the ear : '" + earUrl[0] + "'";
            logger.log(BasicLevel.ERROR, err + " : " + e.getMessage());
            throw new EarServiceException(err, e);
        }

        URL[] classpathURLs = null;
        //Get the urls of the ear class path manager
        try {
            classpathURLs = earCPManager.getResolvedClassPath();
        } catch (EarClassPathManagerException e) {
            String err = "Error while trying to resolve the classpath of the ejbjars and wars of the ear : '"
                    + earUrl[0] + "'";
            logger.log(BasicLevel.ERROR, err + " : " + e.getMessage());
            throw new EarServiceException(err, e);
        }

        /*
         * We have the urls to load at our level We can now create our
         * classLoader
         */
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "Creating the EAR classLoader");
        }
        JClassLoader earClassLoader = new JClassLoader(earUrl[0].toExternalForm() , new URL[0], appsClassLoader);

        //Extract the urls of the jarList
        URL[] jarUrls = null;
        URL[] warUrls = null;
        URL[] connectorUrls = null;
        URL[] clientUrls = null;
        try {
            jarUrls = ejbsList.getURLs(dirUnpackURL.toExternalForm());
            warUrls = warsList.getURLs(dirUnpackURL.toExternalForm());
            connectorUrls = connectorsList.getURLs(dirUnpackURL.toExternalForm());
            clientUrls = clientsList.getURLs(dirUnpackURL.toExternalForm());
        } catch (JarListException e) {
            String err = "Error while geting the Urls from jarlist of the ear : '" + earUrl[0] + "'";
            logger.log(BasicLevel.ERROR, err + " : " + e.getMessage());
            throw new EarServiceException(err, e);
        }

        //Extract context roots for wars
        //They are in webTags. Dump into warsContextRoots
        String[] warsContextRoots = new String[webs.length];
        String ctxRoot = null;
        for (int i = 0; i < webs.length; i++) {
            ctxRoot = webs[i].getContextRoot();
            if (ctxRoot != null) {
                warsContextRoots[i] = ctxRoot;
            }
        }

        //Fill Alt-DD for Ejbs and Wars and Rars
        String altdd = null;
        File fAltDD = null;

        //Transorm the array altDDWebs into an array with the absolute URL to
        // the file
        URL[] warsAltDDs = new URL[altDDWebs.length];
        for (int i = 0; i < altDDWebs.length; i++) {
            if (altDDWebs[i] != null) {
                altdd = altDDWebs[i];
                if (altdd != null) {
                    try {
                        fAltDD = new File(new URL(dirUnpackURL.toExternalForm() + File.separator + altdd).getFile());
                        warsAltDDs[i] = fAltDD.getCanonicalFile().toURL();
                    } catch (MalformedURLException e) {
                        String err = "Can't build URL for alt-dd '" + altdd;
                        logger.log(BasicLevel.ERROR, err + "': " + e.getMessage());
                        throw new EarServiceException(err, e);
                    } catch (IOException ioe) {
                        String err = "Can't get canonicalFile() for the file '" + fAltDD;
                        logger.log(BasicLevel.ERROR, err + "': " + ioe.getMessage());
                        throw new EarServiceException(err, ioe);
                    }
                }
            }
        }

        URL[] ejbsAltDDs = new URL[altDDEjbs.length];
        for (int i = 0; i < altDDEjbs.length; i++) {
            if (altDDEjbs[i] != null) {
                altdd = altDDEjbs[i];
                if (altdd != null) {
                    try {
                        fAltDD = new File(new URL(dirUnpackURL.toExternalForm() + File.separator + altdd).getFile());
                        ejbsAltDDs[i] = fAltDD.getCanonicalFile().toURL();
                    } catch (MalformedURLException e) {
                        String err = "Can't build URL for alt-dd '" + altdd;
                        logger.log(BasicLevel.ERROR, err + "': " + e.getMessage());
                        throw new EarServiceException(err, e);
                    } catch (IOException ioe) {
                        String err = "Can't get canonicalFile() for the file '" + fAltDD;
                        logger.log(BasicLevel.ERROR, err + "': " + ioe.getMessage());
                        throw new EarServiceException(err, ioe);
                    }
                }
            }
        }

        URL[] connectorsAltDDs = new URL[altDDConnectors.length];
        for (int i = 0; i < altDDConnectors.length; i++) {
            if (altDDConnectors[i] != null) {
                altdd = altDDConnectors[i];
                if (altdd != null) {
                    try {
                        fAltDD = new File(new URL(dirUnpackURL.toExternalForm() + File.separator + altdd).getFile());
                        connectorsAltDDs[i] = fAltDD.getCanonicalFile().toURL();
                    } catch (MalformedURLException e) {
                        String err = "Can't build URL for alt-dd '" + altdd;
                        logger.log(BasicLevel.ERROR, err + "': " + e.getMessage());
                        throw new EarServiceException(err, e);
                    } catch (IOException ioe) {
                        String err = "Can't get canonicalFile() for the file '" + fAltDD;
                        logger.log(BasicLevel.ERROR, err + "': " + ioe.getMessage());
                        throw new EarServiceException(err, ioe);
                    }
                }
            }
        }

        URL[] clientAltDDs = new URL[altDDClients.length];
        for (int i = 0; i < altDDClients.length; i++) {
            if (altDDClients[i] != null) {
                altdd = altDDClients[i];
                if (altdd != null) {
                    try {
                        fAltDD = new File(new URL(dirUnpackURL.toExternalForm() + File.separator + altdd).getFile());
                        clientAltDDs[i] = fAltDD.getCanonicalFile().toURL();
                    } catch (MalformedURLException e) {
                        String err = "Can't build URL for alt-dd '" + altdd;
                        logger.log(BasicLevel.ERROR, err + "': " + e.getMessage());
                        throw new EarServiceException(err, e);
                    } catch (IOException ioe) {
                        String err = "Can't get canonicalFile() for the file '" + fAltDD;
                        logger.log(BasicLevel.ERROR, err + "': " + ioe.getMessage());
                        throw new EarServiceException(err, ioe);
                    }
                }
            }
        }
        //Extract roles name from the securityRole array
        String[] roleNames = new String[securityRoles.length];
        String affRoleNames = "";
        for (int i = 0; i < securityRoles.length; i++) {
            roleNames[i] = securityRoles[i];
            affRoleNames += roleNames[i] + ";";
        }

        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "role names = " + affRoleNames);
        }

        /**
         * Now this is the deployment step with sending
         * + rars to the rar service
         * + jars to the ejb service
         * + wars to the web container service
         * + jars and wars to web services service
         */

        /**
         * Ejb ClassLoader is needed for WebServices deployment so We create it
         * in advance ...
         */
        // Set the list of the ejb-jar, war and clients that can do an ejb-link in this
        // ear application. This array can be empty.
        // The alternate urls are given
        EjbManagerWrapper.setAvailableEjbJarsAndAltDDs(earClassLoader, jarUrls, ejbsAltDDs);
        WebManagerWrapper.setAltDD(earClassLoader, warUrls, warsAltDDs);
        ClientManagerWrapper.setAltDD(earClassLoader, clientUrls, clientAltDDs);

        // Construct the ejb classloader for all the ejb-jars of the same
        // ear application. Because there is one ejb classloader for all
        // the ejb-jars of the same ear application.
        URL[] urls = new URL[jarUrls.length + classpathURLs.length];
        System.arraycopy(jarUrls, 0, urls, 0, jarUrls.length);
        System.arraycopy(classpathURLs, 0, urls, jarUrls.length, classpathURLs.length);

        // Set the list of the rars that can be referenced in this
        // ear application. This array can be empty.
        // The alternate urls are given

        //rarDDManager.setAvailableRarsAndAltDDs(earClassLoader, connectorUrls,
        // connectorsAltDDs);

        //Only if the connectorUrls is not empty
        //And if the service is started
        if (resourceService != null && (connectorUrls.length > 0)) {
            try {
                ComponentContext contctx = null;
                try {
                    contctx = new ComponentContext(dirUnpackURL.getFile());
                    contctx.rebind("earUrl", earUrl[0]);
                    contctx.rebind("urls", connectorUrls);
                    contctx.rebind("earClassLoader", earClassLoader);
                    contctx.rebind("altDDs", connectorsAltDDs);
                } catch (NamingException e) {
                    String err = "Can not bind params for the resource service, Can't deploy rars ";
                    throw new ResourceServiceException(err, e);
                }
                resourceService.deployRars(contctx);
            } catch (ServiceException e) {
                String err = "Error during the deployment of the rars files of the Ear file " + fileName;
                logger.log(BasicLevel.ERROR, err + "': " + e.getMessage());
                //remove DDesc
                //resourceService.removeCache(earClassLoader);
                throw new EarServiceException(err, e);
            }
        }

        // Check GenIC has to be done after RAR deployment because
        // RAR resources (classes, ...) are added to the Application ClassLoader
        // only at deploy time.

        /**
         * Check that GenIC was applied on ejb-jar prior to sending them to EJB service.
         * As classloader is defined in this service, classes after will not be taken
         * into account. So GenIC should be called before creating Classloader.
         */
        if (jarUrls.length > 0 && ejbService != null) {

            URL[] compilationURLs = null;
            // if we have some resources, we have to use them in GenIC compilation
            if (resourceService != null) {
                // get the resources from the ear classloader and apps classloader

                // content of the ear loader (application wide resources)
                URL[] myApplicationJars = earClassLoader.getURLs();
                // content of the apps loader (system wide resources)
                URL[] appsJars = ((URLClassLoader) earClassLoader.getParent()).getURLs();

                // merge the 3 Set of URLs
                compilationURLs = new URL[myApplicationJars.length + appsJars.length + urls.length];

                System.arraycopy(urls, 0, compilationURLs, 0, urls.length);
                System.arraycopy(appsJars, 0, compilationURLs, urls.length, appsJars.length);
                System.arraycopy(myApplicationJars, 0, compilationURLs, urls.length + appsJars.length, myApplicationJars.length);

            }

            // Check GenIC :
            for (int i = 0; i < jarUrls.length; i++) {
                ejbService.checkGenIC(jarUrls[i].getFile(), compilationURLs);
            }
        }

        URLClassLoader ejbClassLoader;
        try {
            ejbClassLoader = new EjbJarClassLoader(urls, earClassLoader);
        } catch (IOException ioe) {
            String err = "Cannot Create EJB ClassLoader for EAR '" + fileName + "'";
            logger.log(BasicLevel.ERROR, err + " : " + ioe.getMessage());
            throw new EarServiceException(err, ioe);
        }

        // Only if the WebServicesService is started
        if (wsService != null) {
            try {
                Hashtable ctxRoots = new Hashtable();
                for (int w = 0; w < webs.length; w++) {
                    ctxRoots.put(warUrls[w], webs[w].getContextRoot());
                }
                ComponentContext contctx = null;
                try {
                    contctx = new ComponentContext(dirUnpackURL.getFile());
                    contctx.rebind("unpackDir", dirUnpackURL.toExternalForm());
                    contctx.rebind("jarUrls", jarUrls);
                    contctx.rebind("warUrls", warUrls);
                    contctx.rebind("earURL", earUrl[0]);
                    contctx.rebind("ejbClassLoader", ejbClassLoader);
                    contctx.rebind("earClassLoader", earClassLoader);
                    contctx.rebind("warCtxRootMapping", ctxRoots);
                } catch (NamingException e) {
                    String err = "Cannot bind params for the WebServices service, Can't deploy Web Services Endpoints ";
                    throw new WSServiceException(err, e);
                }
                wsService.deployWebServices(contctx);
            } catch (ServiceException se) {
                String err = "Error during the deployment of the WebServices of the Ear file " + fileName;
                logger.log(BasicLevel.ERROR, err + " : " + se.getMessage());
                wsService.removeCache(earClassLoader);
                try {
                    if (logger.isLoggable(BasicLevel.DEBUG)) {
                        logger.log(BasicLevel.DEBUG, "Undeploying Rars of the ear " + fileName);
                    }
                    if (resourceService != null && connectorUrls.length > 0) {
                        // remove Resource
                        resourceService.unDeployRars(connectorUrls, earUrl[0]);
                    }
                } catch (ServiceException se2) {
                    err = "Error during the undeployment of the rars files of the Ear file " + fileName;
                    logger.log(BasicLevel.ERROR, err + "': " + se2.getMessage());
                }

                throw new EarServiceException(err, se);
            }
        }
        // ejbClassLoader created above

        //Only if the jarUrls is not empty
        //And if the service is started
        if (ejbService != null && (jarUrls.length > 0)) {
            try {
                ComponentContext contctx = null;
                try {
                    contctx = new ComponentContext(dirUnpackURL.getFile());
                    contctx.rebind("earRootUrl", dirUnpackURL);
                    contctx.rebind("earUrl", earUrl[0]);
                    contctx.rebind("jarURLs", jarUrls);
                    contctx.rebind("earClassLoader", earClassLoader);
                    contctx.rebind("ejbClassLoader", ejbClassLoader);
                    contctx.rebind("roleNames", roleNames);
                } catch (NamingException e) {
                    String err = "Can not bind params for the ejb service, Can't deploy jars ";
                    throw new EarServiceException(err, e);
                }
                ejbService.deployJars(contctx);
            } catch (ServiceException e) {
                String err = "Error during the deployment of the jars files of the Ear file " + fileName;
                logger.log(BasicLevel.ERROR, err + "': " + e.getMessage());
                //remove DDesc
                ejbService.removeCache(earClassLoader);
                // Remove the deployed rars
                if (resourceService != null && connectorUrls.length > 0) {
                    // remove Resource
                    resourceService.unDeployRars(connectorUrls, earUrl[0]);
                }
                throw new EarServiceException(err, e);
            }
        }

        /*
         * We have deploy the ejb-jar files if the service is active And we have
         * the ejbClassLoader
         */

        // Link policy context of EJBs and Webs components
        linkPolicyObjects(earDD.getUserToRoleMapping(), jarUrls, warUrls, warsContextRoots);
        // commit ejb policy context (make them available for checking security)
        commitEJBPolicyObjects(jarUrls);

        /*
         * We're going to deploy wars Parent ClassLoader for web container is
         * ejbclassloader if ejbService is active, and a classloader with the
         * visibility of jar file if ejbservice is not active.
         */

        ClassLoader parentWarClassLoader = ejbClassLoader;

        //Only if the warUrls is not empty
        //And if the service is started
        if (webContainerService != null && (warUrls.length > 0)) {
            try {
                ComponentContext contctx = null;
                try {
                    contctx = new ComponentContext(dirUnpackURL.getFile());
                    contctx.rebind("earURL", earUrl[0]);
                    contctx.rebind("urls", warUrls);
                    contctx.rebind("parentClassLoader", parentWarClassLoader);
                    contctx.rebind("earClassLoader", earClassLoader);
                    contctx.rebind("altDDs", warsAltDDs);
                    contctx.rebind("contextRoots", warsContextRoots);
                } catch (NamingException e) {
                    String err = "Can not bind params for the web container service, Can't deploy wars ";
                    throw new EarServiceException(err, e);
                }
                webContainerService.deployWars(contctx);

            } catch (JWebContainerServiceException e) {
                String err = "Error during the deployment of the wars file of the Ear file " + fileName;
                logger.log(BasicLevel.ERROR, err + "': " + e.getMessage());
                //remove DDesc
                webContainerService.removeCache(earClassLoader);
                err = "Undeploy the jars loaded from this ear";
                if (logger.isLoggable(BasicLevel.DEBUG)) {
                    logger.log(BasicLevel.DEBUG, err);
                }
                try {
                    if (ejbService != null && (jarUrls.length > 0)) {
                        //remove DDesc
                        ejbService.removeCache(earClassLoader);
                        //remove the ejbs
                        ejbService.unDeployJars(jarUrls);
                    }
                    // Remove the deployed rars
                    if (resourceService != null && connectorUrls.length > 0) {
                        // remove Resource
                        resourceService.unDeployRars(connectorUrls, earUrl[0]);
                    }
                } catch (ServiceException se) {
                    err = "Error during the undeployment of the jars file of the Ear file " + fileName;
                    logger.log(BasicLevel.ERROR, err + "': " + se.getMessage());
                    throw new EarServiceException(err, e);
                }

                throw new EarServiceException(err, e);
            }
        }

        // Commit JACC policy configuration objects of WebApp components.
        commitWebBPolicyObjects(warUrls, warsContextRoots);

        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "We store the rars/wars/jars associated to the url :" + earUrl[0]);
        }

        // Complete the Deployment of the WebServices
        // Only if the WebServicesService is started
        if (wsService != null) {
            try {
                ComponentContext contctx = null;
                try {
                    contctx = new ComponentContext(earRootUrl.getFile());
                    contctx.rebind(WebServicesService.CLASSLOADER_CTX_PARAM, earClassLoader);
                    ObjectName name = J2eeObjectName.J2EEApplication(getDomainName(),
                                                                     getJonasServerName(),
                                                                     ModuleNamingUtils.fromURL(earUrl[0]));
                    contctx.rebind(WebServicesService.PARENT_OBJECTNAME_CTX_PARAM , name);
                    contctx.rebind(WebServicesService.ISINEAR_CTX_PARAM, Boolean.TRUE);

                } catch (NamingException e) {
                    String err = "Can not bind params for the WebServices service, "
                            + "can't complete deployment of Web Services Endpoints";
                    throw new JWebContainerServiceException(err, e);
                }
                wsService.completeWSDeployment(contctx);
            } catch (ServiceException se) {
                String err = "Error during the deployment of the WebServices of the Ear file '" + earRootUrl + "'";
                logger.log(BasicLevel.ERROR, err + " : " + se.getMessage());
                try {
                    if (logger.isLoggable(BasicLevel.DEBUG)) {
                        logger.log(BasicLevel.DEBUG, "Undeploy the WebApps loaded from this ear");
                    }
                    if (webContainerService != null && (warUrls.length > 0)) {
                        //remove DDesc
                        webContainerService.removeCache(earClassLoader);
                        //remove the webapps
                        webContainerService.unDeployWars(warUrls);
                    }
                    if (logger.isLoggable(BasicLevel.DEBUG)) {
                        logger.log(BasicLevel.DEBUG, "Undeploy the EjbJars loaded from this ear");
                    }
                    if (ejbService != null && (jarUrls.length > 0)) {
                        //remove DDesc
                        ejbService.removeCache(earClassLoader);
                        //remove the ejbs
                        ejbService.unDeployJars(jarUrls);
                    }
                    // Remove the deployed rars
                    if (logger.isLoggable(BasicLevel.DEBUG)) {
                        logger.log(BasicLevel.DEBUG, "Undeploy the Resources loaded from this ear");
                    }
                    if (resourceService != null && connectorUrls.length > 0) {
                        // remove Resource
                        resourceService.unDeployRars(connectorUrls, earUrl[0]);
                    }
                } catch (ServiceException se2) {
                    err = "Error during the undeployment of the webapps/ejbjars/rars file of the Ear file " + fileName;
                    logger.log(BasicLevel.ERROR, err + "': " + se2.getMessage());
                }
                throw new JWebContainerServiceException(err, se);
            }
        }

        // Set the timestamp dir
        File dirUnpackFile = new File(dirUnpackURL.getFile());
        dirUnpackFile.setLastModified((new Date()).getTime());

        // Admin code
        // General info
        String domainName = getDomainName();
        String serverName = getJonasServerName();
        String j2eeApplicationName = ModuleNamingUtils.fromURL(earUrl[0]);
        // Step 1: treat client modules. For each module, get admin information
        // and put this information in a AppClientModule instance.
        AppClientModuleInfo[] appClientModules = new AppClientModuleInfo[clientUrls.length];
        for (int i = 0; i < clientUrls.length; i++) {
            URL clientUrl = clientUrls[i];
            String moduleName =  ModuleNamingUtils.fromURL(clientUrl);
            String moduleFileName = clientUrl.getPath();
            String moduleDD = null;
            String jonasModuleDD = null;
            ClassLoader moduleCL = null;
            try {
                moduleCL = new ClientClassLoader(clientUrl, ejbClassLoader);
            } catch (IOException ioe) {
                // fail case: we use the ejbjar classloader, but the descriptor loading will fail
                // TODO Find a better Exception handling
                moduleCL = ejbClassLoader;
            }
            try {
                ClientContainerDeploymentDesc clientContainerDD = ClientManagerWrapper.getDeploymentDesc(clientUrl, moduleCL, earClassLoader);
                moduleDD = clientContainerDD.getXmlContent();
                jonasModuleDD = clientContainerDD.getJOnASXmlContent();
            } catch (ClientContainerDeploymentDescException e) {
                String err = "Cannot read the deployment descriptors ': " + clientUrls[i] + "'" + e.toString();
                logger.log(BasicLevel.WARN, err, e);
            }
            appClientModules[i] = new AppClientModuleInfo(moduleName, moduleFileName, moduleDD, jonasModuleDD);
        }
        registerAppClientModuleMBeans(domainName, serverName, j2eeApplicationName, appClientModules);
        // Step 2: register MBean for the ear
        String unpackDir = dirUnpackURL.getFile();
        String ddXmlContent = earDD.getXmlContent();
        Ear ear = registerJ2EEApplicationModule(domainName, serverName, j2eeApplicationName, unpackDir
                , earUrl[0], ddXmlContent, jarUrls, warUrls, connectorUrls);
        ears.put(earUrl[0], ear);

        // Remove the Deployment descriptors
        if (ejbService != null) {
            ejbService.removeCache(earClassLoader);
        }

        if (webContainerService != null) {
            webContainerService.removeCache(earClassLoader);
        }

        if (wsService != null) {
            wsService.removeCache(earClassLoader);
        }

        logger.log(BasicLevel.INFO, "Ear " + earUrl[0] + " available.");

        // return ObjectName of the J2EEApplication MBean
        return J2eeObjectName.J2EEApplicationName(domainName, serverName, j2eeApplicationName);
    }

    /**
     * Deploy an ear with its given fileName.
     * @param fileName file to deploy (ear)
     * @return The ObjectName of the J2EE Application MBean associated to the
     *         deployed EAR
     * @throws Exception if the deployment of the EAR failed.
     * @see org.ow2.jonas.ear.EarService#deployEar(java.lang.String)
     */
    public String deployEar(final String fileName) throws Exception {
        Context ctx = null;
        String mbeanOn = null;
        try {
            ctx = new ComponentContext(fileName);
            ctx.rebind("filename", fileName);
        } catch (NamingException e) {
            String err = "Error when deploying the ear file ";
            logger.log(BasicLevel.ERROR, err, e);
            throw e;
        }
        try {
            mbeanOn = deployEar(ctx);
        } catch (ServiceException e) {
            logger.log(BasicLevel.ERROR, "Cannot deploy file '" + fileName + "'");
            throw new Exception("Cannot deploy file '" + fileName + "'", e);
        }
        return mbeanOn;
    }

    /**
     * Start the EAR service.
     * @throws ServiceException if the startup failed.
     */
    protected void doStart() throws ServiceException {
        // Reset embedded server linked to the ear deployer for the JOnAS/Traditional case.
        // In fact, this has been done previously during the service injection but with an EZB Service not started.
        if (easyBeansService != null) {
            earDeployer.setEmbedded(easyBeansService.getEasyBeansServer());
        }

        // get apps ClassLoader
        try {
            LoaderManager lm = LoaderManager.getInstance();
            appsClassLoader = lm.getAppsLoader();
        } catch (Exception e) {
            logger.log(BasicLevel.ERROR, "Cannot get the Applications ClassLoader from EAR Container Service: " + e);
            throw new ServiceException("Cannot get the Applications ClassLoader from EAR Container Service", e);
        }

        if (jmxService != null) {
            // TODO, When ear service will become a bundle, we could use
            // the meta-data located in META-INF/mbeans-descriptors.xml
            jmxService.loadDescriptors(getClass().getPackage().getName(),
                                getClass().getClassLoader());
        }

        //url
        URL earApps = null;
        try {
            earApps = new File(WORK_APPS_DIR + File.separator + getJonasServerName()).toURL();
        } catch (MalformedURLException mue) {
            throw new ServiceException("Error when trying to get the URL of the jonasroot/apps directory", mue);
        }

        File fLog = new File(earApps.getFile() + File.separator + getJonasServerName() + ".log");
        if (!fLog.exists()) {
            try {
                //create
                fLog.getParentFile().mkdirs();
                fLog.createNewFile();
            } catch (IOException e) {
                throw new ServiceException("cannot create the log file" + fLog, e);
            }
        }

        //get the logger
        try {
            earDeployerLog = new DeployerLog(fLog);
        } catch (DeployerLogException e) {
            throw new ServiceException("Can not get an EarDeployerLog", e);
        }

        // create the ear deployment work file control task
        EarCleanTask earCleanTask = new EarCleanTask(this, earApps, earDeployerLog);

        //get the cleaner ref
        workCleaner = WorkCleaner.getInstance();

        // add the ear deployment work file control task
        try {
            workCleaner.registerTask(earCleanTask);
        } catch (CleanerException ce) {
            throw new ServiceException("Cannot register the EAR clean task", ce);
        }

        // force the cleaning now
        workCleaner.executeTasks();

        // set properties
        earDeployer.setAppsClassLoader(appsClassLoader);
        earDeployer.setJMXService(jmxService);

        // Register it
        deployerManager.register(earDeployer);

        // Deploy all ear which are in descriptors section
        Iterator it = earNames.iterator();
        while (it.hasNext()) {
            String fileName = (String) it.next();
            Context contctx = null;
            try {
                contctx = new ComponentContext(fileName);
                contctx.rebind("filename", fileName);
            } catch (NamingException e) {
                throw new ServiceException("Cannot start the EarService", e);
            }
            try {
                deployEar(contctx);
            } catch (EarServiceException ese) {
                String err = "Cannot deploy the file '" + fileName + "' : " + ese.getMessage();
                logger.log(BasicLevel.WARN, err, ese);
            } catch (Exception ue) {
                String err = "Cannot deploy the file (unexpected exception) '" + fileName + "' : " + ue.getMessage();
                logger.log(BasicLevel.WARN, err, ue);
            }
        }
        // Admin code
        registerEarServiceMBean(this, getDomainName());
    }

    /**
     * Undeploy an EAR by sending the request to the EJB container and to the
     * WEB container and the Resource service.
     * @param ctx the context which contains the configuration in order to
     *        undeploy an EAR.
     * @throws EarServiceException if the undeployment of the EAR failed.
     */
    public void unDeployEar(final Context ctx) throws EarServiceException {

        //Get the ear url param which we want to undeploy
        URL earUrl = null;
        try {
            earUrl = (URL) ctx.lookup("filename");
        } catch (NamingException e) {
            throw new EarServiceException("Trying to remove the ear file but there is no filename specified", e);
        }

        //This ear is in loaded EARs files ?
        Ear ear = (Ear) ears.get(earUrl);

        //Not loaded
        if (ear == null) {
            throw new EarServiceException("Trying to remove the ear file " + earUrl.getFile()
                    + " but this file was not found in the loaded Ear files");
        }

        //undeploy wars if the web container service is started
        URL[] warsToUndeploy = ear.getWars();
        if (webContainerService != null && (warsToUndeploy.length > 0)) {
            webContainerService.unDeployWars(warsToUndeploy);
        }

        //undeploy jars if the ejb service is started
        URL[] jarsToUndeploy = ear.getEjbJars();
        if (ejbService != null && (jarsToUndeploy.length > 0)) {
            ejbService.unDeployJars(jarsToUndeploy);
        }

        //undeploy rars if the resource service is started
        URL[] rarsToUndeploy = ear.getRars();
        if (resourceService != null && (rarsToUndeploy.length > 0)) {
            resourceService.unDeployRars(rarsToUndeploy, earUrl);
        }

        File f = new File(earUrl.getFile());
        //Undeploy successful, we remove the ear

        //Only if there is no Exception !
        ears.remove(earUrl);

        // Admin code
        String domainName = getDomainName();
        String serverName = getJonasServerName();
        String j2eeappName = ear.getName();
        unregisterJ2EEApplicationModule(domainName, serverName, j2eeappName);
        unregisterAppClientModuleMBeans(domainName, serverName, j2eeappName);

        logger.log(BasicLevel.INFO, "Ear " + f.getName() + " no longer available.");

    }

    /**
     * Undeploy an EAR by delegating the operation to the unDeployEar() method.
     * This is used for JMX management.
     * @param fileName the fileName of the ear which must be be undeployed.
     * @throws Exception if the undeployment of the EAR failed.
     * @see org.ow2.jonas.ear.EarService#unDeployEar(java.lang.String)
     */
    public void unDeployEar(final String fileName) throws Exception {
        /*
         * We have only the name of the file, not its associated path, so we
         * look in the current directory and in the ear applications directory
         */

        //We check if the url exists in the hashtable
        //compare: First in the current dir, after in the apps directory
        URL url = null;
        URL earUrl = null;
        boolean found = false;
        try {
            Enumeration e = ears.keys();
            url = new File(fileName).getCanonicalFile().toURL();
            while (e.hasMoreElements() && !found) {
                earUrl = (URL) e.nextElement();
                if (earUrl.equals(url)) {
                    //We have found
                    found = true;
                }
            }

            if (fileName.toLowerCase().endsWith(".ear") && !found) {
                // In case of the name is an ear check also in
                // the JONAS_BASE/apps directory
                String earFileName = APPS_DIR + File.separator + fileName;
                e = ears.keys();
                url = new File(earFileName).getCanonicalFile().toURL();
                while (e.hasMoreElements() && !found) {
                    earUrl = (URL) e.nextElement();
                    if (earUrl.equals(url)) {
                        //We have found
                        found = true;
                    }
                }
            }
            if (!found) {
                String err = "Cannot undeploy the ear '" + fileName + "', it is not deployed.";
                logger.log(BasicLevel.ERROR, err);
                throw new Exception(err);
            }
        } catch (MalformedURLException mue) {
            String err = "Error when trying to get the url from" + fileName;
            logger.log(BasicLevel.ERROR, err);
            throw new Exception(err, mue);
        } catch (IOException ioe) {
            String err = "Error when trying to get the canonical file from " + fileName;
            logger.log(BasicLevel.ERROR, err + ioe.getMessage());
            throw new Exception(err, ioe);
        }

        //We've got the file, now we bind the params
        Context ctx = null;
        try {
            ctx = new ComponentContext(fileName);
            //The fileName is an url
            ctx.rebind("filename", url);
        } catch (NamingException e) {
            String err = "Error when binding parameters";
            logger.log(BasicLevel.ERROR, err + e.getMessage());
            throw new Exception(err, e);
        }

        //Call the function
        try {
            unDeployEar(ctx);
        } catch (EarServiceException e) {
            throw new Exception(e);
        }

    }

    /**
     * @return current number of ears deployed in the JOnAS server
     */
    public Integer getCurrentNumberOfEars() {
        return new Integer(ears.size());
    }

    /**
     * Return the list of installed Applications. The EAR files or the
     * directories with expanded Applications are searched in JONAS_BASE/apps
     * and all Applications directories 'autoload'.
     * @return The list of EAR files or the directories with expanded
     *         Applications found
     * @throws Exception if the list can't be retrieved
     */
    public List getInstalledEars() throws Exception {
        // get EAR files found in JONAS_BASE/apps
        ArrayList al = JModule.getInstalledContainersInDir(APPS_DIR,
                                                           JModule.EAR_EXTENSION,
                                                           JModule.EAR_CHILD_DIR,
                                                           JModule.EAR_CONFIRM_FILE);
        // get EAR files found in all autoload directories
        Iterator it = autoloadDirectories.iterator();
        while (it.hasNext()) {
            String name = (String) it.next();
            al.addAll(JModule.getInstalledContainersInDir(name,
                                                          JModule.EAR_EXTENSION,
                                                          JModule.EAR_CHILD_DIR,
                                                          JModule.EAR_CONFIRM_FILE));
        }
        return al;
    }

    /**
     * This method is added temporarily. It will disapear when Ears will have
     * their associated MBeans (when Ears will become manageable)
     * @return the names of the ears currently deployed in the JOnAS server
     */
    public Set getEarNames() {
        HashSet names = new HashSet();
        URL earUrl = null;
        for (Enumeration e = ears.keys(); e.hasMoreElements();) {
            earUrl = (URL) e.nextElement();
            names.add(earUrl.getFile());
        }
        return names;
    }

    /**
     * Add recursively the ears of the specified directory. If the dir has a
     * relative path, this path is relative from where the Application Server is
     * launched. If the dir is not found it will be searched in
     * $JONAS_BASE/apps/ directory.
     * @param dirPath the path to the directory containing the ears to load.
     */
    private void addEars(final String dirPath) {
        boolean found = false;

        // Look the directory relative to the $JONAS_BASE/apps directory
        File    dir = new File(APPS_DIR + File.separator + dirPath);
        found = dir.isDirectory();

        if (found) {
            addEarsFrom(dir);
        } else {
            String err = "Warning: Cannot load dir: '" + dirPath + "' ";
            err += "is not a directory or directory doesn't exist";
            logger.log(BasicLevel.WARN, err);
        }
    }

    /**
     * Add the ears of the specified directory.
     * @param dir the directory from which the ears are loaded.
     * @throws EarServiceException if the argument is not a directory
     */
    private void addEarsFrom(final File dir) throws EarServiceException {
        try {
            if (dir.isDirectory()) {
                File[] files = dir.listFiles();
                for (int i = 0; i < files.length; i++) {
                    if (files[i].getPath().toLowerCase().endsWith(".ear")) {
                        earNames.add(files[i].getCanonicalPath());
                    } else {
                        if (files[i].isDirectory()) {
                            addEarsFrom(files[i]);
                        }
                    }
                }
            } else {
                String err = "Cannot load dir: '" + dir.getPath();
                err += "' is not a directory";
                logger.log(BasicLevel.ERROR, err);
                throw new EarServiceException(err);
            }
        } catch (IOException e) {
            String err = "Invalid file name '" + dir.getPath();
            logger.log(BasicLevel.ERROR, err);
            throw new EarServiceException(err, e);
        }
    }

    /**
     * Test if the specified filename is already deployed or not. This method
     * is a management method provided by the EarServerice MBean.
     * @param fileName the name of the ear file.
     * @return true if the ear is deployed, else false.
     */
    public boolean isEarLoaded(final String fileName) {

        URL url = null;
        boolean isLoaded = false;
        try {
            // Absolute filename
            try {
                url = new File(fileName).getCanonicalFile().toURL();
                //Check if the ear is already deployed or not
                if (ears.get(url) != null) {
                    isLoaded = true;
                } else {
                    // Not found force to test in relative Apps directory
                    url = null;
                }
            } catch (Exception e) {
                url = null;
            }
            // Relative filename
            if (url == null) {
                url = new File(APPS_DIR + File.separator + fileName).getCanonicalFile().toURL();
                //Check if the ear is already deployed or not
                if (ears.get(url) != null) {
                    isLoaded = true;
                }
            }
        } catch (Exception e) {
            String err = "Can not found if the ear is deployed or not";
            logger.log(BasicLevel.ERROR, err);
            return false;
        }

        return isLoaded;
    }

    /**
     * Test if the specified filename is already deployed or not. This
     * method is defined in the EarService interface.
     * @param fileName the name of the ear file.
     * @return true if the ear is deployed, else false.
     */
    public Boolean isEarDeployed(final String fileName) {
        return new Boolean(isEarLoaded(fileName));
    }

    /**
     * Test if the specified unpack name is already deployed or not. This
     * method is defined in the EarService interface.
     * @param unpackName the name of the ear file.
     * @return true if the ear is deployed, else false.
     */
    public boolean isEarDeployedByUnpackName(final String unpackName) {
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "entering for unpackName= " + unpackName);
        }
        // for each ear loaded
        Enumeration lc = ears.elements();
        while (lc.hasMoreElements()) {
            Ear ear = (Ear) lc.nextElement();

            // get unpack name of the ear
            String deployedUnpackName = new File(ear.getUnpackName()).getName();
            if (logger.isLoggable(BasicLevel.DEBUG)) {
                logger.log(BasicLevel.DEBUG, "deployedUnpackName=" + deployedUnpackName);
            }

            if (deployedUnpackName.equals(unpackName)) {
                return true;
            }
            // else, go to the next loop
        }
        // not found
        return false;
    }


    /**
     * Return the list of all loaded Applications.
     * @return The list of deployed Applications
     */
    public List getDeployedEars() {
        ArrayList al = new ArrayList();
        Ear oEar;
        URL oURL;
        for (Enumeration enumEars = ears.elements(); enumEars.hasMoreElements();) {
            oEar = (Ear) enumEars.nextElement();
            oURL = oEar.getEarUrl();
            al.add(oURL.getFile());
        }
        return al;
    }

    /**
     * Return the list of installed Applications ready to deploy.
     * @return The list of deployable Applications
     * @throws Exception if the list can't be built
     */
    public List getDeployableEars() throws Exception {
        List al = getInstalledEars();
        al.removeAll(getDeployedEars());
        return al;
    }

    /**
     * Return the list of "autoload" directories for applications.
     * @return The list of all "autoload" directories
     */
    public List getAutoloadDirectories() {
        ArrayList al = new ArrayList();
        Iterator it = autoloadDirectories.iterator();
        while (it.hasNext()) {
            String fileName = (String) it.next();
            try {
                al.add(new File(fileName).toURL().getPath());
            } catch (Exception e) {
                if (logger.isLoggable(BasicLevel.DEBUG)) {
                    logger.log(BasicLevel.DEBUG, "Can't get autoload directories : " + e.getMessage());
                }
            }
        }
        return al;
    }

    /**
     * Return the Apps directory.
     * @return The Apps directory
     */
    public String getAppsDirectory() {
        String sRet = null;
        try {
            sRet = (new File(APPS_DIR)).toURL().getPath();
        } catch (Exception e) {
            throw new RuntimeException("Cannot get the APPS directory", e);
        }
        return sRet;
    }

    /**
     * @return policy configuration factory
     */
    private PolicyConfigurationFactory getPolicyConfigurationFactory() {
        PolicyConfigurationFactory pcFactory = null;
        try {
            pcFactory = PolicyConfigurationFactory.getPolicyConfigurationFactory();
        } catch (Exception cnfe) {
            String err = "Cannot retrieve current policy configuration factory";
            logger.log(BasicLevel.ERROR, err + "': " + cnfe.getMessage());
            throw new EarServiceException(err, cnfe);
        }
        return pcFactory;
    }

    /**
     * Add context-id for given jar urls to a given List.
     * @param jarUrls list of EJB-Jars
     * @param contextIDs the list of context-id.
     */
    private void addEjbContextIdToList(final URL[] jarUrls, final List contextIDs) {
        // Get contextID of EJB
        if (ejbService != null) {
            for (int u = 0; u < jarUrls.length; u++) {
                contextIDs.add(ejbService.getContainerContextID(jarUrls[u].getFile()));
            }
        }
    }

    /**
     * Add context-id for given web urls to a given List.
     * Also, reset the context-id associated to it.
     * @param warUrls list of EJB-Jars
     * @param contextIDs the list of context-id.
     * @param contextRoots the name of the context-root of the web applications.
     * @param resetPolicyConfiguration reset or not the associated policy configuration.
     * @throws EarServiceException if policy context cannot be get.
     */
    private void addWebBContextIdToList(final URL[] warUrls,
                                        final String[] contextRoots,
                                        final List contextIDs,
                                        final boolean resetPolicyConfiguration) throws EarServiceException {
        if (warUrls != null) {
            for (int u = 0; u < warUrls.length; u++) {
                // build context ID of war
                String ctxId = warUrls[u].getFile() + contextRoots[u];
                // reset the policy configuration associated to this context ID.
                if (resetPolicyConfiguration) {
                    try {
                        getPolicyConfigurationFactory().getPolicyConfiguration(ctxId, true);
                    } catch (PolicyContextException pce) {
                        String err = "Cannot retrieve a policy configuration";
                        logger.log(BasicLevel.ERROR, err + "': " + pce.getMessage());
                        throw new EarServiceException(err, pce);
                    }
                }
                contextIDs.add(ctxId);
            }
        }
    }


    /**
     * Link policy configuration objects of EJB and Web Component.
     * @param userToRoleMapping mapping for user-to-role
     * @param jarUrls list of jars which have been deployed
     * @param warUrls list of wars which have been deployed
     * @param contextRoots list of context-root elements for each war url
     * @throws EarServiceException if the policy objects can't be linked
     */
    private void linkPolicyObjects(final Map userToRoleMapping,
                                   final URL[] jarUrls,
                                   final URL[] warUrls,
                                   final String[] contextRoots) throws EarServiceException {
        List ctxIDs = new LinkedList();
        // Get contextID of EJB
        addEjbContextIdToList(jarUrls, ctxIDs);

        // Now for WebApp
        addWebBContextIdToList(warUrls, contextRoots, ctxIDs, true);

        try {
            // Now link the policy configuration objects
            for (Iterator itCtxId = ctxIDs.iterator(); itCtxId.hasNext();) {
                String toBeLinkedCtxId = (String) itCtxId.next();
                PolicyConfiguration toBeLinkedPC = getPolicyConfigurationFactory().getPolicyConfiguration(toBeLinkedCtxId, false);
                for (Iterator linkCId = ctxIDs.iterator(); linkCId.hasNext();) {
                    String linkedCtxId = (String) linkCId.next();
                    if (!toBeLinkedCtxId.equals(linkedCtxId)) {
                        PolicyConfiguration linkedPC = getPolicyConfigurationFactory().getPolicyConfiguration(linkedCtxId, false);
                        toBeLinkedPC.linkConfiguration(linkedPC);
                    }
                }
            }
        } catch (PolicyContextException pce) {
            String err = "Cannot retrieve a policy configuration";
            logger.log(BasicLevel.ERROR, err + "': " + pce.getMessage());
            throw new EarServiceException(err, pce);
        }

        // Do user-to-role mapping
        if (userToRoleMapping != null) {
            for (Iterator itCtxId = ctxIDs.iterator(); itCtxId.hasNext();) {
                String contextId = (String) itCtxId.next();
                for (Iterator itMapping = userToRoleMapping.keySet().iterator(); itMapping.hasNext();) {
                    String principalName = (String) itMapping.next();
                    List roles = (List) userToRoleMapping.get(principalName);
                    String[] roleNames = (String[]) roles.toArray(new String[roles.size()]);
                    JPolicyUserRoleMapping.addUserToRoleMapping(contextId, principalName, roleNames);
                }
            }

        }
    }

    /**
     * Commit policy configuration objects of EJB Component.
     * @param jarUrls list of jars which have been deployed
     */
    private void commitEJBPolicyObjects(final URL[] jarUrls) {
        List ctxIDs = new LinkedList();
        addEjbContextIdToList(jarUrls, ctxIDs);
        commitPolicyObjects(ctxIDs);
    }


    /**
     * Commit policy configuration objects of Web Component.
    * @param warUrls list of wars which have been deployed
     * @param contextRoots list of context-root elements for each war url
     */
    private void commitWebBPolicyObjects(final URL[] warUrls, final String[] contextRoots) {
        List ctxIDs = new LinkedList();
        addWebBContextIdToList(warUrls, contextRoots, ctxIDs, false);
        commitPolicyObjects(ctxIDs);

    }

    /**
     * Commit policy context IDs of the given list.
     * @param ctxIDs list of context ID to commit.
     */
    private void commitPolicyObjects(final List ctxIDs) {
        String ctxId = null;
        try {
            // commit the policy configuration objects
            for (Iterator itCtxId = ctxIDs.iterator(); itCtxId.hasNext();) {
                ctxId = (String) itCtxId.next();
                PolicyConfiguration pc = getPolicyConfigurationFactory().getPolicyConfiguration(ctxId, false);
                pc.commit();
            }
        } catch (PolicyContextException pce) {
            String err = "Cannot commit policy configuration with Id '" + ctxId + "'";
            logger.log(BasicLevel.ERROR, err + "': " + pce.getMessage());
            throw new EarServiceException(err, pce);
        }

        // refresh policy
        Policy.getPolicy().refresh();
    }

    // Admin code implementation (JMX based)
    // -------------------------------------
    /**
     * Register EAR Service MBean.
     * @param service ear container service to manage
     * @param domainName domain name
     */
    private void registerEarServiceMBean(final Object service, final String domainName) {
        ObjectName on = JonasObjectName.earService(domainName);
        jmxService.registerMBean(service, on);
    }

    /**
     * Unregister EAR Container Service MBean.
     * @param domainName domain name
     */
    private void unregisterEarServiceMBean(final String domainName) {
        ObjectName on = JonasObjectName.earService(domainName);
        jmxService.unregisterMBean(on);
    }

    /**
     * Register an AppClientModule MBean corresponding to a client module within a j2ee application.
     * @param domainName domain name
     * @param serverName server name
     * @param j2eeappName app name
     * @param moduleName module name
     * @param moduleFileName module's file name
     * @param moduleDD module's deployment descriptor
     * @param moduleJonasDD module's JOnAS deployment descriptor
     */
    private void registerAppClientModuleMBean(final String domainName,
                                              final String serverName,
                                              final String j2eeappName,
                                              final String moduleName,
                                              final String moduleFileName,
                                              final String moduleDD,
                                              final String moduleJonasDD) {
        String mbeanName = J2eeObjectName.getAppClientModuleName(domainName, serverName, j2eeappName, moduleName);
        AppClientModule mbean = new AppClientModule(mbeanName, moduleFileName, moduleDD, moduleJonasDD);
        try {
            jmxService.registerModelMBean(mbean, mbeanName);
        } catch (Exception e) {
            logger.log(BasicLevel.WARN, "Could not register AppClientModule MBean " + moduleName, e);
        }
    }

    /**
     * Create and register all the AppClientModule MBeans corresponding to the client modules of a j2ee application.
     * @param domainName domain name
     * @param serverName server name
     * @param j2eeappName j2ee application name
     * @param appClientModules array of AppClientModuleInfo objects containing admin info for each module
     */
    private void registerAppClientModuleMBeans(final String domainName,
                                               final String serverName,
                                               final String j2eeappName,
                                               final AppClientModuleInfo[] appClientModules) {
        if (appClientModules.length == 0) {
            return;
        }
        for (int i = 0; i < appClientModules.length; i++) {
            AppClientModuleInfo appClientModule = appClientModules[i];
            registerAppClientModuleMBean(domainName, serverName, j2eeappName
                    , appClientModule.getModuleName(), appClientModule.getFileName()
                    , appClientModule.getDeploymentDesc(), appClientModule.getJonasDeploymentDesc());
        }
    }

    /**
     * Unregister all the AppClientModule MBeans corresponding to the client modules of a j2ee application.
     * @param domainName domain name
     * @param serverName server name
     * @param j2eeappName j2ee application name
     */
    private void unregisterAppClientModuleMBeans(final String domainName,
                                                 final String serverName,
                                                 final String j2eeappName) {
        MBeanServer mbeanServer = jmxService.getJmxServer();
        ObjectName appClientModuleOns = J2eeObjectName.getAppClientModules(domainName, serverName, j2eeappName);
        Iterator it = mbeanServer.queryNames(appClientModuleOns, null).iterator();
        while (it.hasNext()) {
            ObjectName on = (ObjectName) it.next();
            jmxService.unregisterModelMBean(on);
        }
    }
    /**
     * Create and register the MBean associated to the J2EEApplication module (ear).
     * @param domainName domain name
     * @param serverName server name
     * @param j2eeApplicationName j2ee application name
     * @param unpackDir name of the working copy of the ear
     * @param earUrl the url of this ear
     * @param ddXmlContent the deployment descriptor of the ear
     * @param jarUrls the URLs of ejb-jar files
     * @param warUrls the URLs of the war files
     * @param connectorUrls the URLs of the rar files
     * @return the created MBean
     */
    private Ear registerJ2EEApplicationModule(final String domainName,
                                              final String serverName,
                                              final String j2eeApplicationName,
                                              final String unpackDir,
                                              final URL earUrl,
                                              final String ddXmlContent,
                                              final URL[] jarUrls,
                                              final URL[] warUrls,
                                              final URL[] connectorUrls) {

        String name = J2eeObjectName.J2EEApplicationName(domainName, serverName, j2eeApplicationName);
        // Store the ejb-jar, war and the rar urls associated with the ear URL
        // in case of success deploy
        Ear mbean = new Ear(name, j2eeApplicationName, unpackDir, earUrl, ddXmlContent, jarUrls, warUrls,
                connectorUrls, jmxService);
        try {
            jmxService.registerModelMBean(mbean, name);
        } catch (Exception e) {
            logger.log(BasicLevel.WARN, "Cannot register MBean for J2EEApplicationModule MBean " + j2eeApplicationName, e);
            return null;
        }
        return mbean;
    }

    /**
     * Unregister the MBean associated to the J2EEApplication module (ear).
     * @param domainName domain name
     * @param serverName server name
     * @param j2eeApplicationName j2ee application name
     */
    private void unregisterJ2EEApplicationModule(String domainName, String serverName, String j2eeApplicationName) {
        ObjectName on = J2eeObjectName.J2EEApplication(domainName, serverName, j2eeApplicationName);
        jmxService.unregisterModelMBean(on);
    }

    /**
     * @param jmxService the jmxService to set
     */
    public void setJmxService(final JmxService jmxService) {
        this.jmxService = jmxService;
    }

    /**
     * @param ejbService the ejbService to set
     */
    public void setEjbService(final EJBService ejbService) {
        this.ejbService = ejbService;
        earDeployer.setEjb21Service(ejbService);
    }

    /**
     * Unbind the {@link EJBService}.
     */
    public void unsetEjbService() {
        this.ejbService = null;
        earDeployer.setEjb21Service(null);
    }

    /**
     * @param webContainerService the webContainerService to set
     */
    public void setWebContainerService(final JWebContainerService webContainerService) {
        this.webContainerService = webContainerService;
        earDeployer.setWebContainerService(webContainerService);
    }

    /**
     * Unbind the {@link JWebContainerService}.
     */
    public void unsetWebContainerService() {
        this.webContainerService = null;
        earDeployer.setWebContainerService(null);
    }

    /**
     * @param wsService the wsService to set
     */
    public void setWsService(final WebServicesService wsService) {
        this.wsService = wsService;
        earDeployer.setWsService(wsService);
    }

    /**
     * Unbind the {@link WebServicesService}.
     */
    public void unsetWsService() {
        this.wsService = null;
        earDeployer.setWsService(null);
    }

    /**
     * @param resourceService the resourceService to set
     */
    public void setResourceService(final ResourceService resourceService) {
        this.resourceService = resourceService;
        earDeployer.setResourceService(resourceService);
    }

    /**
     * Unbind the {@link ResourceService}.
     */
    public void unsetResourceService() {
        this.resourceService = null;
        earDeployer.setResourceService(null);
    }

    /**
     * @param service the EJB3 Service to be injected.
     */
    public void setEasyBeansService(final IEasyBeansService service) {
        this.easyBeansService = service;
        earDeployer.setEmbedded(easyBeansService.getEasyBeansServer());
    }

    /**
     * Unbind the {@link IEasyBeansService}.
     */
    public void unsetEasyBeansService() {
        this.easyBeansService = null;
        earDeployer.setEmbedded(null);
    }

    /**
     * @param deployerManager the deployerManager to set
     */
    public void setDeployerManager(final IDeployerManager deployerManager) {
        this.deployerManager = deployerManager;
    }

    /**
     * Unbind the {@link IDeployerManager}.
     */
    public void unsetDeployerManager() {
        this.deployerManager = null;
    }

}
