/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 1999-2007 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JOnASEJBService.java 12661 2008-01-23 15:13:30Z fornacif $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.ejb.internal;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Vector;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import javax.ejb.NoSuchObjectLocalException;
import javax.ejb.Timer;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.resource.spi.work.WorkManager;

import org.ow2.jonas.Version;
import org.ow2.jonas.cmi.CmiService;
import org.ow2.jonas.deployment.common.DeploymentDescException;
import org.ow2.jonas.deployment.ejb.BeanDesc;
import org.ow2.jonas.deployment.ejb.DeploymentDesc;
import org.ow2.jonas.deployment.ejb.EntityBmpDesc;
import org.ow2.jonas.deployment.ejb.EntityCmpDesc;
import org.ow2.jonas.deployment.ejb.MessageDrivenDesc;
import org.ow2.jonas.deployment.ejb.SessionStatefulDesc;
import org.ow2.jonas.deployment.ejb.SessionStatelessDesc;
import org.ow2.jonas.deployment.ejb.lib.EjbDeploymentDescManager;
import org.ow2.jonas.ejb.EJBService;
import org.ow2.jonas.ejb.JTimerHandleInfo;
import org.ow2.jonas.ejb.internal.delegate.HandleDelegateCCFDelegate;
import org.ow2.jonas.ejb.internal.delegate.JTimerHandleDelegate;
import org.ow2.jonas.ejb.internal.mbean.EJBModule;
import org.ow2.jonas.ejb.internal.mbean.EntityBean;
import org.ow2.jonas.ejb.internal.mbean.MessageDrivenBean;
import org.ow2.jonas.ejb.internal.mbean.StatefulSessionBean;
import org.ow2.jonas.ejb.internal.mbean.StatelessSessionBean;
import org.ow2.jonas.generators.genic.wrapper.GenicServiceWrapper;
import org.ow2.jonas.generators.wsgen.wrapper.WsGenWrapper;
import org.ow2.jonas.ha.HaService;
import org.ow2.jonas.jmx.JmxService;
import org.ow2.jonas.lib.bootstrap.JProp;
import org.ow2.jonas.lib.bootstrap.LoaderManager;
import org.ow2.jonas.lib.ejb21.BeanFactory;
import org.ow2.jonas.lib.ejb21.Container;
import org.ow2.jonas.lib.ejb21.JContainer;
import org.ow2.jonas.lib.ejb21.JEntityFactory;
import org.ow2.jonas.lib.ejb21.JEntitySwitch;
import org.ow2.jonas.lib.ejb21.JFactory;
import org.ow2.jonas.lib.ejb21.JSessionFactory;
import org.ow2.jonas.lib.ejb21.JStatelessFactory;
import org.ow2.jonas.lib.ejb21.JTimerService;
import org.ow2.jonas.lib.ejb21.PermissionManager;
import org.ow2.jonas.lib.ejb21.Protocols;
import org.ow2.jonas.lib.ejb21.TraceEjb;
import org.ow2.jonas.lib.loader.EjbJarClassLoader;
import org.ow2.jonas.lib.management.javaee.J2eeObjectName;
import org.ow2.jonas.lib.naming.ComponentContext;
import org.ow2.jonas.lib.service.AbsServiceImpl;
import org.ow2.jonas.lib.timer.TraceTimer;
import org.ow2.jonas.lib.util.JModule;
import org.ow2.jonas.lib.util.JonasObjectName;
import org.ow2.jonas.lib.util.Log;
import org.ow2.jonas.lib.util.ModuleNamingUtils;
import org.ow2.jonas.lib.work.CleanerException;
import org.ow2.jonas.lib.work.DeployerLog;
import org.ow2.jonas.lib.work.DeployerLogException;
import org.ow2.jonas.lib.work.FileManager;
import org.ow2.jonas.lib.work.WorkCleaner;
import org.ow2.jonas.lib.wsgen.WsGenChecker;
import org.ow2.jonas.naming.JComponentContextFactory;
import org.ow2.jonas.naming.JNamingManager;
import org.ow2.jonas.registry.RegistryService;
import org.ow2.jonas.resource.ResourceService;
import org.ow2.jonas.security.SecurityService;
import org.ow2.jonas.service.ServiceException;
import org.ow2.jonas.tm.TransactionManager;
import org.ow2.jonas.tm.TransactionService;
import org.ow2.jonas.workmanager.WorkManagerService;
import org.ow2.jonas.ws.WebServicesService;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.ee.deploy.api.deployer.IDeployerManager;
import org.ow2.util.file.FileUtils;
import org.ow2.util.file.FileUtilsException;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Logger;

/**
 * Implementation of the EJB Container Service for jonas. This class works only
 * with jonas_ejb classes.
 * @author Philippe Coq
 * @author Jeff Mesnil (Security)
 * @author Markus Karg (Novell port)
 * @author Christophe Ney (for making easier Enhydra integration)
 * @author Adriana Danes (complete management methods)
 * @author Florent Benoit & Ludovic Bert (Ear service, deployJars, undeployJars)
 * @author Benjamin Bonnet (max thread pool size)
 * @author Michel-Ange Anton (JSR77 MBean : EJBModule)
 * @author Adriana Danes (JSR77)
 */
public class JOnASEJBService extends AbsServiceImpl implements EJBService, JOnASEJBServiceMBean {

    /** server logger. */
    private static Logger logger = Log.getLogger(Log.JONAS_EJB_PREFIX);

    /** loader logger. */
    private static Logger loaderlog = Log.getLogger(Log.JONAS_LOADER_PREFIX);

    /**
     * Transaction manager reference (tm can be accessed directly from
     * everywhere in the package.).
     */
    private TransactionManager tm = null;

    /** JMX Service. */
    private JmxService jmxService = null;

    /** Security service reference. */
    private SecurityService securityService = null;

    /** web service reference. */
    private WebServicesService wsService = null;

    /**
     * Registry service used to discover active protocols (auto GenIC).
     */
    private RegistryService registryService = null;

    /**
     * Reference to the cleaner.
     */
    private static WorkCleaner workCleaner = null;

    /**
     * List of the ejb names to load when starting the EJB Container Service.
     */
    private List<String> ejbNames = new Vector<String>();

    /** We have a Container by ejb-jar file. */
    private Vector<Container> containers = new Vector<Container>();

    /** The WorkManager instance. */
    private WorkManager workManager;

    /**
     * The name of the JONAS_BASE directory.
     */
    protected static final String JONAS_BASE = JProp.getJonasBase();

    /**
     * The name of the directory containing the ejb-jar files.
     */
    protected static final String EJBJARS_DIR = JONAS_BASE + File.separator + "ejbjars";

    /**
     * List of autoloaded directories.
     */
    private ArrayList<String> autoloadDirectories = new ArrayList<String>();

    /** BeanManaged Management properties. */
    public static final String BMP = "Bean-Managed";

    /** ContainerManaged Management properties. */
    public static final String CMP = "Container-Managed";

    /** applications parent classloader. */
    private ClassLoader appsClassLoader = null;

    /**
     * The name of the working directory.
     */
    private static final String WORK_DIR = JProp.getWorkDir();

    /**
     * The name of the working ejbjars directory.
     */
    private static final String WORK_EJBJARS_DIR = WORK_DIR + File.separator + "ejbjars";

    /**
     *
     */
    private static String ejbjarsDir = null;

    /**
     * Reference on the DeployerLog which is the class that manage the accesses
     * to the log file (to remove the jar).
     */
    private DeployerLog jarDeployerLog = null;

    /**
     * Auto-GenIC has been enabled or not ?
     */
    private boolean autoGenIC = true;

    /**
     * List of arguments for the autoGenIC.
     */
    private List<String> autoGenICArgsList = new ArrayList<String>();

    /**
     * Singleton object.
     */
    private JComponentContextFactory componentContextFactory = null;

    /**
     * CCF Delegate.
     */
    private HandleDelegateCCFDelegate ccfd = null;

    /**
     * TransactionService.
     */
    private TransactionService transactionService;

    /**
     * WorkManager Service.
     */
    private WorkManagerService workManagerService;

    /**
     * NamingManager for EJB binding.
     */
    private JNamingManager naming;

    /**
     * Link to the EJB 2.1 Deployer.
     */
    private EJB21Deployer ejb21Deployer = null;

    /**
     * CMI service reference (can be null).
     */
    private CmiService cmiService = null;

    /**
     * HA service reference (can be null).
     */
    private HaService haService = null;

    /**
     * Resource service reference (can be null).
     */
    private ResourceService resService = null;

    /**
     * DeployerManager service.
     */
    private IDeployerManager deployerManager;


    /**
     * Default constructor.
     */
    public JOnASEJBService() {
        this.ejb21Deployer = new EJB21Deployer();
    }

    /**
     * @param autoGenIC Engage Auto GenIC ?
     */
    public void setAutoGenic(final boolean autoGenIC) {
        this.autoGenIC = autoGenIC;
        if (!autoGenIC) {
            logger.log(BasicLevel.INFO, "Auto GenIC has been disabled");
        }
    }

    /**
     * @param validate Use a validating parser ?
     */
    public void setParsingwithvalidation(final boolean validate) {
        EjbDeploymentDescManager.setParsingWithValidation(validate);
        if (!validate) {
            if (logger.isLoggable(BasicLevel.DEBUG)) {
                logger.log(BasicLevel.DEBUG, "XML parsing without validation");
            }
        }
    }

    /**
     * @param descriptors List of EjbJars to be deployed at startup.
     */
    public void setDescriptors(final String descriptors) {
        List<String> descriptorList = convertToList(descriptors);
        ejbNames = descriptorList;
    }

    /**
     * @param directories Autoloaded directories.
     */
    public void setAutoloaddir(final String directories) {
        List<String> directoryList = convertToList(directories);
        for(String directory : directoryList) {
            String dirName = directory.trim();
            addEjbjars(dirName);
            File file = new File(EJBJARS_DIR, dirName);
            if (file.exists()) {
                try {
                    autoloadDirectories.add(file.getCanonicalPath());
                } catch (IOException e) {
                    String err = "Error when trying to verify Ejbjar autoload directory : " + dirName;
                    logger.log(BasicLevel.ERROR, err, e);
                }
            }
        }
    }

    /**
     * @param args List of arguments to the Auto-GenIC process.
     */
    public void setAutoGenicArgs(final String args) {
        autoGenICArgsList = convertToList(args);
        if (!autoGenICArgsList.isEmpty()) {
            logger.log(BasicLevel.INFO, "Auto GenIC args are set to '" + autoGenICArgsList + "'.");
        }
    }

    /**
     * @see org.ow2.jonas.lib.service.AbsServiceImpl#checkRequirements()
     */
    @Override
    public void checkRequirements() throws ServiceException {
        if (jmxService == null) {
            throwRequirementException("Missing reference on " + JmxService.class);
        }
        if (transactionService == null) {
            throwRequirementException("Missing reference on " + TransactionService.class);
        }
        if (workManagerService == null) {
            throwRequirementException("Missing reference on " + WorkManagerService.class);
        }
    }

    /**
     * Start the Service Initialization of the service is already done.
     */
    public void doStart() throws ServiceException {

        // Get some objects from services
        if (transactionService != null) {
            tm = transactionService.getTransactionManager();
        } else {
            logger.log(BasicLevel.ERROR, "Transaction Service not injected");
            throw new ServiceException("Transaction Service not injected");
        }
        if (workManagerService != null) {
            workManager = workManagerService.getWorkManager();
        } else {
            logger.log(BasicLevel.ERROR, "WorkManager Service not injected");
            throw new ServiceException("WorkManager Service not injected");
        }

        //----------------------- init
        // Init the loggers: Timer & Ejb
        TraceTimer.configure(Log.getLoggerFactory());
        TraceEjb.configure(Log.getLoggerFactory());

        logger.log(BasicLevel.DEBUG, "");

        // Init the deployer
        ejb21Deployer.setEjb21Service(this);

        // Register the deployer
        if (deployerManager != null) {
            deployerManager.register(ejb21Deployer);
        } else {
            logger.log(BasicLevel.ERROR, "DeployerManager Service not injected");
            throw new ServiceException("DeployerManager Service not injected");
        }

        // get apps ClassLoader
        try {
            LoaderManager lm = LoaderManager.getInstance();
            appsClassLoader = lm.getAppsLoader();
        } catch (Throwable e) {
            logger.log(BasicLevel.ERROR, "Cannot get the Applications ClassLoader from EJB Container Service");
            throw new ServiceException("Cannot get the Applications ClassLoader from EJB Container Service", e);
        }

        // add HandleDelegate to CCF
        // TODO Remove this in full OSGi context
        if (!isOSGi()) {
            try {
                ccfd = new HandleDelegateCCFDelegate();
                if (componentContextFactory == null) {
                    logger.log(BasicLevel.ERROR, "componentContextFactory not set");
                    throw new ServiceException("Cannot get ComponentContextFactory");
                }
                componentContextFactory.addDelegate(ccfd);
            } catch (NamingException e) {
                logger.log(BasicLevel.WARN, "Cannot get ComponentContextFactory:", e);
            }
        }

        // add TimerHandleDelegate to CCF
        // TODO Remove this in full OSGi context
        if (!isOSGi()) {
            try {
                JTimerHandleDelegate tccfd = new JTimerHandleDelegate();
                tccfd.setEJBService(this);
                componentContextFactory.addDelegate(tccfd);
            } catch (NamingException e) {
                logger.log(BasicLevel.WARN, "Cannot get ComponentContextFactory:", e);
            }
        }

        // Load the Model MBeans
        if (jmxService != null) {
            // TODO, When ejb service will become a bundle, we could use
            // the meta-data located in META-INF/mbeans-descriptors.xml
            jmxService.loadDescriptors(getClass().getPackage().getName(),
                                getClass().getClassLoader());
        }

        ejbjarsDir = WORK_EJBJARS_DIR + File.separator + getJonasServerName();

        URL ejbjarsUrl = null;
        try {
            ejbjarsUrl = new File(ejbjarsDir).toURL();
        } catch (MalformedURLException mue) {
            throw new ServiceException("Error when trying to get the URL of the jonasroot/apps directory", mue);
        }

        File fLog = new File(ejbjarsUrl.getFile() + File.separator + getJonasServerName() + ".log");
        if (!fLog.exists()) {
            try {
                //create
                fLog.getParentFile().mkdirs();
                fLog.createNewFile();
            } catch (IOException e) {
                throw new ServiceException("cannot create the log file" + fLog, e);
            }
        }

        //get the logger
        try {
            jarDeployerLog = new DeployerLog(fLog);
        } catch (DeployerLogException e) {
            throw new ServiceException("Can not get an EarDeployerLog", e);
        }

        // create the jar deployment work file control task
        JarCleanTask jarCleanTask = new JarCleanTask(jarDeployerLog, this);

        //get the cleaner ref
        workCleaner = WorkCleaner.getInstance();

        // add the jar deployment work file control task
        try {
            workCleaner.registerTask(jarCleanTask);
        } catch (CleanerException ce) {
            throw new ServiceException("Cannot register the JAR clean task", ce);

        }

        // force the cleaning immediately
        workCleaner.executeTasks();

        // Creates all predefined containers
        // 1 container per ejbjar file.
        String fileName = null;
        Context contctx = null;
        for (int i = 0; i < ejbNames.size(); i++) {
            fileName = (String) ejbNames.get(i);
            try {
                contctx = new ComponentContext(fileName);
                contctx.rebind("filename", fileName);
            } catch (NamingException ne) {
                logger.log(BasicLevel.WARN, "Cannot create container for " + fileName
                        + " because of a NamingException : " + ne.toString());
            }

            try {
                createContainer(contctx);
            } catch (Throwable e) {
                logger.log(BasicLevel.WARN, "Cannot create container for " + fileName, e);
                // Close naming context
                try {
                    contctx.close();
                } catch (NamingException nne) {
                    if (logger.isLoggable(BasicLevel.DEBUG)) {
                        logger.log(BasicLevel.DEBUG, "Cannot close deploy context for " + fileName, nne);
                    }
                }
            }
        }

        // Register EJBService MBean
        registerEjbServiceMBean(this, getDomainName());

        logger.log(BasicLevel.INFO, "EJB 2.1 Service started");
    }

    /**
     * Stop the service: - Remove all JOnAS Containers. - Unbinds all the
     * EJBHome names from JNDI
     */
    public void doStop() {
        if (deployerManager != null) {
            // Unregister the deployer
            deployerManager.unregister(ejb21Deployer);
        }

        // remove HandleDelegate to CCF
        if (!isOSGi() && componentContextFactory != null) {
            try {
                componentContextFactory.removeDelegate(ccfd);
            } catch (NamingException e) {
                logger.log(BasicLevel.DEBUG, "Cannot remove the handle delegate", e);
            }
        }

        // Unload containers
        Enumeration lc = containers.elements();
        while (lc.hasMoreElements()) {
            Container cont = (Container) lc.nextElement();
            removeContainer(cont);
        }

        // Admin code
        // Unregister EJBService MBean
        unregisterEjbServiceMBean(getDomainName());

        // TODO Remove ModelMBean from the commons modeler registry

        logger.log(BasicLevel.INFO, "EJB 2.1 Service stopped");
    }

    // -------------------------------------------------------------------
    // EJBService Implementation
    // -------------------------------------------------------------------

    /**
     * Create a JOnAS Container for all the beans that are described in a .xml
     * file, or belong to .jar file.
     * @return The ObjectName of the MBean associated to the container (i.e. to
     *         the deployed module)
     * @throws Exception if an error occur during the creation of the container.
     */
    public String createContainer(Context ctx) throws Exception {
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "");
        }

        // Get the file name
        String fileName = (String) ctx.lookup("filename");
        File f = new File(fileName);

        boolean isEjbJar = fileName.toLowerCase().endsWith(".jar");

        // check if the file exists
        // In case of the name is a ejb-jar file name, check also in
        // the JONAS_BASE/ejbjars directory
        try {
            if (!f.exists() && isEjbJar) {
                String ejbjarFileName = EJBJARS_DIR + File.separator + fileName;
                f = new File(ejbjarFileName).getCanonicalFile();
                if (!f.exists()) {
                    logger.log(BasicLevel.ERROR, "createContainer: " + fileName + " not found");
                    throw new NamingException(fileName + " not found");
                }
            }
        } catch (IOException e) {
            String err = "Invalid ejbjar file name '" + fileName;
            logger.log(BasicLevel.ERROR, err);
            throw new Exception(err, e);
        }
        boolean isEjbJarFile = isEjbJar && f.isFile();
        boolean isEjbJarXml = fileName.toLowerCase().endsWith(".xml") && f.isFile();
        
        // Force fileName with slash to avoid conflict under Windows (slash and back-slash)
        try {
            fileName = f.toURL().getPath();
        } catch (MalformedURLException e) {
            logger.log(BasicLevel.ERROR, "Invalid ejb-jar file name '" + fileName + "'", e);
        }

        if (f.isFile()) {
            if (!isEjbJar && !isEjbJarXml) {
                throw new ServiceException("The ejbjar to deploy is not a jar file nor an xml file");
            }
        }

        // Check if container already exists
        if (getContainer(fileName) != null) {
            logger.log(BasicLevel.ERROR, "createContainer: " + fileName + " already exists");
            throw new Exception("Container already exists");
        }

        // In order to avoid lock problem after undeploy
        // the jar file is copied under the working directory before its loading
        // ! only if the ejbjars isn't included in an ear
        boolean isInEar = true;
        try {
            ctx.lookup("earClassLoader");
        } catch (NamingException ne) {
            isInEar = false;
        }

        // if need, call Genic. Only in single ejb-jar mode.
        // In ear case, this has been checked by ear service.
        if (!isInEar && isEjbJar) {
            checkGenIC(fileName, null);

            String resultFilename = applyWSGenIfNeeded(fileName);
            if (resultFilename.endsWith(".ear")) {
                // WsGen has produced a new Application
                // It has already been deployed by 'ear' service
                // TODO Should return the EJB Module ObjectName of the ear
                return J2eeObjectName.getEJBModule(this.getDomainName(),
                                                   this.getJonasServerName(),
                                                   ModuleNamingUtils.fromFileName(resultFilename),
                                                   ModuleNamingUtils.fromFileName(fileName)).getCanonicalName();
            }
        }

        String wkFileName = null;

        if (!isInEar && isEjbJarFile) {
            // Get the destination file
            wkFileName = ejbjarsDir + File.separator + FileManager.fileToTimeStampDir(f.toURL(), ".jar");

            if (logger.isLoggable(BasicLevel.DEBUG)) {
                logger.log(BasicLevel.DEBUG, "filename=" + fileName);
                logger.log(BasicLevel.DEBUG, "wkFilename=" + wkFileName);
            }

            // copy the file
            FileUtils.copyFile(fileName, wkFileName);

            // add an entry in the log file
            try {
                jarDeployerLog.addEntry(f, new File(wkFileName));
            } catch (DeployerLogException e) {
                String err = "Error while adding the " + fileName + " entry in the log file";
                logger.log(BasicLevel.ERROR, err + " : " + e.getMessage());
                throw new Exception(err, e);
            }

            if (logger.isLoggable(BasicLevel.DEBUG)) {
                logger.log(BasicLevel.DEBUG, "Create wk file :" + wkFileName);
            }

        } else {
            // in this case, no needs to copy the file
            wkFileName = fileName;
            if (logger.isLoggable(BasicLevel.DEBUG)) {
                logger.log(BasicLevel.DEBUG, "No needs wk file :" + wkFileName);
            }
        }

        // Classloader
        URL[] url = null;
        try {
            url = new URL[1];
            url[0] = (new File(wkFileName)).toURL();
        } catch (MalformedURLException e) {
            logger.log(BasicLevel.ERROR, "Invalid ejb-jar file name '" + wkFileName + "'", e);
        }

        // Get the classloaders
        URLClassLoader ejbClassLoader = null;
        URLClassLoader earClassLoader = null;
        try {
            earClassLoader = (URLClassLoader) ctx.lookup("earClassLoader");
            ejbClassLoader = (URLClassLoader) ctx.lookup("ejbClassLoader");
            if (loaderlog.isLoggable(BasicLevel.DEBUG)) {
                loaderlog.log(BasicLevel.DEBUG, "earClassLoader=" + earClassLoader);
            }
        } catch (NamingException ne) {
            // no ear classloader => ejb application case (not ear case).
            if (isEjbJar) {
                ejbClassLoader = new EjbJarClassLoader(url, appsClassLoader);
            } else {
                ejbClassLoader = (URLClassLoader) appsClassLoader;
            }
            if (loaderlog.isLoggable(BasicLevel.DEBUG)) {
                loaderlog.log(BasicLevel.DEBUG, "parent Loader=" + appsClassLoader);
            }
        }
        if (loaderlog.isLoggable(BasicLevel.DEBUG)) {
            loaderlog.log(BasicLevel.DEBUG, "ejbClassLoader=" + ejbClassLoader);
        }

        // Get the deployment descriptor from file
        DeploymentDesc dd = null;
        try {
            // ejb-jar file
            EjbDeploymentDescManager mgr = EjbDeploymentDescManager.getInstance();
            dd = mgr.getDeploymentDesc(url[0], ejbClassLoader, earClassLoader);
        } catch (DeploymentDescException e) {
            String err = "Cannot read the deployment descriptors '" + fileName + "'";
            logger.log(BasicLevel.ERROR, err);
            logger.log(BasicLevel.ERROR, "DeploymentDescException:" + e);
            throw new ServiceException(err, e);
        }

        // container name
        String cname = ModuleNamingUtils.fromFileName(fileName);
        if (cname == null) {
            //use displayName
            cname = "EJB container ";
            if (dd.getDisplayName() != null) {
                cname += dd.getDisplayName();
            }
        }

        // Create the Container Object
        JContainer cont = new JContainer(cname, fileName, wkFileName, ejbClassLoader, dd, cmiService, haService, wsService, jmxService.getJmxServer(), resService);
        cont.setContainerNaming(naming);
        if (componentContextFactory == null) {
            logger.log(BasicLevel.ERROR, "componentContextFactory has not been set");
        }
        cont.setComponentContextFactory(componentContextFactory);
        cont.setTransactionManager(tm);
        cont.setPrincipalFactory(new PrincipalFactoryImpl());

        // Set the name of the ear application containing this container
        // in the case of an ear application.
        URL earUrl = null;
        String earFileName;
        try {
            earUrl = (URL) ctx.lookup("earURL");
            //earFileName = (String) ctx.lookup("earFileName");
            earFileName = earUrl.getFile();
            cont.setEarFileName(earFileName);
        } catch (NamingException ne) {
            earFileName = null;
        }

        // unset the security if no security service
        if (securityService == null) {
            if (logger.isLoggable(BasicLevel.DEBUG)) {
                logger.log(BasicLevel.DEBUG, "EJB Container Service: working without Security Service");
            }
            cont.setSecurity(false);
        } else {
            PermissionManager permissionManager = null;
            try {
                permissionManager = new PermissionManager(dd, cont.getContextId());
                permissionManager.translateEjbDeploymentDescriptor();
                // if not in ear case, commit the policy configuration, else it
                // is done
                // by EAR service after linking all policy configuration objects
                if (!isInEar) {
                    permissionManager.commit();
                }
            } catch (Exception e) {
                logger.log(BasicLevel.ERROR, "Can't build permission manager object for the ejbjar '" + fileName + "'", e);
            }
            cont.setPermissionManager(permissionManager);
        }

        // Give access to the WorkManager
        cont.setWorkManager(workManager);

        // Process each bean from the DD
        BeanDesc[] beans = dd.getBeanDesc();
        ArrayList bflist = new ArrayList();
        // Keep factorys in order to do admin staff later
        Map factoryMap = new HashMap();
        // Keep bean descriptors in order to do admin staff later
        Map beanDescMap = new HashMap();
        for (int i = 0; i < beans.length; i++) {
            BeanDesc beanDesc = beans[i];
            String beanName = null;
            try {
                beanName = beanDesc.getEjbName();
                BeanFactory bf = cont.addBean(beanDesc);
                bflist.add(bf);
                factoryMap.put(beanName, bf);
                beanDescMap.put(beanName, beanDesc);
            } catch (Exception e) {
                e.printStackTrace();
                logger.log(BasicLevel.WARN,"Can't deploy beanName : "+beanName + "Exception: "+e);
            }
        }
        // Init now pools of instances, in case min-pool-size have been defined.
        // This must be done once all beans have been deployed.
        for (Iterator it = bflist.iterator(); it.hasNext();) {
            BeanFactory bf = (BeanFactory) it.next();
            bf.initInstancePool();
        }

        // Deployment was successful, add the bean as deployed
        containers.addElement(cont);

        // Restart Timers. Must be done when containers has been updated.
        for (Iterator it = bflist.iterator(); it.hasNext();) {
            BeanFactory bf = (BeanFactory) it.next();
            bf.restartTimers();
        }

        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "created container for " + fileName);
        }

        // Di admin staff
        String domainName = getDomainName();
        String serverName = getJonasServerName();
        String j2eeappName = ModuleNamingUtils.fromFileName(earFileName);
        String ejbModuleON = registerCompMBeans(domainName, serverName, j2eeappName, cname, cont, fileName, earUrl, dd, factoryMap, beanDescMap);

        // return Model MBean OBJECT_NAME
        return ejbModuleON;
    }

    /**
     * Apply WSGen if needed
     * @param filename
     * @return new filename if generated
     * @throws DeployerException if WSGen cannot be applied.
     */
    private String applyWSGenIfNeeded(String filename) throws DeployerException {

        // WS service present ?
        if (wsService == null) {
            logger.log(BasicLevel.DEBUG, "The WS service is not present, no need to call WSGen");
            return filename;
        }

        // Auto WsGen enabled ?
        if (!wsService.isAutoWsGenEngaged()) {
            logger.log(BasicLevel.DEBUG, "Automatic WsGen is not enabled, no need to call WSGen");
            return filename;
        }

        // Check version in manifest
        String jonasVersionWsGen = getAttributeInManifest(filename, "WsGen-JOnAS-Version");
        if (Version.getNumber().equals(jonasVersionWsGen)) {
            // no changes, just continue the normal deployment process
            logger.log(BasicLevel.DEBUG, "No change: no need to call WSGen");
            return filename;
        }

        // Call the WSGen tool by using the wrapper
        WsGenWrapper wsgen = new WsGenWrapper();
        String nfile = null;
        try {
            nfile = wsgen.callWsGenExecute(filename, false);
            if (wsgen.callWsGenIsInputModifed()) {
                // archive has been changed.
                if (nfile.endsWith(".ear")) {
                    // WsGen produced an Ear from the EjbJar
                    logger.log(BasicLevel.DEBUG, "WsGen produced an EAR '" + nfile + "'");

                    // Copy it from the work directory to apps directory
                    File source = new File(nfile);
                    File destination = new File(JProp.getJonasBase(), "apps" + File.separator + source.getName());
                    try {
                        FileUtils.copyFile(source, destination);
                    } catch (FileUtilsException e) {
                        // can't copy the file
                        throw new DeployerException("Cannot copy the file from '" + nfile + "' to '" + filename + "'", e);
                    }
                    logger.log(BasicLevel.DEBUG, "Deploying '" + nfile + "' (copied from '" + source.getPath() + "') ...");

                    // invoke method on the MBeanServer
                    try {
                        ObjectName earServiceON = JonasObjectName.earService(getDomainName());
                        // deploy the resulting, copied ear
                        Object[] param = {nfile, Boolean.FALSE};
                        String[] signature = {"java.lang.String", "java.lang.Boolean"};
                        MBeanServer mbeanServer = jmxService.getJmxServer();
                        mbeanServer.invoke(earServiceON, "deployEarMBean", param, signature);
                    } catch (InstanceNotFoundException e) {
                        // no ear service in this instance
                        throw new DeployerException("Ear Service not available. Cannot deploy : " + nfile, e);
                    } catch (MBeanException e) {
                        // deployEarMBean thrown an Exception
                        throw new DeployerException("Cannot deploy the produced Application : " + nfile, e.getTargetException());
                    } catch (ReflectionException e) {
                        // cannot invoke mbean operation
                        throw new DeployerException("Cannot invoke deploy operation on Ear Service MBean for application : " + nfile, e);
                    }
                } else {
                    // WsGen did not change the archive type
                    // but produced another (updated) jar
                    logger.log(BasicLevel.DEBUG, "Archive updated, continuing deployment : " + nfile);
                    // replace old file with the new one...
                    try {
                        System.gc();
                        // It seems that windows need some time to remove file locks.
                        Thread.sleep(Long.getLong("jonas.wsgen.sleep", 100).longValue());

                        // delete old file
                        if (!FileUtils.delete(filename)) {
                            throw new DeployerException("Cannot delete old application : " + filename);
                        }
                        FileUtils.copyFile(nfile, filename);
                        FileUtils.delete(nfile);
                    } catch (FileUtilsException e) {
                        // can't copy the file
                        throw new DeployerException("Cannot copy the ear from '" + nfile + "' to '" + filename + "'", e);
                    } catch (InterruptedException e) {
                        // Required by the Thread.sleep() :(
                        throw new DeployerException("Cannot sleep for the given time period", e);
                    }
                    return filename;
                }
            }
        } catch (Exception e) {
            throw new DeployerException("Cannot execute WSGen on '" + filename + "'", e);
        }
        return nfile;
    }

    /**
     * Get the Container by its file name (.xml or .jar)
     * @param fileName given file name on which
     * @return Container
     */
    public Container getContainer(String fileName) {
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, fileName);
        }

        // In case of the name is a ejb-jar file name, check also in
        // the JONAS_BASE/ejbjars directory and JONAS_BASE/ejbjars/autoload

        String ejbjarFileName = EJBJARS_DIR + File.separator + fileName;
        File f = null;
        String pathEjbJarDirFile;
        String pathFile = "";
        try {
            f = new File(ejbjarFileName).getCanonicalFile();
            // Force pathEjbJarDirFile with slash to avoid conflict under
            // Windows (slash and back-slash)
            pathEjbJarDirFile = f.toURL().getPath();
        } catch (Exception e) {
            // If filename was an absolute path, it can throw an exception
            // The pathFile will be empty
            pathEjbJarDirFile = "";
        }
        try {
            f = new File(fileName);
            // Force pathFile with slash to avoid conflict under Windows (slash
            // and back-slash)
            pathFile = f.toURL().getPath();
        } catch (Exception e) {
            String err = "Error while trying to get canonical file '" + fileName + "'";
            logger.log(BasicLevel.ERROR, err);
            return null;
        }

        // for each container loaded
        Enumeration lc = containers.elements();
        while (lc.hasMoreElements()) {
            Container cont = (JContainer) lc.nextElement();
            String contName = cont.getExternalFileName();

            // first, search abs path
            if (contName.equals(pathFile)) {
                return cont;
            } else if (contName.equals(pathEjbJarDirFile)) {
                // next, search in ejbjar path
                return cont;
            } else {
                // else search in autoloadir
                List lAlDirs = getAutoloadDirectories();
                for (int i = 0; i < lAlDirs.size(); i++) {
                    String alDir = (String) lAlDirs.get(i);

                    String alFileName = alDir + File.separator + fileName;
                    try {
                        f = new File(alFileName).getCanonicalFile();
                        // Force alFileName with slash to avoid conflict under
                        // Windows (slash and back-slash)
                        alFileName = f.toURL().getPath();
                    } catch (Exception e) {
                        // If filename was an absolute path, it can throw an
                        // exception
                        alFileName = alDir + File.separator + fileName;
                    }
                    if (logger.isLoggable(BasicLevel.DEBUG)) {
                        logger.log(BasicLevel.DEBUG, "alFileName= " + alFileName);
                    }

                    if (contName.equals(alFileName)) {
                        return cont;
                    }
                }
            }
        }

        // not found
        return null;
    }

    /**
     * Remove a JOnAS container
     */
    public void removeContainer(Container cont) {
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, cont.getName());
        }
        // Admin code
        String domainName = getDomainName();
        String serverName = getJonasServerName();
        String earFileName = cont.getEarFileName();
        String earName = null;
        if (earFileName != null) {
            earName = ModuleNamingUtils.fromFileName(earFileName);
        }
        String moduleName = cont.getName();
        unregisterCompMBeans(domainName, serverName, earName, moduleName);

        // Remove container
        cont.remove();
        containers.removeElement(cont);

        // Run the garbage collector
        Runtime.getRuntime().gc();
    }

    /**
     * List JOnAS container created by EJB Service
     */
    public Container[] listContainers() {
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "size= " + containers.size());
        }

        Container[] ret = new Container[containers.size()];
        containers.copyInto(ret);
        return ret;
    }

    /**
     * Synchronized all entity bean containers
     * @param passivate passivate instances after synchronization.
     */
    public void syncAllEntities(boolean passivate) {
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "");
        }

        Enumeration lc = containers.elements();
        while (lc.hasMoreElements()) {
            JContainer cont = (JContainer) lc.nextElement();
            cont.syncAll(true, passivate);
        }
    }

    /** ******* MBEan methods ******** */

    /**
     * MBean method
     * @return Integer Total Number of Container currently in JOnAS
     */
    public Integer getCurrentNumberOfContainer() {
        return new Integer(containers.size());
    }

    /**
     * MBean method
     * @return Integer Total Number of Bean Type currently in JOnAS
     */
    public Integer getTotalCurrentNumberOfBeanType() {
        int count = 0;
        Enumeration lc = containers.elements();
        while (lc.hasMoreElements()) {
            JContainer cont = (JContainer) lc.nextElement();
            count += cont.getBeanNb();
        }
        return new Integer(count);
    }

    /**
     * MBean method
     * @return Integer Total Number of Bmp Type currently in JOnAS
     */
    public Integer getTotalCurrentNumberOfBMPType() {
        int count = 0;
        Enumeration lc = containers.elements();
        while (lc.hasMoreElements()) {
            JContainer cont = (JContainer) lc.nextElement();
            count += cont.getEntityBMPNb();
        }
        return new Integer(count);
    }

    /**
     * MBean method
     * @return Integer Total Number of Cmp Type currently in JOnAS
     */
    public Integer getTotalCurrentNumberOfCMPType() {
        int count = 0;
        Enumeration lc = containers.elements();
        while (lc.hasMoreElements()) {
            JContainer cont = (JContainer) lc.nextElement();
            count += cont.getEntityCMPNb();
        }
        return new Integer(count);
    }

    /**
     * MBean method
     * @return Integer Total Number of Sbf Type currently in JOnAS
     */
    public Integer getTotalCurrentNumberOfSBFType() {
        int count = 0;
        Enumeration lc = containers.elements();
        while (lc.hasMoreElements()) {
            JContainer cont = (JContainer) lc.nextElement();
            count += cont.getStatefulSessionNb();
        }
        return new Integer(count);
    }

    /**
     * MBean method
     * @return Integer Total Number of Sbl Type currently in JOnAS
     */
    public Integer getTotalCurrentNumberOfSBLType() {
        int count = 0;
        Enumeration lc = containers.elements();
        while (lc.hasMoreElements()) {
            JContainer cont = (JContainer) lc.nextElement();
            count += cont.getStatelessSessionNb();
        }
        return new Integer(count);
    }

    /**
     * MBean method
     * @return Integer Total Number of Mdb Type currently in JOnAS
     */
    public Integer getTotalCurrentNumberOfMDBType() {
        int count = 0;
        Enumeration lc = containers.elements();
        while (lc.hasMoreElements()) {
            JContainer cont = (JContainer) lc.nextElement();
            count += cont.getMessageDrivenNb();
        }
        return new Integer(count);
    }

    /**
     * Remove EJB container corresponding to a .JAR file.
     * This method is called by the J2EESererver MBean's undeployJar(fileName)
     * and by the Adm.removeBeans(fileName).
     * @param fileName name of the file containing the JAR module
     */
    public void removeContainer(String fileName) throws Exception {
        Container cont = null;
        try {
            cont = getContainer(fileName);
        } catch (Exception e) {
            String err = "Error while trying to find file '" + fileName + "'";
            logger.log(BasicLevel.ERROR, err);
            throw new Exception(err, e);
        }

        if (cont != null) {
            removeContainer(cont, false);
        } else {
            String err = "Cannot remove the non-existant container '" + fileName + "'";
            logger.log(BasicLevel.ERROR, err);
            throw new Exception(err);
        }
    }

    /**
     * Create an ejb container for the EJBs contained in a .jar (or .xml) file.
     * @param file Name of the file to be deployed
     * @return The ObjectName of the MBean associated to the container (to the
     *         deployed module)
     */
    public String createContainer(String file) throws Exception {
        String ejbModuleObjectName = null;
        try {
            Context contctx = new ComponentContext(file);
            contctx.rebind("filename", file);
            ejbModuleObjectName = createContainer(contctx);
        } catch (Exception e) {
            throw new Exception("Cannot create Container", e);
        }
        return ejbModuleObjectName;
    }

    /**
     * Test if the specified file is already deployed (if a container is created
     * for this jar).
     * @param fileName the name of the jar file
     * @return true if the jar was deployed, false otherwise
     */
    public Boolean isJarDeployed(String fileName) {
        return new Boolean(isJarLoaded(fileName));
    }

    /**
     * Test if the specified jar identified with its work name is already
     * deployed (if a container is created for this jar).
     * @param workFileName the internal name of the jar file (working copy)
     * @return true if the jar was deployed, false otherwise
     */
    public boolean isJarDeployedByWorkName(String workFileName) {
        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, workFileName);
        }

        // for each container loaded
        Enumeration lc = containers.elements();
        while (lc.hasMoreElements()) {
            Container cont = (JContainer) lc.nextElement();

            // get name of the container (basename)
            // cont.getFileName return the internal name (working copy)
            String contName = new File(cont.getFileName()).getName();
            if (contName.equals(workFileName)) {
                return true;
            }
        }

        // not found
        return false;
    }

    // Kind of resources:
    /** DataSource Resource Type */
    static final int DATASOURCE = 1;

    /** JMS Destination Resource Type */
    static final int JMS_DESTINATION = 2;

    /** JMS Factory Resource Type */
    static final int JMS_FACTORY = 3;

    /** Mail Factory Resource Type */
    static final int MAIL_FACTORY = 4;

    /**
     * This method is used by the getXXXDependence MBean methods. It gathers the
     * ObjectNames of the EJBs using a given resource.
     * @param name the resource name
     * @type the resource type
     * @return the ObjectNames of all the ejb using this resource
     */
    Set getDependence(String name, int type) {
        Set resultObjectName = new HashSet(); // to be returned by this method
        Set resultProperties = new HashSet(); // constructed by the JContainer
        Enumeration lc = containers.elements();
        try {
            Context ctx = new InitialContext();
            ctx.lookup(name);
            ctx.close();
            // this resource is rebind
            while (lc.hasMoreElements()) {
                JContainer cont = (JContainer) lc.nextElement();
                Set depProps = null; // Dependences described as a set of
                // Properties objects
                switch (type) {

                case DATASOURCE:
                    depProps = cont.getDataSourceDependence(name);
                    break;

                case JMS_DESTINATION:
                    depProps = cont.getJmsDestinationDependence(name);
                    break;

                case JMS_FACTORY:
                    depProps = cont.getJmsConnectionFactoryDependence(name);
                    break;

                case MAIL_FACTORY:
                    depProps = cont.getMailFactoryDependence(name);
                    break;

                }
                resultProperties.addAll(depProps);
                // change string to ObjectName
                try {
                    resultObjectName = convertToObjectNames(resultProperties);
                } catch (Exception e) {
                    logger.log(BasicLevel.ERROR, "EjbServiceImpl: Object Name Error", e);
                }
            }
        } catch (NamingException ne) {
            // unexisting ds
            resultObjectName = new HashSet();
        }
        return resultObjectName;
    }

    /**
     * Converts a Set of properties to ObjectNames
     * @param resultProperties properties to be converted
     * @return Returns a Set of ObjectNames
     */
    private Set convertToObjectNames(Set resultProperties) {
        Set resultObjectName = new HashSet();
        Iterator it = resultProperties.iterator();
        String domainName = getDomainName();
        String serverName = getJonasServerName();
        Properties item = null; // a item in resultProperties
        String ejbType = null;
        String ejbName = null;
        String earFileName = null;
        String moduleName = null;
        String j2eeAppName = null;
        while (it.hasNext()) {
            item = (Properties) it.next();
            ObjectName ejbObjectName = null;
            // See JContainer.beansDependence() method for info about the
            // Properties object containt
            ejbType = item.getProperty("type");
            ejbName = item.getProperty("name");
            earFileName = item.getProperty("earFileName");
            //moduleName = buildEJBModuleName(fileName);
            moduleName = item.getProperty("cname");
            if (earFileName != null) {
                j2eeAppName = ModuleNamingUtils.fromFileName(earFileName);
            }

            if (ejbType.equals("ejbbmp") || ejbType.equals("ejbcmp")) {
                ejbObjectName = J2eeObjectName.getEntityBean(domainName, moduleName, serverName, j2eeAppName, ejbName);
            } else if (ejbType.equals("ejbsbf")) {
                ejbObjectName = J2eeObjectName.getStatefulSessionBean(domainName, moduleName, serverName, j2eeAppName,
                        ejbName);
            } else if (ejbType.equals("ejbsbl")) {
                ejbObjectName = J2eeObjectName.getStatelessSessionBean(domainName, moduleName, serverName, j2eeAppName,
                        ejbName);
            } else if (ejbType.equals("ejbmdb")) {
                ejbObjectName = J2eeObjectName.getMessageDrivenBean(domainName, moduleName, serverName, j2eeAppName,
                        ejbName);
            }
            if (ejbObjectName != null) {
                resultObjectName.add(ejbObjectName);
            }
        }
        return resultObjectName;
    }

    /**
     * MBean method
     * @return the ObjectName of all the ejbs using this datasource.
     */
    public Set getDataSourceDependence(String dsName) {
        return getDependence(dsName, DATASOURCE);
    }

    /**
     * Manangement method
     * @return the ObjectName of all the ejb using this destination.
     */
    public Set getJmsDestinationDependence(String destName) {
        return getDependence(destName, JMS_DESTINATION);
    }

    /**
     * Management method
     * @return the ObjectName of all the ejb using this Connection Factory.
     */
    public Set getJmsConnectionFactoryDependence(String cfName) {
        return getDependence(cfName, JMS_FACTORY);
    }

    /**
     * Management method
     * @return the ObjectName of all the ejb using a given Mail Factory.
     */
    public Set getMailFactoryDependence(String mfName) {
        return getDependence(mfName, MAIL_FACTORY);
    }

    /**
     * Return the list of installed EJB containers. The JAR files or the
     * directories with expanded EJB container are searched in
     * JONAS_BASE/ejbjars and all ejbjar directories 'autoload'.
     * @return The list of JAR files or the directories with expanded EJB
     *         container found
     * @throws Exception if the list can't be retrieved
     */
    public List getInstalledJars() throws Exception {
        // get JAR files found in JONAS_BASE/ejbjars
        ArrayList al = JModule.getInstalledContainersInDir(EJBJARS_DIR, JModule.EJBJAR_EXTENSION,
                JModule.EJBJAR_CHILD_DIR, JModule.EJBJAR_CONFIRM_FILE);
        // get JAR files found in all autoload directories
        for (int i = 0; i < autoloadDirectories.size(); i++) {
            al.addAll(JModule.getInstalledContainersInDir(autoloadDirectories.get(i).toString(),
                    JModule.EJBJAR_EXTENSION, JModule.EJBJAR_CHILD_DIR, JModule.EJBJAR_CONFIRM_FILE));
        }
        return al;
    }

    /**
     * Deploy the given ejb-jars of an ear file with the specified parent
     * classloader (ear classloader). (This method is only used for the ear
     * applications, not for the ejb-jar applications).
     * @param ctx the context containing the configuration to deploy the
     *        ejbjars. <BR>This context contains the following parameters :
     *        <BR>- earRootUrl the root of the ear application. <BR>- earURL
     *        filename of the EAR. <BR>- earClassLoader the ear classLoader of
     *        the ear application. <BR>- ejbClassLoader the ejb classLoader for
     *        the ejbjars. <BR>- jarURLs the list of the urls of the ejb-jars
     *        to deploy. <BR>- roleNames the role names of the security-role.
     *        <BR>
     * @throws ServiceException if an error occurs during the deployment.
     */
    public void deployJars(Context ctx) throws ServiceException {

        // Get the 6 parameters from the context
        // - earRootUrl the root of the ear application.
        // - earClassLoader the ear classLoader of the ear application.
        // - ejbClassLoader the ejb classLoader for the ejbjars.
        // - jarURLs the list of the urls of the ejb-jars to deploy.
        // - roleNames the role names of the security-role
        URL earRootUrl = null;
        URL earUrl = null;
        ClassLoader earClassLoader = null;
        ClassLoader ejbClassLoader = null;
        URL[] jarURLs = null;
        String[] roleNames = null;
        try {
            earRootUrl = (URL) ctx.lookup("earRootUrl");
            earUrl = (URL) ctx.lookup("earUrl");
            earClassLoader = (ClassLoader) ctx.lookup("earClassLoader");
            ejbClassLoader = (ClassLoader) ctx.lookup("ejbClassLoader");
            jarURLs = (URL[]) ctx.lookup("jarURLs");
            roleNames = (String[]) ctx.lookup("roleNames");
        } catch (NamingException e) {
            String err = "Error while getting parameter from context param :" + e.getMessage();
            logger.log(BasicLevel.ERROR, err);
            throw new ServiceException(err, e);
        }

        // Deploy all the ejb-jars of the ear application.
        for (int i = 0; i < jarURLs.length; i++) {

            // Get the name of an ejb-jar to deploy.
            String fileName = jarURLs[i].getFile();
            if (logger.isLoggable(BasicLevel.DEBUG)) {
                logger.log(BasicLevel.DEBUG, "Deploy '" + fileName + "' for the ear service.");
            }

            // The context to give for the creation of the container
            // associated to the ejb-jar.
            Context contctx = null;
            try {
                contctx = new ComponentContext(fileName);
                contctx.rebind("filename", fileName);
                contctx.rebind("earClassLoader", earClassLoader);
                contctx.rebind("ejbClassLoader", ejbClassLoader);
                contctx.rebind("earRoot", earRootUrl.getFile());
                //contctx.rebind("earFileName", earUrl.getFile());
                contctx.rebind("earURL", earUrl);
                contctx.rebind("roleNames", roleNames);
                createContainer(contctx);
            } catch (Exception e) {
                // An ejb-jar is corrupted so undeploy all the deployed jar
                // of the ear application.
                logger.log(BasicLevel.ERROR, "Error when deploying '" + fileName + "'", e);
                logger.log(BasicLevel.ERROR, "Undeploy ejb-jar of the ear application");

                for (int j = 0; j <= i; j++) {
                    Container cont = getContainer(jarURLs[j].getFile());
                    if (cont != null) {
                        removeContainer(cont, true);
                    } else {
                        logger.log(BasicLevel.ERROR, "Cannot remove the non-existant container '" + fileName + "'");
                    }
                }
                throw new ServiceException("Error during the deployment", e);
            }
        }
        //return ejbClassLoader;
    }

    /**
     * Undeploy the given ejb-jars of an ear file. (This method is only used for
     * the ear applications, not for the ejb-jar applications).
     * @param urls the list of the urls of the ejb-jars to undeploy.
     */
    public void unDeployJars(URL[] urls) {
        for (int i = 0; i < urls.length; i++) {
            String fileName = urls[i].getFile();
            Container cont = getContainer(urls[i].getFile());
            if (cont != null) {
                removeContainer(cont, true);
            } else {
                logger.log(BasicLevel.ERROR, "Cannot remove the non-existant container '" + fileName + "'");
            }
        }
    }

    /**
     * Remove the specified container.
     * @param cont the container to remove.
     * @param isEarCase true if only if the removeContainer method is called in
     *        the ear case, false otherwise.
     */
    public void removeContainer(Container cont, boolean isEarCase) {
        if (isEarCase == (cont.getEarFileName() != null)) {
            removeContainer(cont);
        } else {
            String err = "Cannot remove container '" + cont.getName()
                    + "' it is in an ear application. You must undeploy the ear associated.";
            logger.log(BasicLevel.ERROR, err);
        }
    }

    /**
     * Make a cleanup of the cache of deployment descriptor. This method must be
     * invoked after the ear deployment by the EAR service. the deployment of an
     * ear by .
     * @param earClassLoader the ClassLoader of the ear application to remove
     *        from the cache.
     */
    public void removeCache(ClassLoader earClassLoader) {
        EjbDeploymentDescManager mgr = EjbDeploymentDescManager.getInstance();
        mgr.removeCache(earClassLoader);
    }

    /**
     * Add recursively the ejbjars of the specified directory. If the dir has a
     * relative path, this path is relative from where the Application Server is
     * launched. If the dir is not found it will be searched in
     * $JONAS_BASE/ejbjars/ directory.
     * @param dirPath the path to the directory containing the ejbjars to load.
     */
    private void addEjbjars(String dirPath) {
        boolean found = false;

        // Look the directory relative to the $JONAS_BASE/ejbjars directory
        File dir = new File(EJBJARS_DIR + File.separator + dirPath);
        found = dir.isDirectory();

        if (found) {
            addEjbjarsFrom(dir);
        } else {
            String err = "Warning: Cannot load dir: '" + dirPath + "' ";
            err += "is not a directory or directory doesn't exist";
            logger.log(BasicLevel.WARN, err);
        }
    }

    /**
     * Add the ejbjars of the specified directory.
     * @param dir the directory from which the ejbjars are loaded.
     * @throws ServiceException if the argument is not a directory
     */
    private void addEjbjarsFrom(File dir) throws ServiceException {

        try {
            if (dir.isDirectory()) {
                File[] files = dir.listFiles();
                for (int i = 0; i < files.length; i++) {
                    if (files[i].getPath().toLowerCase().endsWith(".jar")) {
                        ejbNames.add(files[i].getCanonicalPath());
                    } else if (files[i].isDirectory()) {
                        addEjbjarsFrom(files[i]);
                    }
                }
            } else {
                String err = "Cannot load dir: '" + dir.getPath();
                err += "' is not a directory";
                logger.log(BasicLevel.ERROR, err);
                throw new ServiceException(err);
            }
        } catch (IOException e) {
            String err = "Invalid file name '" + dir.getPath();
            logger.log(BasicLevel.ERROR, err);
            throw new ServiceException(err, e);
        }
    }

    /**
     * Test if the specified filename is already deployed or not
     * @param fileName the name of the jar file.
     * @return true if the jar is deployed, else false.
     */
    public boolean isJarLoaded(String fileName) {
        return (getContainer(fileName) != null);
    }

    /**
     * Return the list of all loaded EJB container.
     * @return The list of deployed EJB container
     */
    public List getDeployedJars() {
        ArrayList al = new ArrayList();
        Container oContainer;
        for (Enumeration jars = containers.elements(); jars.hasMoreElements();) {
            oContainer = (Container) jars.nextElement();
            try {
                al.add((new File(oContainer.getExternalFileName())).toURL().getPath());
            } catch (Exception e) {
                // none
            }
        }
        return al;
    }

    /**
     * Return the list of installed EJB container ready to deploy.
     * @return The list of deployable EJB container
     * @throws Exception when cannot get installed jars list
     */
    public List getDeployableJars() throws Exception {
        List al = getInstalledJars();
        al.removeAll(getDeployedJars());
        return al;
    }

    /**
     * Return the list of "autoload" directories for EJB containers.
     * @return The list of all "autoload" directories
     */
    public List getAutoloadDirectories() {
        ArrayList al = new ArrayList();
        for (int i = 0; i < autoloadDirectories.size(); i++) {
            try {
                al.add((new File(autoloadDirectories.get(i).toString())).toURL().getPath());
            } catch (Exception e) {
                // none
            }
        }
        return al;
    }

    /**
     * Return the Ejbjars directory.
     * @return The Ejbjars directory
     */
    public String getJarsDirectory() {
        String sRet;
        try {
            sRet = (new File(EJBJARS_DIR)).toURL().getPath();
        } catch (Exception e) {
            sRet = null;
        }
        return sRet;
    }

    /**
     * Check if current carol protocol choose is include into those choose during generation
     * @param listProtocolGenerated (string with comma separator)
     * @param listCurrentProtocol (string with comma separator)
     * @return true if we have find at least one current protocol not in those generated
     */
    private static boolean checkCurrentProtocolIncludeIntoGenerated(String listProtocolGenerated, String listCurrentProtocol) {
        Protocols generated = new Protocols(listProtocolGenerated);
        Protocols current = new Protocols(listCurrentProtocol);
        return generated.isSupported(current);
    }

    /**
     * get the attibut value in manifest file
     * @param jarName : the jar file containg manifest
     * @param attributName : the attribut name
     * @return the value corresponding to the attributName if it's exist null otherwise
     */
    private static String getAttributeInManifest(String jarName, String attributName) {
        JarFile jar = null;
        try {
            File jarFile = new File(jarName);
            jar = new JarFile(jarFile);
            Manifest man = jar.getManifest();
            Attributes atts = man.getMainAttributes();
            return atts.getValue(attributName);
        } catch (Exception e) {
            logger.log(BasicLevel.ERROR, "Cannot read Manifest: " + e);
            return null;
        } finally {
            try {
                if (jar != null) {
                    jar.close();
                }
            } catch (IOException e) {
                logger.log(BasicLevel.ERROR, "Cannot close the Jar File: " + e);
            }
        }
    }

    /**
     * Check that GenIC have been applied on the given ejb-jar file
     * If it was not done, it run GenIC against the file.
     * @param fileName given EJB-JAR file.
     * @param urls Array of URLs used as CLASSPATH during EJB compilation
     */
    public void checkGenIC(String fileName, URL[] urls) {
        // Detect if auto genIC is disabled or not in jonas.properties file.
        if (!autoGenIC) {
            return;
        }

        // get Carol configuration to intialize Genic Arguments
        List<String> protocols = registryService.getActiveProtocolNames();


        // Extra args are stored in autoGenICArgsList

        String[] genicArgsString = null;

        int argIndex = 0;
        // append classpath
        if (urls != null && urls.length > 0) {
            // we have a CP to use
            genicArgsString = new String[4 + autoGenICArgsList.size()];

            StringBuffer classpath = new StringBuffer();
            for (int i = 0; i < urls.length; i++) {
                classpath.append(urls[i].getPath());
                if (i != (urls.length - 1)) {
                    classpath.append(File.pathSeparator);
                }
            }

            genicArgsString[argIndex++] = "-classpath";
            genicArgsString[argIndex++] = classpath.toString();
        } else {
            // no CP
            genicArgsString = new String[2 + autoGenICArgsList.size()];
        }

        genicArgsString[argIndex++] = "-protocols";
        String listProtocols = "";
        for (Iterator<String> i = protocols.iterator(); i.hasNext(); ) {
            String protocol = i.next();
            listProtocols = listProtocols + protocol;
            if (i.hasNext()) {
                listProtocols = listProtocols + ",";
            }
        }
        genicArgsString[argIndex++] = listProtocols;

        // Add the extra args
        Iterator itExtraArg = autoGenICArgsList.iterator();
        while (itExtraArg.hasNext()) {
            genicArgsString[argIndex++] = (String) itExtraArg.next();
        }


        String jonasVersionGenic = getAttributeInManifest(fileName, "Genic-Jonas-Version");
        String listProtocolGenerated = getAttributeInManifest(fileName, "Genic-Jonas-protocols");

        boolean isCurrentProtocolIncludeIntoGenerated = checkCurrentProtocolIncludeIntoGenerated(listProtocolGenerated, listProtocols);

        if ((jonasVersionGenic == null) || !jonasVersionGenic.equals(Version.getNumber())
                || !isCurrentProtocolIncludeIntoGenerated) {
            // Print a little explaination before calling Genic
            if (jonasVersionGenic == null) {
                logger.log(BasicLevel.INFO, "JOnAS version was not found in the '" + fileName + "' manifest file. Auto-generating container classes...");
            } else if (!jonasVersionGenic.equals(Version.getNumber())) {
                logger.log(BasicLevel.INFO, "JOnAS version found in the '" + fileName + "' manifest file :" + jonasVersionGenic + " is different of the current JOnAS version : " + Version.getNumber() + ". Auto-generating container classes...");
            } else if (!isCurrentProtocolIncludeIntoGenerated) {
                logger.log(BasicLevel.INFO, "Current Carol protocol is not included in the protocols found in the '" + fileName + "' manifest file. Auto-generating container classes...");
            }
            try {
                callGenic(fileName, genicArgsString);
            } catch (ServiceException e) {
                logger.log(BasicLevel.ERROR, "Cannot apply GenIC on the file '" + fileName + "' with the args '" + Arrays.asList(genicArgsString) + "'.", e);
            }
        }
    }


    /**
     * {@inheritDoc}
     */
    public String getContainerContextID(String containerFileName) {
        return getContainer(containerFileName).getContextId();
    }


    // =====================================================
    // TimerHandle delegation methods
    // =====================================================

    /**
     * {@inheritDoc}
     */
    public Timer getTimer(JTimerHandleInfo info) {

        // get the Bean TimerService
        JTimerService timerservice = getJTimerService(info);

        // Get the Timer from the list.
        Timer ret = timerservice.getTimerByTime(info.getDuration(), info.getPeriod(), info.getInfo());
        if (ret == null) {
            // The Timer should have been canceled.
            throw new NoSuchObjectLocalException("The Timer should have been canceled");
        }
        return ret;
    }

    /**
     * {@inheritDoc}
     */
    public Timer restartTimer(JTimerHandleInfo info) {

        // get the Bean TimerService
        JTimerService timerservice = getJTimerService(info);

        // Recreate a Timer with recomputed initial duration.
        long newDuration = info.getDuration() + info.getStartTime() - System.currentTimeMillis();
        if (newDuration < 100) {
            newDuration = 100;
        }
        // assign new value.
        info.setDuration(newDuration);
        Timer timer = timerservice.createTimer(info.getDuration(),
                                               info.getPeriod(),
                                               info.getInfo());
        if (timer.getTimeRemaining() > 0) {
            TraceTimer.logger.log(BasicLevel.DEBUG, "timer restarted");
        } else {
            TraceTimer.logger.log(BasicLevel.DEBUG, "timer terminated");
            timer.cancel();
        }
        return timer;
    }

    /**
     * @param info TimerHandle info for TimerService retrieving.
     * @return Returns the associated JTimerService instance.
     */
    private JTimerService getJTimerService(JTimerHandleInfo info) {

        // First get the Container
        JContainer cont = (JContainer) getContainer(info.getContainerId());
        if (cont == null) {
            TraceTimer.logger.log(BasicLevel.ERROR, "Cannot get container =" + info.getContainerId());
            throw new IllegalStateException("Cannot get container");
        }

        // The the BeanFactory
        JFactory bf = (JFactory) cont.getBeanFactory(info.getBeanId());

        // Finally get the associated TimerService
        JTimerService timerservice = null;
        if (bf instanceof JEntityFactory) {

            // entity bean
            Serializable pk = info.getPk();
            JEntityFactory ef = (JEntityFactory) bf;
            Serializable pks = ef.decodePK(pk);
            if (TraceTimer.isDebug()) {
                TraceTimer.logger.log(BasicLevel.DEBUG, "encoded PK=" + pk);
                TraceTimer.logger.log(BasicLevel.DEBUG, "decoded PK=" + pks);
            }
            JEntitySwitch es = ef.getEJB(pks);
            if (es == null) {
                TraceTimer.logger.log(BasicLevel.DEBUG, "No entity for this pk");
                throw new NoSuchObjectLocalException("No entity for this pk");
            }
            timerservice = (JTimerService) es.getEntityTimerService();
        } else {

            // stateless session or message driven bean
            timerservice = (JTimerService) bf.getTimerService();
        }
        if (timerservice == null) {
            TraceTimer.logger.log(BasicLevel.DEBUG, "Cannot retrieve TimerService");
            throw new IllegalStateException("Cannot retrieve TimerService");
        }
        return timerservice;
    }

    // Admin code (JMX based)
    // ---------------------
    /**
     * Register EJB Container Service MBean
     * @param service ejb container service to manage
     * @param domainName domain name
     */
    private void registerEjbServiceMBean(Object service, String domainName) {
        ObjectName on = JonasObjectName.ejbService(domainName);
        jmxService.registerMBean(service, on);
    }
    /**
     * Unregister EJB Container Service MBean
     * @param domainName domain name
     */
    private void unregisterEjbServiceMBean(String domainName) {
        ObjectName on = JonasObjectName.ejbService(domainName);
        jmxService.unregisterMBean(on);
    }

    private EJBModule registerEJBModuleMBean(String domainName, String serverName, String j2eeappName, String moduleName
                , JContainer container, String fileName, URL earUrl) {

        ObjectName ejbModuleOn = J2eeObjectName.getEJBModule(domainName, serverName, j2eeappName, moduleName);
        EJBModule ejbModuleMBean = new EJBModule(ejbModuleOn, container, fileName, moduleName, j2eeappName, earUrl);
        try {
            jmxService.registerModelMBean(ejbModuleMBean, ejbModuleOn);
        } catch (Exception e) {
            logger.log(BasicLevel.WARN, "Could not register EJBModule MBean", e);
        }
        return ejbModuleMBean;
    }

    private String registerEntityBeanMBean(String domainName, String serverName, String j2eeappName, String moduleName, String beanName
            , JEntityFactory jef, String persist) {
        String entityBeanON = J2eeObjectName.getEntityBeanName(domainName, moduleName, serverName,
                j2eeappName, beanName);
        EntityBean entityBeanMBean = new EntityBean(entityBeanON, jef, persist, jmxService);
        try {
            jmxService.registerModelMBean(entityBeanMBean, entityBeanON);
        } catch (Exception e) {
            logger.log(BasicLevel.WARN, "Could not register EntityBean MBean", e);
            return null;
        }
        return entityBeanON;
    }

    private String registerStatefulSessionBeanMBean(String domainName, String serverName, String j2eeappName, String moduleName
            , String beanName, JSessionFactory jsf) {
        String ssBeanON = J2eeObjectName.getStatefulSessionBeanName(domainName, moduleName, serverName,
                j2eeappName, beanName);
        StatefulSessionBean ssBeanMBean = new StatefulSessionBean(ssBeanON, jsf, jmxService);
        try {
            jmxService.registerModelMBean(ssBeanMBean, ssBeanON);
        } catch (Exception e) {
            logger.log(BasicLevel.WARN, "Could not register StatefulSessionBean MBean for " + beanName, e);
            return null;
        }
        return ssBeanON;
    }

    private String registerStatelessSessionBeanMBean(String domainName, String serverName, String j2eeappName, String moduleName
            , String beanName, JStatelessFactory jsf) {
        String ssBeanON = J2eeObjectName.getStatelessSessionBeanName(domainName, moduleName, serverName,
                j2eeappName, beanName);
        StatelessSessionBean ssBeanMBean = new StatelessSessionBean(ssBeanON, jsf, jmxService);
        try {
            jmxService.registerModelMBean(ssBeanMBean, ssBeanON);
        } catch (Exception e) {
            logger.log(BasicLevel.WARN, "Could not register StatelessSessionBean MBean" + beanName, e);
            return null;
        }
        return ssBeanON;
    }

    private String registerMdbBeanMBean(String domainName, String serverName, String j2eeappName, String moduleName
            , String beanName, JFactory jsf) {
        String mdbBeanON = J2eeObjectName.getMessageDrivenBeanName(domainName, moduleName, serverName,
                j2eeappName, beanName);
        MessageDrivenBean mdbBeanMBean = new MessageDrivenBean(mdbBeanON, jsf, jmxService);
        try {
            jmxService.registerModelMBean(mdbBeanMBean, mdbBeanON);
        } catch (Exception e) {
            logger.log(BasicLevel.WARN, "Could not register MessageDrivenBean MBean" + beanName, e);
            return null;
        }
        return mdbBeanON;
    }


    /**
     * Create and register MBeans for the module and for each EJB in the module
     * @param domainName the domain name
     * @param serverName the server name
     * @param j2eeappName the J2EE Application to which the module belongs or null if the module is not inside an application
     * @param moduleName the module name
     * @param container the container object corresponding to the module
     * @param fileName the module's file name
     * @param earUrl if the module is inside a ear, this is the url of the that ear
     * @param dd the module's deployment descriptor
     * @param factoryMap a map containing for each bean name the JFactory object necessary to construct
     * an EJB MBean
     * @return the module MBean's OBJECT_NAME
     */
    private String registerCompMBeans(String domainName, String serverName, String j2eeappName, String moduleName
            , JContainer container, String fileName, URL earUrl, DeploymentDesc dd, Map factoryMap, Map beanDescMap) {

        // Create and register EJBModule MBean
        EJBModule ejbModuleMBean = registerEJBModuleMBean(domainName, serverName, j2eeappName, moduleName
                , container, fileName, earUrl);
        String ejbModuleON = J2eeObjectName.getEJBModuleName(domainName, serverName, j2eeappName, moduleName);
        ejbModuleMBean.setDeploymentDescriptor(dd.getXmlContent());
        ejbModuleMBean.setJonasDeploymentDescriptor(dd.getJOnASXmlContent());

        // Create and register the different Bean MBeans types
        // The keys if factoryMap give the bean Names
        Iterator it = factoryMap.keySet().iterator();
        while(it.hasNext()) {
            String beanON = null;
            String beanName = (String) it.next();
            BeanFactory bf = (BeanFactory) factoryMap.get(beanName);
            BeanDesc beanDesc = (BeanDesc) beanDescMap.get(beanName);
            if (beanDesc instanceof EntityBmpDesc) {
                beanON = registerEntityBeanMBean(domainName, serverName, j2eeappName, moduleName, beanName
                        , (JEntityFactory) bf, BMP);
            } else if (beanDesc instanceof EntityCmpDesc) {
                beanON = registerEntityBeanMBean(domainName, serverName, j2eeappName, moduleName, beanName
                        , (JEntityFactory) bf, CMP);
            } else if (beanDesc instanceof SessionStatefulDesc) {
                beanON = registerStatefulSessionBeanMBean(domainName, serverName, j2eeappName, moduleName, beanName
                        , (JSessionFactory) bf);
            } else if (beanDesc instanceof SessionStatelessDesc) {
                beanON = registerStatelessSessionBeanMBean(domainName, serverName, j2eeappName, moduleName, beanName
                        , (JStatelessFactory) bf);
            } else if (beanDesc instanceof MessageDrivenDesc) {
                beanON = registerMdbBeanMBean(domainName, serverName, j2eeappName, moduleName, beanName
                        , (JFactory) bf);
            }
            ejbModuleMBean.addEjb(beanON);
        }
        return ejbModuleON;
    }


    /**
     * Create and register MBeans for the module and for each EJB in the module
     * @param domainName the domain name
     * @param serverName the server name
     * @param j2eeappName the J2EE Application to which the module belongs or null if the module is not inside an application
     * @param moduleName the module name
     * @param container the container object corresponding to the module
     * @param fileName the module's file name
     * @param earUrl if the module is inside a ear, this is the url of the that ear
     * @param dd the module's deployment descriptor
     * @param factoryMap a map containing for each bean name the JFactory object necessary to construct
     * an EJB MBean
     * @return the module MBean's OBJECT_NAME
     */
    private void unregisterCompMBeans(String domainName, String serverName, String j2eeappName, String moduleName) {

        ObjectName onEjbModule = J2eeObjectName.getEJBModule(domainName, serverName, j2eeappName, moduleName);

        MBeanServer mbeanServer = jmxService.getJmxServer();
        // try to remove MBeans corresponding to the EJBs in the module
        ObjectName onEjb = null;
        try {
            String[] onEjbs = (String[]) mbeanServer.getAttribute(onEjbModule, "ejbs");
            for (int i = 0; i < onEjbs.length; i++) {
                onEjb = new ObjectName(onEjbs[i]);
                jmxService.unregisterModelMBean(onEjb);
            }
        } catch (Exception e) {
            logger.log(BasicLevel.ERROR, "Could not unregsiter MBean " + onEjb.toString(), e);
        }

        // Unegister EJBModule MBean
        jmxService.unregisterModelMBean(onEjbModule);
    }

    /**
     * Call GenIC by using a wrapper.
     * @param jarPath path of the jar file
     * @param genicArgs arguments for GenIC
     */
    private void callGenic(final String jarPath, final String[] genicArgs) {
        String[] args;
        if (genicArgs != null) {
            args = new String[genicArgs.length + 1];
            for (int i = 0; i < genicArgs.length; i++) {
                args[i] = genicArgs[i];
            }
            args[genicArgs.length] = jarPath;
        } else {
            args = new String[1];
            args[0] = jarPath;
        }

        if (logger.isLoggable(BasicLevel.DEBUG)) {
            logger.log(BasicLevel.DEBUG, "Calling GenIC with arguments :" + Arrays.asList(args));
        }
        GenicServiceWrapper.callGenic(args);
    }

    /**
     * @param jmxService the jmxService to set
     */
    public void setJmxService(final JmxService jmxService) {
        this.jmxService = jmxService;
    }

    /**
     * @param securityService the securityService to set
     */
    public void setSecurityService(final SecurityService securityService) {
        this.securityService = securityService;
    }

    /**
     * @param transactionService the transactionService to set
     */
    public void setTransactionService(final TransactionService transactionService) {
        this.transactionService = transactionService;
    }

    /**
     * @param workManagerService the workManagerService to set
     */
    public void setWorkManagerService(final WorkManagerService workManagerService) {
        this.workManagerService = workManagerService;
    }

    /**
     * @param webServicesService the webServicesService to set
     */
    public void setWebServicesService(final WebServicesService webServicesService) {
        this.wsService = webServicesService;
    }

    /**
     * @param naming the naming to set
     */
    public void setNaming(final JNamingManager naming) {
        this.naming = naming;
    }

    /**
     * @param componentContextFactory the componentContextFactory to set
     */
    public void setComponentContextFactory(final JComponentContextFactory componentContextFactory) {
        this.componentContextFactory = componentContextFactory;
    }

    /**
     * @param registry the registry to set
     */
    public void setRegistry(final RegistryService registry) {
        this.registryService = registry;
    }

    /**
     * Set the reference on HA service.
     * @param haService the reference on HA service
     */
    public void setHaService(final HaService haService) {
        this.haService = haService;
    }

    /**
     * Set the reference on resource service.
     * @param resService the reference on Resource service
     */
    public void setResourceService(final ResourceService resService) {
        this.resService = resService;
    }

    /**
     * @return the reference on CMI service
     */
    public CmiService getCmiService() {
        return cmiService;
    }

    /**
     * Set the reference on CMI service.
     * @param cmiService the reference on CMI service
     */
    public void setCmiService(final CmiService cmiService) {
        this.cmiService = cmiService;
    }

    /**
     * @param deployerManagerService the deployerManagerService to set
     */
    public void setDeployerManager(final IDeployerManager deployerManager) {
        this.deployerManager = deployerManager;
    }

}
