/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 1999-2004 Bull S.A.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * Initial developer(s): Benoit PELLETIER
 * --------------------------------------------------------------------------
 * $Id: JarCleanTask.java 12246 2007-12-09 21:42:38Z benoitf $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.ejb.internal;

import java.io.File;
import java.net.MalformedURLException;
import java.util.Vector;

import org.ow2.jonas.ejb.EJBService;
import org.ow2.jonas.lib.work.AbsCleanTask;
import org.ow2.jonas.lib.work.CleanerException;
import org.ow2.jonas.lib.work.DeployerLog;
import org.ow2.jonas.lib.work.DeployerLogException;
import org.ow2.jonas.lib.work.FileManager;
import org.ow2.jonas.lib.work.FileManagerException;
import org.ow2.jonas.lib.work.LogEntry;

import org.objectweb.util.monolog.api.BasicLevel;

/**
 * JOnAS Jar unused copy clean task class. This class provides a way for
 * removing copies wich are unconsistent for jar files.
 * @author Florent BENOIT
 * @author Benoit PELLETIER
 */
public class JarCleanTask extends AbsCleanTask {

    /**
     * reference to the deployerLog
     */
    private static DeployerLog jarDeployerLog = null;

    private EJBService ejbService = null;

    /**
     * Default constructor : Construct a new cleaner.
     * @param jarDeployerLog the deployer logger
     */
    public JarCleanTask(DeployerLog jarDeployerLog, EJBService ejb) {
        super();
        JarCleanTask.jarDeployerLog = jarDeployerLog;
        ejbService = ejb;
    }

    /**
     * Return true if the work copy exists and is up to date
     * @param logEntry entry in a deploy log
     * @return true if the work copy exists and is up to date
     * @throws CleanerException if it fails
     */
    protected boolean isValidLogEntry(LogEntry logEntry) throws CleanerException {
        String fTimeStamp = null;
        File jarOriginalFile = logEntry.getOriginal();
        File jarCopyFile = logEntry.getCopy();

        if (getLogger().isLoggable(BasicLevel.DEBUG)) {
            getLogger().log(BasicLevel.DEBUG, "LogEntry exist :" + jarOriginalFile.exists());
        }

        // if the file doesn't exist, return
        if (!jarOriginalFile.exists()) {
            return false;
        }

        //get the timestamp
        try {
            fTimeStamp = FileManager.fileToTimeStampDir(jarOriginalFile.toURL(), ".jar");
        } catch (FileManagerException efme) {
            throw new CleanerException("Can't get the timestamp of the file " + jarOriginalFile + " : "
                    + efme.getMessage());
        } catch (MalformedURLException mue) {
            throw new CleanerException("Can't get the timestamp of the file " + jarOriginalFile + " : "
                    + mue.getMessage());
        }

        if (getLogger().isLoggable(BasicLevel.DEBUG)) {
            getLogger().log(BasicLevel.DEBUG, "LogEntry fTimeStamp :" + fTimeStamp);
            getLogger().log(BasicLevel.DEBUG, "LogEntry isValid :" + fTimeStamp.equalsIgnoreCase(jarCopyFile.getName()));
        }

        //compare
        return (fTimeStamp.equalsIgnoreCase(jarCopyFile.getName()));

    }

    /**
     * Remove the work copy specified in the log entry and the log entry
     * @param logEntry entry in a deploy log
     * @throws CleanerException if it fails
     */
    protected void removeLogEntry(LogEntry logEntry) throws CleanerException {
        // it's a file but this should work
        removeRecursiveDirectory(logEntry.getCopy());

        try {
            jarDeployerLog.removeEntry(logEntry);
        } catch (DeployerLogException edle) {
            throw new CleanerException("Can't remove an entry" + edle.getMessage());
        }

    }

    /**
     * Gets the log entries
     * @return the log entries
     */
    protected Vector getLogEntries() {
        return jarDeployerLog.getEntries();
    }

    /**
     * Check if the package pointed by the log entry is currently deploy
     * @param logEntry entry in a deploy log
     * @return true if the package pointed by the log entry is currently deployed
     * @throws CleanerException if it fails
     */
    protected boolean isDeployLogEntry(LogEntry logEntry) throws CleanerException {

        // check if the jar file is deployed
        return ejbService.isJarDeployedByWorkName(logEntry.getCopy().getName());

    }
}