/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: EJB21Deployer.java 13435 2008-03-28 13:31:04Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.ejb2.internal;

import java.io.File;
import java.net.URL;

import org.ow2.jonas.ejb2.EJBService;
import org.ow2.util.ee.deploy.api.archive.ArchiveException;
import org.ow2.util.ee.deploy.api.deployable.EJB21Deployable;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.ee.deploy.api.deployer.IDeployer;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.url.URLUtils;

/**
 * Deployer of the EJB 2.1 deployable.
 * @author Florent BENOIT
 */
public class EJB21Deployer implements IDeployer {

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(EJB21Deployer.class);

    /**
     * EJb 2.1 service used by this deployer.
     */
    private EJBService ejb21Service = null;

    /**
     * Undeploy the given EJB-JAR.
     * @param ejbJarDeployable the deployable to remove.
     * @throws DeployerException if the EJB-JAR is not undeployed.
     */
    protected void undeployEJB21JAR(final EJB21Deployable ejbJarDeployable) throws DeployerException {
        logger.info("Undeploying {0}", ejbJarDeployable);

        // Undeploy the EJb 2.1 file
        try {
            ejb21Service.removeContainer(getFile(ejbJarDeployable).getAbsolutePath());
        } catch (Exception e) {
            throw new DeployerException("Cannot deploy the EJB 2.1 deployable '"  + ejbJarDeployable + "'.", e);
        }

    }


    /**
     * Deploy the given EJB-JAR.
     * @param ejbJarDeployable the deployable to add.
     * @throws DeployerException if the EJB-JAR is not deployed.
     */
    protected void deployEJB21JAR(final EJB21Deployable ejbJarDeployable) throws DeployerException {
        logger.info("Deploying {0}", ejbJarDeployable);

        // Deploy
        try {
            ejb21Service.createContainer(getFile(ejbJarDeployable).getAbsolutePath());
        } catch (Exception e) {
            throw new DeployerException("Cannot deploy the EJB 2.1 deployable '"  + ejbJarDeployable + "'.", e);
        }
    }



    /**
     * Deploy an EJB deployable.
     * @param deployable a given deployable
     * @throws DeployerException if the deployment is not done.
     */
    public void deploy(final IDeployable<?> deployable) throws DeployerException {
        check(deployable);

        // Deploy the EJB-JAR Deployable
        if (deployable instanceof EJB21Deployable) {
            deployEJB21JAR((EJB21Deployable) deployable);
        }
    }

    /**
     * Checks if the given deployable is deployed or not.
     * @param deployable test if a given deployable is already deployed.
     * @return true if the deployable is deployed.
     * @throws DeployerException if the undeploy operation fails.
     */
    public boolean isDeployed(final IDeployable<?> deployable) throws DeployerException {
        check(deployable);

        // Check if EJB-JAR has been deployed
        return ejb21Service.isJarDeployed(getFile(deployable).getAbsolutePath());
    }


    /**
     * Allows to get a File from the given Deployable.
     * @param deployable the given deployable.
     * @return a File object of this deployable
     * @throws DeployerException if the File can't be obtained.
     */
    protected File getFile(final IDeployable<?> deployable) throws DeployerException {
        // Get URL
        URL url = null;
        try {
            url = deployable.getArchive().getURL();
        } catch (ArchiveException e) {
            throw new DeployerException("Cannot get URL from deployable '" + deployable + "'.", e);
        }

        // Get File
        return URLUtils.urlToFile(url);
    }


    /**
     * Checks if the given deployable is supported by the Deployer.
     * @param deployable the deployable to be checked
     * @return true if it is supported, else false.
     */
    public boolean supports(final IDeployable<?> deployable) {
        return (deployable instanceof EJB21Deployable);
    }

    /**
     * Undeploy the given EJB 2.1 deployable.
     * @param deployable a given deployable to undeploy
     * @throws DeployerException if the undeploy operation fails.
     */
    public void undeploy(final IDeployable<?> deployable) throws DeployerException {
        check(deployable);

        // Undeploy the EJB 2.1 Deployable
        if (deployable instanceof EJB21Deployable) {
            undeployEJB21JAR((EJB21Deployable) deployable);
        }
    }

    /**
     * Check if the given deployable is deployable or not.
     * @param deployable the deployable to check.
     * @throws DeployerException if the deployable is not supported.
     */
    private void check(final IDeployable<?> deployable) throws DeployerException {
        if (!supports(deployable)) {
            throw new DeployerException("The deployment of the deployable'" + deployable
                    + "' is not supported by this deployer.");
        }
    }

    /**
     * Sets the EJB 2.1 service.
     * @param ejb21Service the EJB 2.1 service.
     */
    public void setEjb21Service(final EJBService ejb21Service) {
        this.ejb21Service = ejb21Service;
    }


}
