/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jonas-team@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: EasyBeansService.java 10876 2007-07-08 17:21:03Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.ejb.easybeans;

import java.net.URL;

import javax.naming.Context;

import org.objectweb.jonas.service.AbsServiceImpl;
import org.objectweb.jonas.service.ServiceException;
import org.ow2.easybeans.api.EZBServer;
import org.ow2.easybeans.naming.interceptors.ENCManager;
import org.ow2.easybeans.server.Embedded;
import org.ow2.easybeans.server.EmbeddedConfigurator;
import org.ow2.easybeans.server.EmbeddedException;
import org.ow2.easybeans.server.ServerConfig;
import org.ow2.jonas.deployer.DeployerManagerService;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Implementation of the service that runs the EasyBeans EJB3 container.
 * @author Florent Benoit
 */
public class EasyBeansService extends AbsServiceImpl {

    /**
     * Name of the configuration file of EasyBeans for JOnAS.
     */
    public static final String EASYBEANS_CONFIG_FILE = "easybeans-jonas.xml";

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(EasyBeansService.class);

    /**
     * URL of the configuration file.
     */
    private URL xmlConfigurationURL = null;

    /**
     * Embedded instance.
     */
    private Embedded embedded = null;

    /**
     * Reference to the deployer.
     */
    private EasyBeansDeployer easyBeansDeployer = null;

    /**
     * Abstract initialization method to be implemented by sub-classes.
     * @param ctx configuration of the service
     * @throws ServiceException service initialization failed
     */
    protected void doInit(final Context ctx) throws ServiceException {

        // Get the context classloader
        ClassLoader threadCL = Thread.currentThread().getContextClassLoader();

        // JOnAS configuration
        logger.debug("Using ''{0}'' as classloader to load the resource named ''{1}''", threadCL, EASYBEANS_CONFIG_FILE);
        xmlConfigurationURL = threadCL.getResource(EASYBEANS_CONFIG_FILE);

        if (xmlConfigurationURL == null) {
            throw new ServiceException("The configuration file '" + EASYBEANS_CONFIG_FILE + "' was not found in the classloader");
        }

        // Init EasyBeans ENC Manager
        ENCManager.setInterceptorClass(JOnASENCInterceptor.class);

    }

    /**
     * Abstract start-up method to be implemented by sub-classes.
     * @throws ServiceException service start-up failed
     */
    protected void doStart() throws ServiceException {

        // Create instance
        try {
            embedded = EmbeddedConfigurator.create(xmlConfigurationURL);
        } catch (EmbeddedException e) {
            throw new ServiceException("Cannot create the EasyBeans server", e);
        }

        // Update configuration
        ServerConfig serverConfig = embedded.getServerConfig();

        // no infinite loop (managed by JOnAS)
        serverConfig.setShouldWait(false);

        // Enable MBeans
        serverConfig.setUseMBeans(true);

        // Disable the naming
        serverConfig.setUseNaming(false);

        // Use JOnAS JACC provider
        serverConfig.setInitJACC(false);

        // Using JOnAS monitoring
        serverConfig.setDirectoryScanningEnabled(false);

        // Using JOnAS JMX Connector
        serverConfig.setStartJMXConnector(false);

        // Start the EasyBeans server
        try {
            embedded.start();
        } catch (EmbeddedException e) {
            throw new ServiceException("Cannot start the EasyBeans Server", e);
        }

        // Create deployer
        easyBeansDeployer = new EasyBeansDeployer();
        easyBeansDeployer.setEmbedded(embedded);

        // Register the deployer
        DeployerManagerService.register(easyBeansDeployer);

    }

    /**
     * Abstract method for service stopping to be implemented by sub-classes.
     * @throws ServiceException service stopping failed
     */
    protected void doStop() throws ServiceException {

        // Unregister the deployer
        DeployerManagerService.unregister(easyBeansDeployer);

        // Stop the EasyBeans server
        if (embedded != null) {
            try {
                embedded.stop();
            } catch (EmbeddedException e) {
                throw new ServiceException("Cannot stop the EasyBeans component", e);
            }
        }
    }

    /**
     * TODO: Remove it when EarDeployer will use EasyBeans Deployer.
     * @return the Embedded instance used by this service.
     */
    public EZBServer getEasyBeansServer() {
        return embedded;
    }

}
