 /*
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 1999 Bull S.A.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * Initial developer(s): Eric HARDESTY
 * --------------------------------------------------------------------------
 * $Id: XAResourceImpl.java 15428 2008-10-07 11:20:29Z sauthieg $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.ee.jdbc;

import java.sql.SQLException;

import javax.resource.spi.ConnectionEvent;
import javax.transaction.xa.XAException;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;

import org.objectweb.util.monolog.api.BasicLevel;

/**
 * This is an implementation of XAResource used for transaction mgmt
 * with a JDBC driver.  The standard JDBC drivers don't support real
 * XA, so the prepare will just return OK and work will be done in the
 * commit/rollback.
 */
public class XAResourceImpl implements XAResource {

    // Current Xid
    private Xid currentXid = null;

    // Is a xa transaction started ?
    boolean started = false;
    boolean ended = true;

    protected ManagedConnectionImpl mc = null;
    private XAResource xares = null;

    /**
     * XAResourceImpl constructor
     * @param mc ManagedConnectionImpl used with this object
     * @param xares XAResource from DB if supported
     */
    public XAResourceImpl(ManagedConnectionImpl mc, XAResource xares) {
        this.mc = mc;
        this.xares = xares;
        if (mc.trace.isLoggable(BasicLevel.DEBUG)) {
            mc.trace.log(BasicLevel.DEBUG, "" + mc + "," + xares);
        }
    }

    /**
     * Commit the global transaction specified by xid.
     */
    public void commit(Xid xid, boolean onePhase) throws XAException {
        if (mc.trace.isLoggable(BasicLevel.DEBUG)) {
            mc.trace.log(BasicLevel.DEBUG, "" + xid + "," + onePhase);
        }

        // Commit the transaction
        try {
            if (xares != null) {
                xares.commit(xid, onePhase);
                if (xid.equals(currentXid)) {
                    started = false;
                }
            } else {
                // Make sure that this call is for the XID
                if (currentXid == null || !currentXid.equals(xid) || !started) {
                    mc.trace.log(BasicLevel.ERROR, "passed xid(" + xid
                                                   + "),current Xid("
                                                   + currentXid
                                                   + "),started(" + started + ")");
                    throw new XAException("Commit: Must call correct XAResource for its started XID");
                }
                mc.connection.commit();
                started = false;
            }
        } catch (XAException xe) {
            mc.trace.log(BasicLevel.ERROR, xe.getMessage());
            throw xe;
        } catch (SQLException e) {
            mc.trace.log(BasicLevel.ERROR, e.getMessage());
            try {
                mc.signalEvent(ConnectionEvent.CONNECTION_ERROR_OCCURRED, null, e);
            } catch (Exception ex) {}
            throw new XAException("Error on commit");
        }

        try {
            if (!mc.connection.getAutoCommit()) {
                mc.connection.setAutoCommit(true);
            }
        } catch (Exception exc) {
            if (xares == null) {
                if (mc.trace.isLoggable(BasicLevel.DEBUG)) {
                    mc.trace.log(BasicLevel.DEBUG,
                            "Unable to set autoCommit to true:", exc);
                }
            }
        }
    }

    /**
     * Ends the transaction
     */
    public void end(Xid xid, int flags) throws XAException {
        if (mc.trace.isLoggable(BasicLevel.DEBUG)) {
            mc.trace.log(BasicLevel.DEBUG, "" + xid + "," + flags);
        }
        // Make sure that the Xid has started
        if (currentXid == null || !currentXid.equals(xid)) {
            throw new XAException(XAException.XA_RBPROTO);
        }
        if (!started && ended) {
            throw new XAException(XAException.XA_RBPROTO);
        }
        ended = true;
        if (xares != null) {
            xares.end(xid, flags);
        }
    }

    /**
     * Need to forget the heuristically completed Xid
     */
    public void forget(Xid xid) throws XAException {
        if (mc.trace.isLoggable(BasicLevel.DEBUG)) {
            mc.trace.log(BasicLevel.DEBUG, "" + xid);
        }
        if (xares != null) {
            xares.forget(xid);
        }
    }

    /**
     * Get the current transaction timeout for this XAResource
     */
    public int getTransactionTimeout() throws XAException {
        if (xares != null) {
            return xares.getTransactionTimeout();
        }
        return 0;
    }

    /**
     * Determine if the resource manager instance is the same as the
     * resouce manager being passed in
     */
    public boolean isSameRM(XAResource xaRes) throws XAException {

        boolean ret = false;
        if (xaRes.equals(this)) {
            ret = true;
        } else if (!(xaRes instanceof XAResourceImpl)) {
            ret = false;
        } else {
            XAResourceImpl xaResImpl = (XAResourceImpl) xaRes;
            if (mc == xaResImpl.mc) {
                ret = true;
            }
        }
        if (mc.trace.isLoggable(BasicLevel.DEBUG)) {
            mc.trace.log(BasicLevel.DEBUG, "" + xaRes + "," + this + " is " + ret);
        }
        return ret;
    }

    /**
     * Prepare the Xid for a transaction commit
     */
    public int prepare(Xid xid) throws XAException {
        if (mc.trace.isLoggable(BasicLevel.DEBUG)) {
            mc.trace.log(BasicLevel.DEBUG, "" + xid);
        }
        if (xares != null) {
            int ret = xares.prepare(xid);
            // If read-only, then do what would have been done in the commit call
            if (ret == XA_RDONLY) {
                started = false;
                try {
                    if (!mc.connection.getAutoCommit()) {
                        mc.connection.setAutoCommit(true);
                    }
                } catch (Exception exc) { }
            }
            return ret;
        }
        // Just return true, since we are just simulating XA with our wrapper
        return XA_OK;
    }

    /**
     * Obtain a list of prepared transaction branches from a resource manager.
     */
    public Xid[] recover(int flag) throws XAException {
        if (mc.trace.isLoggable(BasicLevel.DEBUG)) {
            mc.trace.log(BasicLevel.DEBUG, "" + flag);
        }
        if (xares != null) {
            return xares.recover(flag);
        }
        // Not a full RM, so just return null
        return null;
    }

    /**
     * Roll back work done on the Xid
     */
    public void rollback(Xid xid) throws XAException {
        if (mc.trace.isLoggable(BasicLevel.DEBUG)) {
            mc.trace.log(BasicLevel.DEBUG, "" + xid);
        }

        // Rollback the Xid
        try {
            if (xares != null) {
                xares.rollback(xid);
                if (xid.equals(currentXid)) {
                    started = false;
                }
            } else {
                // Make sure that this call is for the XID
                if (currentXid == null || !currentXid.equals(xid) || !started) {
                    mc.trace.log(BasicLevel.ERROR, "passed xid(" + xid
                                     + "),current Xid(" + currentXid
                                     + "),started(" + started + ")");
                    throw new XAException("Rollback: Must call correct XAResource for its started XID");
                }
                mc.connection.rollback();
                started = false;
            }
        } catch (XAException xe) {
            mc.trace.log(BasicLevel.ERROR, xe.getMessage());
            throw xe;
        } catch (SQLException e) {
            try {
                mc.trace.log(BasicLevel.ERROR, e.getMessage());
                mc.signalEvent(
                    ConnectionEvent.CONNECTION_ERROR_OCCURRED, null, e);
            } catch (Exception ex) {}
            throw(new XAException("Error on rollback"));
        }
        try {
            if (!mc.connection.getAutoCommit()) {
                mc.connection.setAutoCommit(true);
            }
        } catch (Exception exc) {
            if (xares == null) {
                if (mc.trace.isLoggable(BasicLevel.DEBUG)) {
                    mc.trace.log(BasicLevel.DEBUG,
                            "Unable to set autoCommit to true:", exc);
                }
            }
        }
    }

    /**
     * Set the current transaction timeout value for this XAResource
     */
    public boolean setTransactionTimeout(int seconds) throws XAException {
        if (xares != null) {
            return xares.setTransactionTimeout(seconds);
        }
        return false;
    }

    /**
     * Start work on this Xid.  If TMJOIN is on, then join an existing transaction
     * already started by the RM.
     */
    public void start(Xid xid, int flags) throws XAException {
        if (mc.trace.isLoggable(BasicLevel.DEBUG)) {
            mc.trace.log(BasicLevel.DEBUG, "" + xid + "," + flags);
        }
        try {
            if (mc.connection.getAutoCommit()) {
                mc.connection.setAutoCommit(false);
            }
        } catch (Exception ex) {
            ex.printStackTrace();
            mc.trace.log(BasicLevel.ERROR,
                "Unable to set autoCommit to false:" + ex.getMessage());
            throw(new XAException(
                "Error : Unable to set autoCommit to false in start"));
        }

        if (xares != null) {
            xares.start(xid, flags);
        } else {
            if (started && (currentXid == null || !currentXid.equals(xid))) {
                mc.trace.log(BasicLevel.ERROR,
                    "Must call correct XAResource for its started XID");
                throw new XAException("XAResourceImpl.start : Must call correct XAResource for its started XID");
            }
        }
        currentXid = xid;
        started = true;
        ended = false;
    }
}
