/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2007-2008 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: MBeansRegistration.java 12708 2008-01-27 12:00:18Z fornacif $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.lib.jmbeans.osgi;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.remote.JMXServiceURL;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Logger;
import org.osgi.framework.BundleContext;
import org.ow2.jonas.Version;
import org.ow2.jonas.configuration.ConfigurationManager;
import org.ow2.jonas.jmx.JmxService;
import org.ow2.jonas.lib.jmbeans.J2EEServer;
import org.ow2.jonas.lib.jmbeans.JavaVm;
import org.ow2.jonas.lib.jmbeans.LogManager;
import org.ow2.jonas.lib.jmbeans.OsgiServiceListener;
import org.ow2.jonas.lib.jmbeans.OsgiServicesUtil;
import org.ow2.jonas.lib.jmbeans.ServiceManager;
import org.ow2.jonas.lib.jmbeans.ServiceManagerUpdater;
import org.ow2.jonas.lib.management.domain.J2EEDomain;
import org.ow2.jonas.lib.management.javaee.J2eeObjectName;
import org.ow2.jonas.lib.timer.TimerManager;
import org.ow2.jonas.lib.util.JonasObjectName;
import org.ow2.jonas.lib.util.Log;
import org.ow2.jonas.properties.ServerProperties;
import org.ow2.jonas.registry.RegistryService;
import org.ow2.util.ee.deploy.api.archive.IArchiveManager;
import org.ow2.util.ee.deploy.api.deployer.IDeployerManager;

/**
 * Activator for the bundle which creates the basic JSR77 MBeans.
 * Provides an OSGi service to update service manager.
 * @author Adriana Danes
 */
public class MBeansRegistration implements ServiceManagerUpdater {

    /**
     * Property value for the default file
     */
    private static final String DEF_LOG_CONFIGFILE = "trace";

    /**
     * Property for the configuration log file
     */
    private static final String LOG_CONFIGFILE = "jonas.log.configfile";

    /**
     * File for log configuration
     */
    private String logConfigFile;

    private static Logger logger = Log.getLogger("org.ow2.jonas.lib.jmbeans");

    /**
     * Jmx service instance.
     */
    private JmxService jmxService = null;

    /**
     * Registery service instance.
     */
    private RegistryService registryService = null;

    private ConfigurationManager configManager = null;

    /**
     * Basic naming properties:
     *
     * The management domain name.
     */
    private String domainName = null;
    /**
     * The server instance name.
     */
    private String serverName = null;
    /**
     * OSGi BundleContext.
     */
    private BundleContext bc = null;

    /**
     * ServerProperties service instance.
     */
    private ServerProperties serverProperties = null;

    /**
     * DeployerManager instance.
     */
    private IDeployerManager deployerManager = null;
    /**
     * ArchiveManager instance.
     */
    private IArchiveManager archiveManager = null;

    /**
     * TimerManager instance.
     */
    private TimerManager tm = null;
    
    /**
     * Service manager.
     */
    private ServiceManager serviceManager = null;

    /**
     * Constructor.
     * @param bc OSGi {@link BundleContext}.
     */
    public MBeansRegistration(BundleContext bc) {
        this.bc = bc;
    }

    public void start() throws Exception {
        logger.log(BasicLevel.DEBUG, "Starting jonas-mbeans Bundle...");

        // Get domainName and serverName from the Bootstrap service
        // Got all we need to create and register JSR77 MBeans
        try {
            domainName = serverProperties.getDomainName();
            serverName = serverProperties.getServerName();

            // TODO get a reference on a TimerManager service.
            tm = TimerManager.getInstance();

            // Set the domain name in JonasObjectName
            if (!JonasObjectName.isDomainSet()) {
                JonasObjectName.setDomain(domainName);
            }
            // set configuration info into the Log System (monolog)
            logConfigFile = serverProperties.getValue(LOG_CONFIGFILE, DEF_LOG_CONFIGFILE);
            Log.configure(logConfigFile);

            registerMBeans(serverProperties);
            // unset props as we don't need properties any more
            serverProperties = null;
        } catch (MalformedObjectNameException me) {
            logger.log(BasicLevel.ERROR, "Failed to register mbeans: " + me.getMessage());
            return;
        } catch (Exception e) {
            System.out.println("Failed to register mbeans");
            e.printStackTrace();
            return;
        }
    }

    /**
     * Stop the bundle.
     * @throws Exception
     */
    public void stop() throws Exception {
        logger.log(BasicLevel.DEBUG, "Stopping jonas-mbeans Bundle...");
        if (jmxService != null) {
            unregisterMBeans(domainName, serverName);
        }
        registryService = null;
    }

    /**
     * Register JSR77 MBeans.
     * @param props JONAS server properties
     * @throws Exception problem with some MBeans creation or registration
     */
    private void registerMBeans(final ServerProperties props) throws Exception {
        // Create and register a J2EEServer MBean
        // --------------------------------------
        String name = domainName + ":j2eeType=J2EEServer,name=" + serverName;
        ObjectName serverOn = ObjectName.getInstance(name);
        J2EEServer server = new J2EEServer(serverOn.toString(), true, false, true, bc);
        server.setServerVendor(Version.VENDOR);
        server.setServerVersion(Version.getNumber());
        server.setServerName(serverName);
        server.setDomainName(domainName);
        server.setAllProperties(props);
        server.setJonasBase(props.getValue("jonas.base"));
        server.setProtocols(getProtocols());
        server.setArchiveManager(archiveManager);
        server.setDeployerManager(deployerManager);
        server.setConfigManager(configManager);
        server.setOsgi(props.isOSGi());

        server.initRepositoryDirs();
        // TODO get the list of dirs from the DeployableMonitor service MBean
        String dirs = "ejbjars,ejbjars/autoload,webapps,webapps/autoload,rars,rars/autoload,apps,apps/autoload";
        server.setDirectories(dirs);

        // TODO un-comment below when install.root accessible
        //String versions = props.getVersionsFile();
        String versions = "Vesions";
        server.setVersions(versions);
        // activate memory monitoring
        server.setTimerManager(tm);
        server.setActivated(true);
        // Load the J2EEServer, J2EEDomain, JVM, WorkManager MBean descriptions
        Package p = server.getClass().getPackage();
        jmxService.loadDescriptors(p.getName(), getClass().getClassLoader());

        // Register J2EEServer MBean
        jmxService.registerModelMBean(server, serverOn);
        logger.log(BasicLevel.DEBUG, "MBeans registered: " + serverOn.toString());

        // Create and register J2EEDomain MBean
        // ------------------------------------
        ObjectName domainOn= J2eeObjectName.J2EEDomain(domainName);
        J2EEDomain domain = new J2EEDomain(domainOn.toString());
        domain.setJmxService(jmxService);
        boolean master = props.isMaster();
        domain.setMaster(master, serverName);
        jmxService.registerModelMBean(domain, domainOn);
        logger.log(BasicLevel.DEBUG, "MBeans registered: " + domainOn.toString());

        // Set relation J2EEDomain -> J2EEServer
        domain.setMyJ2EEServerOn(serverOn.toString());

        // Domain set-up
        JMXServiceURL[] connectorServerURLs = jmxService.getConnectorServerURLs();
        ArrayList<String> urls = new ArrayList<String>();
        for (JMXServiceURL url : connectorServerURLs) {
            if (url != null) {
                urls.add(url.toString());
            }
        }
        domain.addLocalServer(serverName, urls);// this initializes the Domain map

        // Create and register JavaVM MBean
        // --------------------------------
        String jvmName = serverName; // maybe this could be another name
        ObjectName objectName= J2eeObjectName.JVM(domainName, serverName, jvmName);
        JavaVm jvm = new JavaVm(objectName.toString(), props);
        jmxService.registerModelMBean(jvm, objectName);
        logger.log(BasicLevel.DEBUG, "MBeans registered: " + objectName.toString());

        // Set relation J2EEServer -> JavaVM
        server.addJavaVM(objectName.toString());

        logger.log(BasicLevel.DEBUG, "MBeans registered: " + objectName);

        // Create and register a MBean for the logging management
        String nameLogging = domainName + ":type=service,name=log,fname=" + fileNameForObjectName(logConfigFile);
        objectName = ObjectName.getInstance(nameLogging);
        LogManager logManager = new LogManager(objectName.toString());
        jmxService.registerModelMBean(logManager, objectName);
        logger.log(BasicLevel.DEBUG, "MBeans registered: " + objectName.toString());

        // Create ServiceManager and OsgiListener
        // --------------------------------------
        serviceManager = new ServiceManager(server);
        server.setServiceManager(serviceManager);
        updateServiceManager(props);

        // Create OsgiServiceListener
        OsgiServiceListener listener = new OsgiServiceListener(serviceManager);
        bc.addServiceListener(listener);

        //testAttributeListener();
    }

    /**
     * Unregister JSR77 MBeans
     * @param domainName
     * @param serverName
     * @throws Exception
     */
    private void unregisterMBeans(final String domainName, final String serverName) throws Exception {
        // Unregister J2EEServer MBean
        String name = domainName + ":j2eeType=J2EEServer,name=" + serverName;
        ObjectName objectName = ObjectName.getInstance(name);
        jmxService.unregisterModelMBean(objectName);
        logger.log(BasicLevel.DEBUG, "MBeans unregistered: " + objectName.toString());

        // Unregister JVM MBean
        String jvmName = serverName; // maybe this could be another name
        objectName = J2eeObjectName.JVM(domainName, serverName, jvmName);
        jmxService.unregisterModelMBean(objectName);
        logger.log(BasicLevel.DEBUG, "MBeans unregistered: " + objectName.toString());

        // Unregister J2EEDomain MBean
        objectName = J2eeObjectName.J2EEDomain(domainName);
        jmxService.unregisterModelMBean(objectName);
        logger.log(BasicLevel.DEBUG, "MBeans unregistered: " + objectName.toString());

        // Unregister the MBean for logging management
        String nameLogging = domainName + ":type=service,name=log,fname=" + fileNameForObjectName(logConfigFile);
        objectName = ObjectName.getInstance(nameLogging);
        jmxService.unregisterModelMBean(objectName);
        logger.log(BasicLevel.DEBUG, "MBeans unregistered: " + objectName.toString());

        //testAttributeListenerUndo();
    }

    /**
     * @return Get the protocols list from the registry service
     */
    private String getProtocols() {
        String protocols = "";
        String prefix = "rmi/";
        List protocolList = registryService.getActiveProtocolNames();
        Iterator it = protocolList.iterator();
        while (it.hasNext()) {
            String protocolName = (String) it.next();
            if (protocols == "") {
                protocols = prefix + protocolName;
            } else {
                protocols += "," +  prefix + protocolName;
            }
        }
        return protocols;
    }

    /**
     * Update ServiceManager with services state.
     * @throws Exception 
     */
    public void updateServiceManager(ServerProperties props) throws Exception {
        // Get already running non-mandatory services list
        ArrayList<String> runningServices = OsgiServicesUtil.getJonasServices(bc);
        // Get initially defined services list in JOnAS configuration file
        String[] services = props.getValueAsArray("jonas.services");
        
        serviceManager.deleteAllServices();
        for (String service : services) {
            // ignore "bootstrap" services
            if (service.equals("registry")) {
                continue;
            }
            if (service.equals("jmx")) {
                continue;
            }
            serviceManager.addService(service);
            if (runningServices.contains(service)) {
                serviceManager.setServiceState(service, OsgiServicesUtil.runningState());
            }
        }
    }

    /**
     * TODO this code comes from JonasAdminName. Put here temporary.
     * @param fileName
     * @return
     */
    private String fileNameForObjectName(String fileName) {
        return fileName.replace(':', '|');
    }
}
