/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: J2EEManagedObject.java 12999 2008-02-28 14:42:20Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.lib.jmbeans;

import javax.management.MBeanException;
import javax.management.Notification;
import javax.management.RuntimeOperationsException;
import javax.management.modelmbean.ModelMBean;

import org.ow2.jonas.lib.reconfig.ReconfigEmitter;

public class J2EEManagedObject  {
    // ------------------------------------------------------------- Private
    // Constants
    // the following strings represent key properties within
    // a J2EEManagedObject's objectName cf. JSR 77
    /**
     * Name constant
     */
    public static final String NAME = "name";

    /**
     * Constant for j2ee server
     */
    public static final String J2EE_TYPE_SERVER = "J2EEServer";

    /**
     * Constant for a j2ee application
     */
    public static final String J2EE_TYPE_APPLICATION = "J2EEApplication";

    // ------------------------------------------------------------- Properties

    /**
     * The managed object name
     */
    private String objectName;

    /**
     * State management support for this managed object (start, stop, ...)
     */
    private boolean stateManageable;

    /**
     * Performance statistics support for this managed object.
     */
    private boolean statisticsProvider;

    /**
     * Event provider support for this managed object.
     */
    private boolean eventProvider;

    /**
     * Reference on the model MBean
     */
    private ModelMBean modelMBean = null;

    // ------------------------------------------------------------- Contructors

    /**
     * MBean constructor
     * @param objectName The complete name of the managed object
     */
    protected J2EEManagedObject(final String objectName) {
        this.objectName = objectName;
        this.stateManageable = false;
        this.statisticsProvider = false;
        this.eventProvider = false;
    }

    /**
     * MBean constructor
     * @param objectName object name of the managed object
     * @param stateManageable if true, this managed object implements J2EE State
     *        Management Model
     * @param statisticsProvider if true, this managed object implements the
     *        J2EE StatisticProvide Model
     * @param eventProvider if true, this managed object implements the J2EE
     *        EventProvider Model
     */
    protected J2EEManagedObject(final String objectName, final boolean stateManageable, final boolean statisticsProvider,
            final boolean eventProvider) {
        this.objectName = objectName;
        this.stateManageable = stateManageable;
        this.statisticsProvider = statisticsProvider;
        this.eventProvider = eventProvider;
    }

    // ------------------------------------------------------------- Properties
    // accessors

    /**
     * Return this MBean's name
     * @return The name of the MBean (see OBJECT_NAME in the JSR77)
     */
    public String getObjectName() {
        return objectName;
    }

    /**
     * @return true if it is an event provider
     */
    public boolean isEventProvider() {
        return eventProvider;
    }

    /**
     * @return true if this managed object implements J2EE State Management
     *         Model
     */
    public boolean isStateManageable() {
        return stateManageable;
    }

    /**
     * @return true if this managed object implements the J2EE StatisticProvider
     *         Model
     */
    public boolean isStatisticsProvider() {
        return statisticsProvider;
    }

    /**
     * Set reference on my ModelMBean
     * @param modelMBean
     */
    public void setModelMBean(final ModelMBean modelMBean) {
        this.modelMBean = modelMBean;
    }

    /**
     * Send a <i>save configuration</i> notification to the registerd listeners.
     * @param sequenceNumber notification attribute
     * @param resourceName the name of a reconfigurable resource or of a JOnAS service
     */
    public void sendSaveNotification(final long sequenceNumber, final String resourceName) {
        // create save notification
       Notification saveNotif = new Notification(ReconfigEmitter.SAVE_RECONFIG_TYPE, modelMBean, sequenceNumber, resourceName);
       sendNotification(saveNotif);
   }

    /**
     * Send a <i>reconfiguration</i> notification to the registerd listener.
     * @param sequenceNumber notification attribute
     * @param resourceName the name of a reconfigurable resource or of a JOnAS service
     * @param userData data containing the name and the value of the reconfigured property(or properties)
     */
    public void sendReconfigNotification(final long sequenceNumber, final String resourceName, final Object userData) {
        // create notification and set the userData
        Notification configNotif =  new Notification(ReconfigEmitter.RECONFIG_TYPE, modelMBean, sequenceNumber, resourceName);
        configNotif.setUserData(userData);
        sendNotification(configNotif);
    }

    protected void sendNotification(final Notification notif) {
        try {
            if (modelMBean != null) {
                modelMBean.sendNotification(notif);
            } else {
                System.out.println("Can't send notification " + notif.toString());
            }
        } catch (RuntimeOperationsException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (MBeanException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
    }
}
