/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServiceManager.java 12922 2008-02-15 12:13:19Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.lib.jmbeans;

import java.util.Hashtable;
import java.util.Iterator;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

public class ServiceManager {
     private Log logger = LogFactory.getLog(J2EEServer.class);
    /**
     * Reference to the J2EEServer object associated to the managed server.
     */
    private J2EEServer server = null;
    /**
     * This table contains the managed services state.
     * The keys are the service names.
     */
    private Hashtable<String, Object> servicesState = null;

    /**
     * Construct a service manager for JOnAS server.
     * @param server the JOnAS server name
     */
    public ServiceManager(J2EEServer server) {
        this.server = server;
        this.servicesState = new Hashtable<String, Object> ();
    }

    /**
     * Add a server at server startup or restart.
     * @param serviceName the service name
     * @return the state of the service if a new services added, or old
     * state if its already present.
     */
    public Object addService(String serviceName) {
        return servicesState.put(serviceName, OsgiServicesUtil.stoppedState());
    }

    /**
     * Delete all services.
     */
    public void deleteAllServices() {
        servicesState.clear();
    }

    /**
     * Modify a service state. This may arrive at startup when a service registers itself
     * before the J2EEServer server initialization phase, or due to a service state
     * notification (ServiceEvent in OSGI).
     * @param serviceName
     * @param state
     * @return
     */
    public Object setServiceState(String serviceName, Object state) {
        if (servicesState == null) {
            logger.error("setServiceState called but servicesState table null");
            return null;
        }
        Object prevState = servicesState.put(serviceName, state);
        if (prevState != null && prevState.equals(state)) {
            return prevState;
        } else {
            // State changed,
            checkServerState();
            return state;
        }
    }

    public Object getServiceState(String serviceName) {
        return servicesState.get(serviceName);
    }

    public void checkServerState() {
        /**
         * Implement STARTING --> RUNNING transition
         * When all services are running.
         */
        if (server.isStopped()) {
            server.setStarting();
        }
        if (server.isStarting()) {
            // Check if all services are running
            boolean servicesRunning = true;
            Iterator<Object> values = servicesState.values().iterator();
            while (values.hasNext()) {
                Object value = values.next();
                if (!serviceRunning(value)) {
                    servicesRunning = false;
                    break;
                }
            }
            if (servicesRunning) {
                server.setServicesRunning();
            }
        }
        /**
         * Implement RUNNING --> STOPPED transition
         * Implement STOPPING --> STOPPED transition
         * When all services are stopped
         */
        if (server.isRunning() || server.isStopping()) {
            // Check if all services are stopped
            boolean stopped = true;
            Iterator<Object> values = servicesState.values().iterator();
            while (values.hasNext()) {
                Object value = values.next();
                if (!serviceStopped(value)) {
                    stopped = false;
                    break;
                }
            }
            if (stopped) {
                server.setStopped();
            }
        }
    }

    public void dumpServicesState() {
        java.util.Enumeration<String> names = servicesState.keys();
        logger.debug("--------------------------------");
        while (names.hasMoreElements()) {
            String name = names.nextElement();
            Boolean value = (Boolean) servicesState.get(name);
            if (value) {
                logger.debug("Service " + name + " running");
            } else {
                logger.debug("Service " + name + " stopped");
            }
        }
        logger.debug("");
    }
    private boolean serviceRunning(Object state) {
        Boolean value = (Boolean) state;
        if (value) {
            return true;
        } else {
            return false;
        }
    }
    private boolean serviceStopped(Object state) {
        Boolean value = (Boolean) state;
        if (!value) {
            return true;
        } else {
            return false;
        }
    }
}
