/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: OsgiServicesUtil.java 13980 2008-05-05 14:49:55Z danesa $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.lib.jmbeans;

import java.util.ArrayList;

import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;

/**
 * This class provides facilities for managing the JOnAS services implemented
 * as OSGI services.
 * @author danesa
 *
 */
public class OsgiServicesUtil {

    /**
     * OSGI service property defined for all the JOnAS services, giving the service name
     * for JOnAS management support.
     */
    public static final String SERVICE_NAME_PROP = "jonas.service";
    /**
     * OSGI service property defined for all the JOnAS services, having value true
     * for mandatory services: carol registry and jmx.
     */
    public static final String SERVICE_MANDATORY_PROP = "jonas.service.mandatory";
    /**
     * If the given ServiceReference has the "jonas.service" property set,
     * return its value representing the JOnAS service name.
     * @param ref the ServiceReference
     * @return "jonas.service" property value if set, null otherwise.
     */
    public static String jonasService(final ServiceReference ref) {
        Object prop = ref.getProperty(SERVICE_NAME_PROP);
        if (prop != null) {
            return (String) prop;
        } else {
            return null;
        }
    }
    /**
     * If the given ServiceReference has the "jonas.service.mandatory" property set,
     * return this property value (true). Return false otherwise.
     * @param ref the ServiceReference
     * @return true if "jonas.service.mandatory" property set, false otherwise
     */
    public static boolean jonasMandatoryService(final ServiceReference ref) {
        Object prop = ref.getProperty(SERVICE_MANDATORY_PROP);
        if (prop != null) {
            return (Boolean) prop;
        } else {
            return false;
        }
    }

    /**
     * Current implementation for the RUNNING state of a service.
     * @return true
     */
    public static Object runningState() {
        return ServiceState.RUNNING;
    }

    /**
     * Current implementation for the STOPPED state of a service.
     * @return false
     */
    public static Object stoppedState() {
        return ServiceState.STOPPED;
    }

    /**
     * Returns the names of the JOnAS services that are registered in the OSGI platform.
     * @param bc the BundleContext
     * @return running services' names
     * @throws InvalidSyntaxException could not get the OSGI services
     */
    public static ArrayList<String> getJonasServices(final BundleContext bc) throws InvalidSyntaxException {
        ServiceReference[] refs = bc.getAllServiceReferences(null, null);
        ArrayList<String> jonasServicesList = new ArrayList<String>();
        for (ServiceReference ref : refs) {
            String name = jonasService(ref);
            if (name != null) {
                jonasServicesList.add(name);
            }
        }
        return jonasServicesList;
    }
    /**
     * Returns the names of the JOnAS mandatory services that are registered in the OSGI platform.
     * @param bc the BundleContext
     * @return running services' names
     * @throws InvalidSyntaxException could not get the OSGI services
     */
    public static ArrayList<String> getJonasMandatoryServices(final BundleContext bc) throws InvalidSyntaxException {
        ServiceReference[] refs = bc.getAllServiceReferences(null, null);
        ArrayList<String> jonasServicesList = new ArrayList<String>();
        for (ServiceReference ref : refs) {
            String name = jonasService(ref);
            if (name != null) {
                if (jonasMandatoryService(ref)) {
                    jonasServicesList.add(name);
                }
            }
        }
        return jonasServicesList;
    }
}
