/**
 *
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 1999-2011 Bull S.A.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: javaURLContextFactory.java 20920 2011-02-16 12:42:32Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.lib.naming.java;

import java.util.Hashtable;

import javax.naming.Context;
import javax.naming.Name;
import javax.naming.spi.ObjectFactory;

import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Context factory for javaURLContext objects.
 * This factory will be used for all "java:..." urls provided as Name objects
 * for all JNDI operations.
 *
 * @author Philippe Durieux
 * @author Philippe Coq
 */
public class javaURLContextFactory implements ObjectFactory {

    /**
     * Logger used by JOnAS.
     */
    private static Log logger = LogFactory.getLog(javaURLContextFactory.class);

    /**
     * Returns an instance of javaURLContext for a java URL.
     *
     * If url is null, the result is a context for resolving java URLs.
     * If url is a URL, the result is a context named by the URL.
     *
     * @param url   String with a "java:" prefix or null.
     * @param name  Name of context, relative to ctx, or null.
     * @param ctx   Context relative to which 'name' is named.
     * @param env   Environment to use when creating the context
     * @return new created object.
     * @throws Exception if it fails.
     */
    public Object getObjectInstance(final Object url, final Name name, final Context ctx, final Hashtable env) throws Exception {
        if (url == null) {
            // All naming operations with "java:..." comes here
            // Users are encouraged to used intermediate contexts:
            // ctx = ic.lookup("java:comp/env") called only once (perfs)
            return new javaURLContext(env);
        }
        if (url instanceof String) {
            // Don't know what to do here
            logger.error("javaURLContextFactory.getObjectInstance(''{0}'')", url);
            return null;
        } else if (url instanceof String[]) {
            // Don't know what to do here
            logger.error("javaURLContextFactory.getObjectInstance(''{0}'')", url);
            return null;
        } else {
            // invalid argument
            throw (new IllegalArgumentException("javaURLContextFactory"));
        }
    }
}
