/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: RARDeployer.java 13922 2008-04-27 16:29:21Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.resource.internal;

import java.io.File;
import java.net.URL;

import org.ow2.jonas.resource.ResourceService;
import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.api.deployable.RARDeployable;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;
import org.ow2.util.ee.deploy.api.deployer.IDeployer;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;
import org.ow2.util.url.URLUtils;

/**
 * This deployer will deploy RAR module.
 * @author Florent BENOIT
 */
public class RARDeployer implements IDeployer {

    /**
     * Logger.
     */
    private Log logger = LogFactory.getLog(RARDeployer.class);

    /**
     * Resource service used by this deployer.
     */
    private ResourceService resourceService = null;

    /**
     * Undeploy the given RAR.
     * @param rarDeployable the deployable to remove.
     * @throws DeployerException if the RAR is not undeployed.
     */
    protected void undeployRAR(final RARDeployable rarDeployable) throws DeployerException {
        logger.info("Undeploying {0}", rarDeployable);

        // Undeploy the RAR file
        try {
            resourceService.unDeployRar(getFile(rarDeployable).getAbsolutePath());
        } catch (Exception e) {
            throw new DeployerException("Cannot deploy the RAR deployable '" + rarDeployable + "'.", e);
        }

    }

    /**
     * Deploy the given RAR.
     * @param rarDeployable the deployable to add.
     * @throws DeployerException if the RAR is not deployed.
     */
    protected void deployRAR(final RARDeployable rarDeployable) throws DeployerException {
        logger.info("Deploying {0}", rarDeployable);

        // Deploy the RAR file
        try {
            resourceService.deployRar(getFile(rarDeployable).getPath());
        } catch (Exception e) {
            throw new DeployerException("Cannot deploy the RAR deployable '" + rarDeployable + "'.", e);
        }
    }

    /**
     * Allows to get a File from the given RAR Deployable.
     * @param rarDeployable the given RAR deployable.
     * @return a File object of this deployable
     * @throws DeployerException if the File can't be obtained.
     */
    protected File getFile(final RARDeployable rarDeployable) throws DeployerException {
        // Get URL
        URL rarURL = null;
        try {
            rarURL = rarDeployable.getArchive().getURL();
        } catch (ArchiveException e) {
            throw new DeployerException("Cannot get URL from RAR deployable '" + rarDeployable + "'.", e);
        }

        // Get File
        return URLUtils.urlToFile(rarURL);
    }

    /**
     * Deploy a RAR deployable.
     * @param deployable a given deployable
     * @throws DeployerException if the deployment is not done.
     */
    public void deploy(final IDeployable<?> deployable) throws DeployerException {
        check(deployable);

        // Deploy the EAR Deployable
        if (deployable instanceof RARDeployable) {
            deployRAR((RARDeployable) deployable);
        }
    }

    /**
     * Checks if the given deployable is deployed or not.
     * @param deployable test if a given deployable is already deployed.
     * @return true if the deployable is deployed.
     * @throws DeployerException if the undeploy operation fails.
     */
    public boolean isDeployed(final IDeployable<?> deployable) throws DeployerException {
        check(deployable);

        // Check RarDeployable
        if (deployable instanceof RARDeployable) {
            return resourceService.isRarLoaded(getFile((RARDeployable) deployable).getAbsolutePath());
        }

        // Shouldn't go here as check method will let only RAR deployable
        return false;
    }

    /**
     * Checks if the given deployable is supported by the Deployer.
     * @param deployable the deployable to be checked
     * @return true if it is supported, else false.
     */
    public boolean supports(final IDeployable<?> deployable) {
        return (deployable instanceof RARDeployable);
    }

    /**
     * Undeploy the given RAR deployable.
     * @param deployable a given deployable to undeploy
     * @throws DeployerException if the undeploy operation fails.
     */
    public void undeploy(final IDeployable<?> deployable) throws DeployerException {
        check(deployable);

        // Undeploy the EAR Deployable
        if (deployable instanceof RARDeployable) {
            undeployRAR((RARDeployable) deployable);
        }
    }

    /**
     * Check if the given deployable is deployable or not.
     * @param deployable the deployable to check.
     * @throws DeployerException if the deployable is not supported.
     */
    private void check(final IDeployable<?> deployable) throws DeployerException {
        if (!supports(deployable)) {
            throw new DeployerException("The deployment of the deployable'" + deployable
                    + "' is not supported by this deployer.");
        }
    }

    /**
     * Sets the RAR service.
     * @param resourceService RAR service.
     */
    public void setResourceService(final ResourceService resourceService) {
        this.resourceService = resourceService;
    }

}
