/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 1999-2004 Bull S.A.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * Initial developer: Florent BENOIT
 * --------------------------------------------------------------------------
 * $Id: PolicyProvider.java 12690 2008-01-25 12:39:08Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.security.internal;

import org.ow2.jonas.lib.security.jacc.JPolicy;
import org.ow2.jonas.lib.util.Log;



import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Logger;

/**
 * Helper class for initializing the JACC provider
 * @author Florent Benoit
 */
public class PolicyProvider {

    /**
     * Only internal constructor, as it is an utility class
     */
    private PolicyProvider() {

    }

    /**
     * Logger which is used
     */
    private static Logger logger = null;

    /**
     * Init the JACC configuration
     * Defines in JACC Section 2
     * @throws SecurityException if JACC policy provider can not be set
     */
    public static void init() throws SecurityException {

        if (logger == null) {
            logger = Log.getLogger(Log.JONAS_SECURITY_PREFIX);
        }

        // Check if we have to use an existing policy provider
        // Section 2.7

        String javaPolicy = System.getProperty("javax.security.jacc.policy.provider");

        if (javaPolicy != null) {
            try {
                java.security.Policy.setPolicy((java.security.Policy) Class.forName(javaPolicy).newInstance());
            } catch (ClassNotFoundException cnfe) {
                // problem with property value of classpath
                throw new SecurityException(cnfe.getMessage());
            } catch (IllegalAccessException iae) {
                // problem with policy class definition
                throw new SecurityException(iae.getMessage());
            } catch (InstantiationException ie) {
                // problem with policy instantiation
                throw new SecurityException(ie.getMessage());
            } catch (ClassCastException cce) {
                // Not instance of java.security.policy
                throw new SecurityException(cce.getMessage());
            }
            logger.log(BasicLevel.INFO, "Using policy provider '" + javaPolicy + "'");
        }

        // Defines the JOnAS JACC provider if no provider is already defined
        // Section 2.3
        String jaccFactoryProvider = System.getProperty("javax.security.jacc.PolicyConfigurationFactory.provider");
        if (jaccFactoryProvider == null) {
            logger.log(BasicLevel.INFO, "Using JOnAS PolicyConfigurationFactory provider and JOnAS Policy provider");
            System.setProperty("javax.security.jacc.PolicyConfigurationFactory.provider",
                    "org.ow2.jonas.lib.security.jacc.PolicyConfigurationFactoryWrapper");
            // Add the JOnAS delegating policy provider
            java.security.Policy.setPolicy(JPolicy.getInstance());
        } else {
            logger.log(BasicLevel.INFO, "Using factory '" + jaccFactoryProvider
                    + "' as PolicyConfigurationFactory provider");
        }

        // TODO : Register Context Handler


    }

}
