/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 1999-2004 Bull S.A.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or 1any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * Initial developer: Florent Benoit
 * --------------------------------------------------------------------------
 * $Id: JResourceLDAPMBean.java 12246 2007-12-09 21:42:38Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.security.internal.realm.factory;

/**
 * Define the methods for the MBean
 * @author Florent Benoit
 */
public interface JResourceLDAPMBean extends JResourceMBean {

    /**
     * Clear the cache
     */
    void clearCache();

    // Getters

    /**
     * Return the initial context factory of this LDAP realm
     * @return the initial context factory
     */
    String getInitialContextFactory();

    /**
     * Get the Url of the ldap server of this LDAP realm
     * @return Url of the ldap server
     */
    String getProviderUrl();

    /**
     * Get the authentication used during the authentication to the LDAP server
     * of this LDAP realm
     * @return authentication used during the authentication to the LDAP server
     */
    String getSecurityAuthentication();

    /**
     * Get the DN of the Principal(username) of this LDAP realm
     * @return DN of the Principal(username)
     */
    String getSecurityPrincipal();

    /**
     * Get the Credential(password) of the principal of this LDAP realm
     * @return Credential(password) of the principal
     */
    String getSecurityCredentials();

    /**
     * Get the security protocol to use of this LDAP realm
     * @return security protocol to use
     */
    String getSecurityProtocol();

    /**
     * Get the preferred language to use with the service of this LDAP realm
     * @return language preferred language to use with the service
     */
    String getLanguage();

    /**
     * Get how referrals encountered by the service provider are to be processed
     * @return how referrals encountered by the service provider are to be
     *         processed
     */
    String getReferral();

    /**
     * Get the the list of state factories of this LDAP realm
     * @return list of state factories
     */
    String getStateFactories();

    /**
     * Get the mode for validate the authentication of this LDAP realm
     * @return BIND_AUTHENTICATION_MODE or COMPARE_AUTHENTICATION_MODE
     */
    String getAuthenticationMode();

    /**
     * Get the attribute in order to get the password of this LDAP realm
     * @return attribute in order to get the password of this LDAP realm
     */
    String getUserPasswordAttribute();

    /**
     * Get the attribute in order to get the user role from the ldap server
     * @return attribute in order to get the user role from the ldap server
     */
    String getUserRolesAttribute();

    /**
     * Get the role name when performing a lookup on a role
     * @return role name when performing a lookup on a role
     */
    String getRoleNameAttribute();

    /**
     * Get the DN used for the lookup of this LDAP realm
     * @return baseDN DN used for the lookup
     */
    String getBaseDN();

    /**
     * Get the DN used when searching the user DN. Override the baseDN if it is
     * defined
     * @return userDN DN used when searching the user DN
     */
    String getUserDN();

    /**
     * Get the filter used when searching the user
     * @return userSearchFilter filter used when searching the user
     */
    String getUserSearchFilter();

    /**
     * Get the DN used when searching the role DN. Override the baseDN if it is
     * defined
     * @return roleDN DN used when searching the role DN. Override the baseDN if
     *         it is defined
     */
    String getRoleDN();

    /**
     * Get the filter used when searching the role
     * @return roleSearchFilter filter used when searching the role
     */
    String getRoleSearchFilter();

    /**
     * Get the default algorithm
     * @return the default algorithm
     */
    String getAlgorithm();

    // Setters

    /**
     * Set the initial context factory of this LDAP realm
     * @param initialContextFactory the initial context factory
     */
    void setInitialContextFactory(String initialContextFactory);

    /**
     * Set the Url of the ldap server of this LDAP realm
     * @param providerUrl Url of the ldap server
     */
    void setProviderUrl(String providerUrl);

    /**
     * Set the authentication used during the authentication to the LDAP server
     * of this LDAP realm
     * @param securityAuthentication authentication used during the
     *        authentication to the LDAP server
     */
    void setSecurityAuthentication(String securityAuthentication);

    /**
     * Set the DN of the Principal(username) of this LDAP realm
     * @param securityPrincipal DN of the Principal(username)
     */
    void setSecurityPrincipal(String securityPrincipal);

    /**
     * Set the Credential(password) of the principal of this LDAP realm
     * @param securityCredentials Credential(password) of the principal
     */
    void setSecurityCredentials(String securityCredentials);

    /**
     * Set the security protocol to use of this LDAP realm
     * @param securityProtocol security protocol to use
     */
    void setSecurityProtocol(String securityProtocol);

    /**
     * Set the preferred language to use with the service of this LDAP realm
     * @param language preferred language to use with the service
     */
    void setLanguage(String language);

    /**
     * Set how referrals encountered by the service provider are to be processed
     * @param referral how referrals encountered by the service provider are to
     *        be processed
     */
    void setReferral(String referral);

    /**
     * Set the the list of state factories of this LDAP realm
     * @param stateFactories list of state factories
     */
    void setStateFactories(String stateFactories);

    /**
     * Set the mode for validate the authentication of this LDAP realm
     * @param authenticationMode BIND_AUTHENTICATION_MODE or
     *        COMPARE_AUTHENTICATION_MODE
     */
    void setAuthenticationMode(String authenticationMode);

    /**
     * Set the attribute in order to get the password of this LDAP realm
     * @param userPasswordAttribute attribute in order to get the password of
     *        this LDAP realm
     */
    void setUserPasswordAttribute(String userPasswordAttribute);

    /**
     * Set the attribute in order to get the user role from the ldap server
     * @param userRolesAttribute attribute in order to get the user role from
     *        the ldap server
     */
    void setUserRolesAttribute(String userRolesAttribute);

    /**
     * Set the role name when performing a lookup on a role
     * @param roleNameAttribute role name when performing a lookup on a role
     */
    void setRoleNameAttribute(String roleNameAttribute);

    /**
     * Set the DN used for the lookup of this LDAP realm
     * @param baseDN DN used for the lookup
     */
    void setBaseDN(String baseDN);

    /**
     * Set the DN used when searching the user DN. Override the baseDN if it is
     * defined
     * @param userDN DN used when searching the user DN
     */
    void setUserDN(String userDN);

    /**
     * Set the filter used when searching the user
     * @param userSearchFilter filter used when searching the user
     */
    void setUserSearchFilter(String userSearchFilter);

    /**
     * Set the DN used when searching the role DN. Override the baseDN if it is
     * defined
     * @param roleDN DN used when searching the role DN. Override the baseDN if
     *        it is defined
     */
    void setRoleDN(String roleDN);

    /**
     * Set the filter used when searching the role
     * @param roleSearchFilter filter used when searching the role
     */
    void setRoleSearchFilter(String roleSearchFilter);

    /**
     * Set the default algorithm to use
     * @param algorithm algorithm to be used
     */
    void setAlgorithm(String algorithm);

}