/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 1999 Bull S.A.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JResourceManagerWrapper.java 12246 2007-12-09 21:42:38Z benoitf $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.security.lib.wrapper;

import java.io.Reader;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.ow2.jonas.lib.bootstrap.LoaderManager;
import org.ow2.jonas.security.internal.JResources;
import org.ow2.jonas.security.internal.SecurityServiceException;


/**
 * @author Guillaume Sauthier
 */
public class JResourceManagerWrapper {

    /**
     * JResourcesManager instance
     */
    private static Object managerInstance = null;

    /**
     *
     */
    private static Method addResourcesMethod = null;

    /**
     * Empty private constructor for Utility Classes
     */
    private JResourceManagerWrapper() {
    }

    /**
     * Add JResource(s) created from parsing of the reader contents inside the
     * JResources given instance.
     * @param jres JResources element where JResource will be addded.
     * @param reader XML Content Reader
     * @param xml filename / xml : used in Error messages
     * @throws SecurityServiceException When parsing fails
     */
    public static void addResources(JResources jres, Reader reader, String xml) throws SecurityServiceException {
        if (managerInstance == null) {
            managerInstance = getJResourceManagerInstance();
        }

        if (addResourcesMethod == null) {
            addResourcesMethod = getJResourceManagerMethod();
        }

        try {
            addResourcesMethod.invoke(managerInstance, new Object[] {jres, reader, xml});
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            if (t instanceof SecurityServiceException) {
                throw (SecurityServiceException) t;
            } else if (t instanceof Error) {
                throw (Error) t;
            }
            throw new SecurityServiceException("Exception during JResourcesManager.addResources invocation", e);
        } catch (Exception e) {
            throw new SecurityServiceException("Exception during JResourcesManager.addResources invocation", e);
        }
    }

    /**
     * @return Returns the JResourceManager.addResources Method
     * @throws SecurityServiceException When addResources invocation fails
     */
    private static Method getJResourceManagerMethod() throws SecurityServiceException {
        try {
            return managerInstance.getClass().getMethod("addResources",
                    new Class[] {JResources.class, Reader.class, String.class});
        } catch (Exception e) {
            throw new SecurityServiceException("Cannot get JResourcesManager.addResources method", e);
        }
    }

    /**
     * @return Returns the JResourceManager unique instance.
     * @throws SecurityServiceException When getInstance invocation fails
     */
    private static Object getJResourceManagerInstance() throws SecurityServiceException {

        LoaderManager lm = LoaderManager.getInstance();
        try {
            ClassLoader tools = lm.getToolsLoader();

            Class jrmClass = tools.loadClass("org.ow2.jonas.security.lib.JResourceManager");
            Method m = jrmClass.getMethod("getInstance", new Class[] {});
            return m.invoke(null, new Object[] {});
        } catch (InvocationTargetException e) {
            Throwable t = e.getTargetException();
            if (t instanceof SecurityServiceException) {
                throw (SecurityServiceException) t;
            } else if (t instanceof Error) {
                throw (Error) t;
            }
            throw new SecurityServiceException("InvocationTargetException during JResourcesManager.getInstance invocation : " + e.getMessage(), e);
        } catch (Exception e) {
            throw new SecurityServiceException("Exception during JResourcesManager.getInstance invocation : " + e.getMessage(), e);
        }
    }

}