/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JonasVersionsServlet.java 20452 2010-10-12 15:42:21Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.versions.servlet;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.jar.Attributes;
import java.util.logging.Logger;

import javax.servlet.Servlet;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletResponse;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Instantiate;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.Validate;
import org.osgi.service.http.HttpContext;
import org.osgi.service.http.HttpService;
import org.osgi.service.http.NamespaceException;
import org.ow2.util.archive.api.IArchive;
import org.ow2.util.archive.api.IArchiveMetadata;
import org.ow2.util.archive.impl.ArchiveManager;
import org.ow2.util.ee.deploy.api.deployable.IDeployable;
import org.ow2.util.ee.deploy.impl.helper.DeployableHelper;
import org.ow2.util.ee.deploy.impl.helper.DeployableHelperException;

/**
 * OSGi HTTP servlet that returns the contents of the versions.properties file
 * as well as the versions of all files in <code>JONAS_BASE/deploy</code>.
 */
@Component(name = JonasVersionsServlet.COMPONENT_NAME, immediate = true)
@Instantiate(name = JonasVersionsServlet.COMPONENT_NAME)
public class JonasVersionsServlet implements Servlet {

    public static final String COMPONENT_NAME = "JonasVersionsServlet";

    public static final String CONTEXT = "/services/versions";

    private static final String NEWLINE = "\r\n";

    private static final String SEPARATOR = "   ";

    private static final Logger logger = Logger.getLogger(JonasVersionsServlet.class.getName());

    @Requires
    private HttpService httpService;

    private ServletConfig servletConfig;

    @Validate
    public void start() throws ServletException, NamespaceException {
        final HttpContext httpContext = this.httpService.createDefaultHttpContext();
        Dictionary<String, String> initParams;
        initParams = new Hashtable<String, String>();
        initParams.put("servlet-name", JonasVersionsServlet.CONTEXT);
        this.httpService.registerServlet(JonasVersionsServlet.CONTEXT, this, initParams, httpContext);

        JonasVersionsServlet.logger.info("JonasVersionsServlet listening on context " + JonasVersionsServlet.CONTEXT);
    }

    @Invalidate
    public void stop() {
        this.httpService.unregister(JonasVersionsServlet.CONTEXT);
    }

    public void init(final ServletConfig servletConfig) throws ServletException {
        this.servletConfig = servletConfig;
    }

    public ServletConfig getServletConfig() {
        return this.servletConfig;
    }

    public void service(final ServletRequest request, final ServletResponse rsp) throws ServletException, IOException {
        HttpServletResponse response = (HttpServletResponse) rsp;

        response.setContentType("text/plain;charset=us-ascii");
        response.setCharacterEncoding("US-ASCII");

        PrintWriter writer = response.getWriter();

        File[] jonasBaseDeployContents = new File(System.getProperty("jonas.base"), "deploy").listFiles();
        this.printVersions("deployables' versions", jonasBaseDeployContents, writer);

        BufferedReader versionsFileReader = new BufferedReader(new FileReader(new File(System.getProperty("jonas.root"),
            "versions.properties")));
        try {
            String read;
            while ((read = versionsFileReader.readLine()) != null) {
                writer.write(read);
                writer.write(JonasVersionsServlet.NEWLINE);
            }
        } finally {
            versionsFileReader.close();
            versionsFileReader = null;
        }
    }

    private void printVersions(final String directoryName, final File[] directoryContents, final PrintWriter writer)
        throws IOException {
        if (directoryContents != null) {
            writer.write("# ");
            writer.write(directoryName);

            writer.write(JonasVersionsServlet.NEWLINE);
            writer.write(JonasVersionsServlet.NEWLINE);

            for (File deployableFile : directoryContents) {
                IArchive archive = ArchiveManager.getInstance().getArchive(deployableFile);
                if (archive == null) {
                    continue;
                }
                IArchiveMetadata metadata = archive.getMetadata();
                if (metadata == null) {
                    continue;
                }

                IDeployable<?> deployable;
                try {
                    deployable = DeployableHelper.getDeployable(archive);
                } catch (DeployableHelperException e) {
                    IOException exception = new IOException("Failed opening archive: " + e);
                    exception.initCause(e);
                    throw exception;
                }
                if (deployable == null) {
                    continue;
                }

                String versionNumber = metadata.get(Attributes.Name.IMPLEMENTATION_VERSION.toString());
                String name = deployable.getModuleName();
                if (name != null && versionNumber != null) {
                    String versionNumberSuffixShort = "-" + versionNumber;
                    String versionNumberSuffixLong = "-version" + versionNumber;
                    if (name.endsWith(versionNumberSuffixShort)) {
                        name = name.substring(0, name.length() - versionNumberSuffixShort.length());
                    } else if (name.endsWith(versionNumberSuffixLong)) {
                        name = name.substring(0, name.length() - versionNumberSuffixLong.length());
                    }

                    writer.write(name);
                    writer.write(JonasVersionsServlet.SEPARATOR);
                    writer.write(versionNumber);
                    writer.write(JonasVersionsServlet.NEWLINE);
                }
            }

            writer.write(JonasVersionsServlet.NEWLINE);
            writer.write(JonasVersionsServlet.NEWLINE);
        }
    }

    public String getServletInfo() {
        return this.toString();
    }

    public void destroy() {
        // Nothing to do
    }
}
