/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or 1any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JOnASUtils.java 12428 2007-12-14 18:39:45Z fornacif $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.launcher.felix.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.Properties;

/**
 * JOnAS utilities.
 * @author Guillaume Sauthier
 */
public final class JOnASUtils {

    /**
     * System property name for JONAS_ROOT.
     */
    private static final String JONAS_ROOT_PROP = "jonas.root";

    /**
     * System property name for JONAS_BASE.
     */
    private static final String JONAS_BASE_PROP = "jonas.base";

    /**
     * JOnAS Development mode.
     */
    public static final String JONAS_DEVELOPER_PROP = "jonas.developer";

    /**
     * JOnAS Development mode.
     */
    public static final String JONAS_PROPERTIES = "jonas.properties";

    /**
     * JOnAS Base directory.
     */
    private static File base;

    /**
     * JOnAS Root directory.
     */
    private static File root;

    /**
     * Are we in developer mode ?
     * TODO Will be initialized by System property.
     */
    private static Boolean developerMode = null;


    /**
     * Empty private constructor.
     */
    private JOnASUtils() {
    }

    /**
     * @return the JOnAS Base to be used
     */
    public static File getJOnASBase() {
        if (base == null) {
            // Init JOnAS Base value (using jonas.root if unset)
            String jb = System.getProperty(JONAS_BASE_PROP,
                                           System.getProperty(JONAS_ROOT_PROP));
            base = new File(jb);
        }
        return base;
    }


    /**
     * Returns the value associated to the given property in jonas.properties file.
     * @param property the property name
     * @param defaultValue the default value
     * @return The value associated to the given property.
     * @throws Exception If the property value cannot be retrieved
     */
    public static String getServerProperty(final String property, final String defaultValue) throws Exception {
        Properties props = new Properties();
        InputStream fis = new FileInputStream(JOnASUtils.getJOnASBase() + File.separator + "conf" + File.separator
                + JOnASUtils.JONAS_PROPERTIES);
        props.load(fis);
        return (String) props.getProperty(property, defaultValue);
    }

    /**
     * @return the JOnAS Root to be used. May throw an {@link IllegalStateException}
     * if <code>jonas.root</code> System property is not set.
     */
    public static File getJOnASRoot() {
        if (root == null) {
            // Init JOnAS Root value
            String jr = System.getProperty(JONAS_ROOT_PROP);
            if (jr == null) {
                throw new IllegalStateException("Property 'jonas.root' is not set but is required");
            }
            root = new File(jr);
        }
        return root;
    }

    /**
     * @return the JOnAS version from the Implementation-Version Manifest entry.
     */
    public static String getVersion() {
        return JOnASUtils.class.getPackage().getImplementationVersion();
    }

    /**
     * @return true if the <code>jonas.developer</code> system property has been set.
     */
    public static boolean isDeveloperMode() {
        if (developerMode == null) {
            boolean mode = Boolean.getBoolean(JONAS_DEVELOPER_PROP);
            developerMode = Boolean.valueOf(mode);
        }
        return developerMode.booleanValue();
    }

}
