/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: IOUtils.java 14368 2008-06-23 15:49:31Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.launcher.felix.util;

import java.io.File;
import java.io.IOException;

/**
 * IO utilities.
 * @author Guillaume Sauthier
 */
public final class IOUtils {

    /**
     * Empty private constructor.
     */
    private IOUtils() {
    }

    /**
     * Recursively delete a directory.
     * @param dir directory to be deleted
     * @return true if everything went smoothly.
     */
    public static boolean deleteDir(final File dir) {

        // not a directory, abort quickly
        if (!dir.isDirectory()) {
            return false;
        }

        // to see if this directory is actually a symbolic link to a directory,
        // we want to get its canonical path - that is, we follow the link to
        // the file it's actually linked to
        File candir;
        try {
            candir = dir.getCanonicalFile();
        } catch (IOException e) {
            return false;
        }

        // a symbolic link has a different canonical path than its actual path,
        // unless it's a link to itself
        if (!candir.equals(dir.getAbsoluteFile())) {
            // this file is a symbolic link, and there's no reason for us to
            // follow it, because then we might be deleting something outside of
            // the directory we were told to delete
            return false;
        }

        // now we go through all of the files and sub-directories in the
        // directory and delete them one by one
        File[] files = candir.listFiles();
        if (files != null) {
            for (int i = 0; i < files.length; i++) {
                File file = files[i];

                // in case this directory is actually a symbolic link, or it's
                // empty, we want to try to delete the link before we try
                // anything
                boolean deleted = file.delete();
                if (!deleted) {
                    // deleting the file failed, so maybe it's a non-empty
                    // directory
                    if (file.isDirectory()) {
                        deleteDir(file);
                    }

                    // otherwise, there's nothing else we can do
                }
            }
        }

        // now that we tried to clear the directory out, we can try to delete it
        // again
        return dir.delete();
    }

	/**
	 * Return a OS valid path.
	 * @param base basis directory
	 * @param relative relative path name to the basis directory
	 * @return the OS dependent path name
	 */
	public static String getSystemPath(final File base, final String relative) {
	    return getSystemFile(base, relative).getPath();
	}

	/**
	 * Return a OS valid File.
	 * @param base basis directory
	 * @param relative relative path name to the basis directory
	 * @return the OS dependent File
	 */
	public static File getSystemFile(final File base, final String relative) {
	    return new File(base, relative.replace('/', File.separatorChar));
	}
}
