/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2005 Bull S.A.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DomainManagement.java 12246 2007-12-09 21:42:38Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.jonas.ee.mejb;

import java.rmi.RemoteException;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.ObjectName;
import javax.management.QueryExp;
import javax.management.ReflectionException;
import javax.management.j2ee.Management;

/**
 * This is a variation on the javax.management.j2ee.Management interface
 * required to be exposed by JOnAS through the MEJB session bean. This extends
 * the main functionality of the MEJB session bean by allowing clients to query
 * mbean information from other servers within the same domain.<br/><br/>
 * Please Note: <code>domainServerName</code> in any of the exposed methods
 * can be set to one of the values of the array returned by the following call
 * to a <i> master</i> server:
 * <code> getAttribute(null, domainObjectName, "serverNames")
 * </code> where
 * <code>ObjectName domainObjectName</code> corresponds to the ObjectName:
 * <i>domainName</i>:j2eeType=J2EEDomain,name=<i>domainName</i>
 *
 * @author Vivek Lakshmanan
 */
public interface DomainManagement extends Management {

	/**
	 * Gets the value of a specified attribute inside the mbean corresponding to
	 * the objectname <code>name</code> for the server with name
	 * <code> domainServerName </code>.
	 *
	 * @param domainServerName
	 *            Name of the server in domain. Use <code>null</code> for
	 *            referring to this server.
	 * @param name
	 *            ObjectName corresponding to the MBean to query.
	 * @param attribute
	 *            The member attribute to be queried.
	 * @return Value of the attribute.
	 * @throws MBeanException
	 * @throws AttributeNotFoundException
	 * @throws InstanceNotFoundException
	 * @throws ReflectionException
	 * @throws RemoteException
	 */
	Object getAttribute(String domainServerName, ObjectName name,
			String attribute) throws MBeanException,
			AttributeNotFoundException, InstanceNotFoundException,
			ReflectionException, RemoteException;

	/**
	 * Gets the values of attributes in <code>attributes</code> inside the
	 * mbean corresponding to the objectname <code>name</code> for the server
	 * with name <code>domainServerName</code>.
	 *
	 * @param domainServerName
	 *            Name of the server in domain. Use <code>null</code> for
	 *            referring to this server.
	 * @param name
	 *            ObjectName corresponding to the MBean to query.
	 * @param attributes
	 *            Array of attribute names to be queried.
	 * @return Value of the attributes.
	 * @throws InstanceNotFoundException
	 * @throws ReflectionException
	 * @throws RemoteException
	 */
	AttributeList getAttributes(String domainServerName, ObjectName name,
			String[] attributes) throws InstanceNotFoundException,
			ReflectionException, RemoteException;

	/**
	 * The number of MBeans registered in the mbean server corresponding to the
	 * server <code>domainServerName</code> in the domain.
	 *
	 * @param domainServerName
	 *            Name of the server in domain. Use <code>null</code> for
	 *            referring to this server.
	 * @return Number of MBeans currently registered in server.
	 * @throws MBeanException
	 * @throws AttributeNotFoundException
	 * @throws InstanceNotFoundException
	 * @throws ReflectionException
	 * @throws RemoteException
	 */
	Integer getMBeanCount(String domainServerName) throws MBeanException,
			AttributeNotFoundException, InstanceNotFoundException,
			ReflectionException, RemoteException;

	/**
	 * Get information about a specific MBean with ObjectName <code>name</code>
	 * registered on server <code>domainServerName</code>.
	 *
	 * @param domainServerName
	 *            Name of the server in domain. Use <code>null</code> for
	 *            referring to this server.
	 * @param name
	 *            ObjectName corresponding to the MBean to query.
	 * @return Information about queried MBean.
	 * @throws IntrospectionException
	 * @throws InstanceNotFoundException
	 * @throws ReflectionException
	 * @throws RemoteException
	 */
	MBeanInfo getMBeanInfo(String domainServerName, ObjectName name)
			throws IntrospectionException, InstanceNotFoundException,
			ReflectionException, RemoteException;

	/**
	 * Invoke an operation on an MBean with ObjectName <code>name</code>.
	 *
	 * @param domainServerName
	 *            Name of the server in domain. Use <code>null</code> for
	 *            referring to this server.
	 * @param name
	 *            ObjectName corresponding to the MBean to query.
	 * @param operationName
	 *            Name of operation to invoke.
	 * @param params
	 *            Parameters to pass to the operation.
	 * @param signature
	 *            Signature of operation.
	 * @return Result of invocation.
	 * @throws MBeanException
	 * @throws InstanceNotFoundException
	 * @throws ReflectionException
	 * @throws RemoteException
	 */
	Object invoke(String domainServerName, ObjectName name,
			String operationName, Object[] params, String[] signature)
			throws MBeanException, InstanceNotFoundException,
			ReflectionException, RemoteException;

	/**
	 * Check if the MBean with ObjectName <code>name</code> is registered in
	 * the server with the name <code>domainServerName</code>.
	 *
	 * @param domainServerName
	 *            Name of the server in domain. Use <code>null</code> for
	 *            referring to this server.
	 * @param name
	 *            ObjectName corresponding to the MBean to query.
	 * @return <code>true</code> if registered.
	 * @throws RemoteException
	 */
	boolean isRegistered(String domainServerName, ObjectName name)
			throws RemoteException;

	/**
	 * Query for ObjectNames on the server named <code>domainServerName</code>based
	 * on a query string.
	 *
	 * @param domainServerName
	 *            Name of the server in domain. Use <code>null</code> for
	 *            referring to this server.
	 * @param name
	 *            ObjectName corresponding to the query string.
	 * @param query
	 *            Query expression to apply on ObjectName.
	 * @return MBeans matching the query.
	 * @throws RemoteException
	 */
	Set queryNames(String domainServerName, ObjectName name, QueryExp query)
			throws RemoteException;

	/**
	 * Sets the value of a specified attribute inside the mbean corresponding to
	 * the objectname <code>name</code> for the server with name
	 * <code> domainServerName </code>.
	 *
	 * @param domainServerName
	 *            Name of the server in domain. Use <code>null</code> for
	 *            referring to this server.
	 * @param name
	 *            ObjectName corresponding to the query string.
	 * @param attribute
	 *            The member attribute to be set.
	 * @throws MBeanException
	 * @throws AttributeNotFoundException
	 * @throws InstanceNotFoundException
	 * @throws InvalidAttributeValueException
	 * @throws ReflectionException
	 * @throws RemoteException
	 */
	void setAttribute(String domainServerName, ObjectName name,
			Attribute attribute) throws MBeanException,
			AttributeNotFoundException, InstanceNotFoundException,
			InvalidAttributeValueException, ReflectionException,
			RemoteException;

	/**
	 * Sets the values of attributes in <code>attributes</code> inside the
	 * mbean corresponding to the objectname <code>name</code> for the server
	 * with name <code>domainServerName</code>.
	 *
	 * @param domainServerName
	 *            Name of the server in domain. Use <code>null</code> for
	 *            referring to this server.
	 * @param name
	 *            ObjectName corresponding to the MBean.
	 * @param attributes
	 *            Array of attribute names to be set.
	 * @return Values of the attributes.
	 * @throws InstanceNotFoundException
	 * @throws ReflectionException
	 * @throws RemoteException
	 */
	AttributeList setAttributes(String domainServerName, ObjectName name,
			AttributeList attributes) throws InstanceNotFoundException,
			ReflectionException, RemoteException;

}
