/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2005 Bull S.A.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ManagementEndpoint.java 15428 2008-10-07 11:20:29Z sauthieg $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.ee.mejb;

import java.rmi.Remote;
import java.rmi.RemoteException;

/**
 * This is the interface for a monitoring webservice used to remotely
 * monitor mbeans in the domain.
 * @author Matt Wringe
 * @author Vivek Lakshmanan
 */
public interface ManagementEndpoint extends Remote {

    /**
     * Returns the name of the current domain.
     * @return The name of the domain.
     * @throws RemoteException    If an error occurs communicating with the server.
     */
    String getDefaultDomain() throws RemoteException;

    /**
     * Returns the server names in the domain.
     * @return The servers in the domain.
     * @throws RemoteException                If an error occurs with the web service.
     * @throws ManagementEndpointException    If any error occurs with getServers.
     */
    String[] getServers() throws ManagementEndpointException, RemoteException;

    /**
     * Returns the attribute value for an mbean on a server in the domain.
     * @param domainServerName The name of the server in the domain.
     * @param objectName The objectname of the mbean.
     * @param attribute The attribute to be returned.
     * @return The value of the attribute.
     * @throws ManagementEndpointException    If any error occurs with getAttribute.
     * @throws RemoteException                If an error occurs with the web service.
     */
    String[] getAttribute(String domainServerName, String objectName, String attribute)
        throws ManagementEndpointException, RemoteException;

    /**
     * Returns the number of mbeans for a server in the domain.
     * @param domainServerName The name of the server in the domain.
     * @return The number of mbeans for the specified server.
     * @throws RemoteException    If an error occurs communicating with the server.
     */
    Integer getMBeanCount(String domainServerName) throws RemoteException;

    /**
     * Returns True if the mbean is registered with the server in the domain,
     * false otherwise.
     * @param domainServerName The name of the server in the domain.
     * @param objectName The objectname of the mbean.
     * @return True if the mbean is registered, false if not.
     * @throws ManagementEndpointException    If any error occurs with isRegistered.
     * @throws RemoteException                If an error occurs with the web service.
     */
    boolean isRegistered(String domainServerName, String objectName)
        throws ManagementEndpointException, RemoteException;

    /**
     * Returns the search results for mbeans on a server in the domain. The query
     * parameter currently does nothing.
     * @param domainServerName The name of the server in the domain.
     * @param objectName The objectname of the mbean.
     * @param query Currently does nothing.
     * @return The ObjectNames for the queried mbeans.
     * @throws ManagementEndpointException    If any error occurs with queryNames.
     * @throws RemoteException                If an error occurs with the web service.
     */
    String[] queryNames(String domainServerName, String objectName, String query)
        throws ManagementEndpointException, RemoteException;

    /**
     * Returns a list of attributes for a specified mbean on a server in the
     * domain.
     * @param domainServerName The name of the server in the domain.
     * @param objectName The objectname of the mbean.
     * @return The list of attributes for the mbean.
     * @throws ManagementEndpointException    If any error occurs with getAttributesList.
     * @throws RemoteException                If an error occurs with the web service.
     */
    String[] getAttributesList(String domainServerName, String objectName)
        throws ManagementEndpointException, RemoteException;

    /**
     * Returns the description for an mbean on a server in the domain.
     * @param domainServerName The name of the server in the domain.
     * @param objectName The objectname of the mbean.
     * @return The description for the mbean.
     * @throws ManagementEndpointException    If any error occurs with getDescription.
     * @throws RemoteException                If an error occurs with the web service.
     */
    String getDescription(String domainServerName, String objectName)
        throws ManagementEndpointException, RemoteException;

    /**
     * Returns the list of operations for an mbean on a server in the domain.
     * @param domainServerName The name of the server in the domain.
     * @param objectName The objectname of the mbean.
     * @return The operations for the mbean.
     * @throws ManagementEndpointException    If any error occurs with getOperations.
     * @throws RemoteException                If an error occurs with the web service.
     */
    String[] getOperations(String domainServerName, String objectName)
        throws ManagementEndpointException, RemoteException;

    /**
     * Invokes an mbean operation having String params on a server in the domain.
     * @param domainServerName The name of the server in the domain.
     * @param objectName The objectname of the mbean.
     * @param operationName operation name
     * @param params String parameters
     * @return The operation result
     * @throws ManagementEndpointException    If any error occurs with getOperations.
     * @throws RemoteException                If an error occurs with the web service.
     */
    String[] invoke(String domainServerName, String objectName, String operationName, String[] params)
        throws ManagementEndpointException, RemoteException;

}
