/**
 * JOnAS Configurator
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Jonas4.java 18711 2009-09-22 12:51:31Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.tools.configurator.impl;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.logging.Logger;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.listener.TimestampedLogger;
import org.objectweb.jonas.ant.JOnASBaseTask;
import org.objectweb.jonas.ant.JOnASClusterConfigTask;
import org.objectweb.jonas.ant.cluster.ClusterDaemon;
import org.objectweb.jonas.ant.jonasbase.Carol;
import org.objectweb.jonas.ant.jonasbase.Db;
import org.objectweb.jonas.ant.jonasbase.Discovery;
import org.objectweb.jonas.ant.jonasbase.Ha;
import org.objectweb.jonas.ant.jonasbase.JdbcRa;
import org.objectweb.jonas.ant.jonasbase.Jms;
import org.objectweb.jonas.ant.jonasbase.JonasProperties;
import org.objectweb.jonas.ant.jonasbase.Mail;
import org.objectweb.jonas.ant.jonasbase.Services;
import org.objectweb.jonas.ant.jonasbase.WebContainer;
import org.objectweb.jonas.ant.jonasbase.web.Ajp;
import org.objectweb.jonas.ant.jonasbase.web.Cluster;
import org.objectweb.jonas.ant.jonasbase.web.Http;
import org.objectweb.jonas.ant.jonasbase.web.Https;
import org.objectweb.jonas.ant.jonasbase.web.Tomcat;
import org.objectweb.jonas.ant.jonasbase.wsdl.WsdlPublish;
import org.ow2.jonas.tools.configurator.api.JonasConfigurator;

/**
 * JonasConfigurator implementation for Jonas 4.x
 * 
 * @author Remy Bresson
 */
public class Jonas4 extends FileReplacerHelper implements JonasConfigurator {

    /**
     * JONAS_BASE
     */
    private File jonasFileBase;

    /**
     * Link to the jonas ant task
     */
    private JOnASBaseTask jonasBase;

    /**
     * The cluster daemon configuration task
     */
    private JOnASClusterConfigTask clusterDaemonTask;

    /**
     * The cluster daemon confguration
     */
    private ClusterDaemon clusterDaemon;

    /**
     * Configuration logger
     */
    private static Logger logger = Logger.getLogger(Jonas4.class.getName());

    /**
     * To configure protocols
     */
    private Carol carol;

    /**
     * Has JOnAS protocols list been set?
     */
    private boolean isProtocolsListSet = false;

    /**
     * To configure databases service
     */
    private Db db;

    /**
     * Whether the optional db service has been activated
     */
    private boolean isDbActivated = false;

    /**
     * To configure discovery sevice
     */
    private Discovery discovery;

    /**
     * Whether Discovery is set as master
     */
    private boolean isDiscoveryMaster = false;

    /**
     * To configure jdbc
     */
    private JdbcRa jdbcRa;

    /**
     * Whether the optional jdbc adapter has been activated
     */
    private boolean isJdbcRaActivated = false;

    /**
     * To configure jms
     */
    private Jms jms;

    /**
     * To configure jonas global properties
     */
    private JonasProperties jonasProperties;

    /**
     * To configure mail service
     */
    private Mail mail;

    /**
     * Whether the optional mail service has been activated
     */
    private boolean isMailActivated = false;

    /**
     * To configure active services
     */
    private Services services;

    /**
     * To configure web container
     */
    private WebContainer webContainer;

    /**
     * To configure the tomcat (in the web container)
     */
    private Tomcat tomcat;

    /**
     * To configure ajp
     */
    private Ajp ajp;

    /**
     * To configure http
     */
    private Http http;

    /**
     * To configure https
     */
    private Https https;

    /**
     * To configure cluster
     */
    private Cluster cluster;

    /**
     * To configure wsdl wsdlPublisherFile
     */
    private org.objectweb.jonas.ant.jonasbase.wsdl.File wsdlPublisherFile;

    /**
     * To configure WSDL publishing
     */
    private WsdlPublish wsdlPublish;

    /**
     * Whether the optional WSDL publishing service has been activated
     */
    private boolean isWsdlPublishActivated = false;

    /**
     * To configure ha
     */
    private Ha ha;

    /**
     * The project, not really usefull, only to have a project to pass to ant
     * task.
     */
    private Project project;

    /**
     * Save the jonas root to be able to re use it after setting
     */
    private boolean isJonasRootSet = false;

    /**
     * Save the jonas base to be able to re use it after setting
     */
    private boolean isJonasBaseSet = false;

    /**
     * Is http webservice activated
     */
    private Boolean isHttpActivated = true;

    /**
     * Is https webservice activated
     */
    private Boolean isHttpsActivated = false;

    /**
     * Is ajp webservice activated
     */
    private Boolean isAjpActivated = false;

    /**
     * Is http replication webservice activated
     */
    private Boolean isHttpReplicationActivated = false;

    /**
     * Is ha activated
     */
    private Boolean isHaActivated = false;

    /**
     * Is ejb clustering activated
     */
    private Boolean isEjbClusteringActivated = false;

    /**
     * Save the value of mcast_addr for cmi configuration
     */
    private String saveCmiMCastAddr;

    /**
     * Save the value of mcast_port for cmi configuration
     */
    private String saveCmiMCastPort;

    /**
     * Default constructor
     */
    public Jonas4() {
        this.jonasBase = new JOnASBaseTask();
        this.createProject();
        this.jonasBase.setProject(this.project);

        this.clusterDaemonTask = new JOnASClusterConfigTask();
        this.clusterDaemonTask.setProject(this.project);

        this.carol = new Carol();
        this.db = new Db();
        this.discovery = new Discovery();
        this.jdbcRa = new JdbcRa();

        this.jms = new Jms();
        try {
            this.jonasProperties = new JonasProperties();
        } catch (NoClassDefFoundError e) {
            // JOnAS before revision 12011, which is just after the 4.8.6
            // release, didn't have JonasProperties. We'll handle that manually.
            this.jonasProperties = null;
            this.services = new Services();
        }
        this.mail = new Mail();

        this.webContainer = new WebContainer();
        this.tomcat = new Tomcat();
        this.ajp = new Ajp();
        this.http = new Http();
        this.https = new Https();
        this.cluster = new Cluster();
        this.wsdlPublish = new WsdlPublish();
        this.wsdlPublisherFile = new org.objectweb.jonas.ant.jonasbase.wsdl.File();
        this.ha = new Ha();

        this.clusterDaemon = new ClusterDaemon();
        this.clusterDaemon.setProtocol("jrmp");
        this.clusterDaemon.setPort("1099");

        Jonas4.logger.finest("Created the JOnAS 4 configurator instance");
    }

    /**
     * Create the project
     */
    private void createProject() {
        this.project = new Project();
        this.project.init();
        this.project.initProperties();

        TimestampedLogger logger = new TimestampedLogger();
        logger.setMessageOutputLevel(Project.MSG_DEBUG);

        logger.setOutputPrintStream(new PrintStream(new OutputStream() {
            StringBuffer buf = new StringBuffer();

            @Override
            public void write(final int b) throws IOException {
                if (b == '\n' || b == '\r') {
                    if (this.buf.length() > 0) {
                        Jonas4.logger.finest(this.buf.toString());
                        this.buf.setLength(0);
                    }
                } else {
                    this.buf.append((char) b);
                }
            }
        }));

        this.project.addBuildListener(logger);
    }

    /* ------------------------------- */
    /* Jonas global parameters */

    /**
     * {@inheritDoc}
     */
    public void setJdk(final String jdk) {
        Jonas4.logger.finest("[GlobalParameters] setting jdk : " + jdk);
        NotApplicableHelper.notApplicable("GlobalParameters.Jdk");
    }

    /**
     * {@inheritDoc}
     */
    public void setJavaOpts(final String javaOpts) {
        Jonas4.logger.finest("[GlobalParameters] setting java opts : " + javaOpts);
        if (this.jonasProperties != null) {
            this.jonasProperties.setJvmopts(javaOpts);
        } else {
            // Setting seems not to be applied in any wsdlPublisherFile,
            // skipping
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasRoot(final String jonasRoot) {
        Jonas4.logger.finest("[GlobalParameters] setting jonas root : " + jonasRoot);
        File jonasFileRoot = new File(jonasRoot);
        this.jonasBase.setJonasRoot(jonasFileRoot);
        this.discovery.setJonasRoot(jonasRoot);
        this.isJonasRootSet = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasBase(final String jonasBase) {
        Jonas4.logger.finest("[GlobalParameters] setting jonas base : " + jonasBase);
        this.jonasFileBase = new File(jonasBase);
        this.jonasBase.setDestDir(this.jonasFileBase);
        this.isJonasBaseSet = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasName(final String jonasName) {
        Jonas4.logger.finest("[GlobalParameters] setting jonas name : " + jonasName);
        if (this.jonasProperties != null) {
            this.jonasProperties.setServerName(jonasName);
        } else {
            // Setting seems not to be applied in any wsdlPublisherFile,
            // skipping
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasDomain(final String jonasDomain) {
        Jonas4.logger.finest("[GlobalParameters] setting jonas domain : " + jonasDomain);
        if (this.jonasProperties != null) {
            this.jonasProperties.setDomainName(jonasDomain);
        } else {
            // Setting seems not to be applied in any wsdlPublisherFile,
            // skipping
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setHost(final String jonasHost) {
        Jonas4.logger.finest("[GlobalParameters] setting host : " + jonasHost);
        this.addReplacement("carol.properties", "://localhost:", "://" + jonasHost + ":");
        NotApplicableHelper.notApplicable("GlobalParameters.JMSHost");
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasDevelopment(final boolean development) {
        Jonas4.logger.finest("[GlobalParameters] setting development : " + development);
        NotApplicableHelper.notApplicable("GlobalParameters.development");
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Jonas protocols configuation */
    /**
     * {@inheritDoc}
     */
    public void setProtocolsList(final String protocolsList) {
        Jonas4.logger.finest("[Protocols configuration] setting protocols list : " + protocolsList);
        this.carol.setProtocols(protocolsList);
        this.isProtocolsListSet = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsIrmiPort(final String irmiPort) {
        Jonas4.logger.finest("[Protocols configuration] setting irmi port : " + irmiPort);
        this.carol.setIrmiPort(irmiPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsIiopPort(final String iiopPort) {
        Jonas4.logger.finest("[Protocols configuration] setting iiop port : " + iiopPort);
        this.carol.setIiopPort(iiopPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsJrmpPort(final String jrmpPort) {
        Jonas4.logger.finest("[Protocols configuration] setting jrmp port : " + jrmpPort);
        this.carol.setJrmpPort(jrmpPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsCmiPort(final String cmiPort) {
        Jonas4.logger.finest("[Protocols configuration] setting cmi port : " + cmiPort);
        this.carol.setCmiPort(cmiPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsLocalCallOptimized(final Boolean localCallOptimized) {
        Jonas4.logger.finest("[Protocols configuration] setting jndi local call optimisation : " + localCallOptimized);
        this.carol.setJrmpOptimization(localCallOptimized);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Active services */

    /**
     * {@inheritDoc}
     */
    public void setServices(final String services) {
        Jonas4.logger.finest("setting active services list : " + services);
        if (this.jonasProperties != null) {
            this.jonasProperties.setServices(services);
        } else {
            this.services.setNames(services);
        }
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service web configuration */
    /**
     * {@inheritDoc}
     */
    public void setHttpsConnectorActivation(final Boolean activation) {
        Jonas4.logger.finest("[Web configuration] setting isHttpsActivated : " + activation);
        this.isHttpsActivated = activation;
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpConnectorActivation(final Boolean activation) {
        Jonas4.logger.finest("[Web configuration] setting isAjpActivated : " + activation);
        this.isAjpActivated = activation;
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpSessionReplicationActivation(final Boolean activation) {
        Jonas4.logger.finest("[Web configuration] setting isHttpReplicationActivated : " + activation);
        this.isHttpReplicationActivated = activation;
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpPort(final String httpPort) {
        Jonas4.logger.finest("[Web configuration] setting http port : " + httpPort);
        this.http.setPort(httpPort);
        this.addReplacement("uddi.properties", ":9000/", ":" + httpPort + "/");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpMaxThreads(final String httpMaxThreads) {
        Jonas4.logger.finest("[Web configuration] setting max http threads : " + httpMaxThreads);
        NotApplicableHelper.notApplicable("HTTP.maxThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpMinSpareThreads(final String httpMinSpareThreads) {
        Jonas4.logger.finest("[Web configuration] setting min http spare threads : " + httpMinSpareThreads);
        NotApplicableHelper.notApplicable("HTTP.minSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpMaxSpareThreads(final String httpMaxSpareThreads) {
        Jonas4.logger.finest("[Web configuration] setting max http spare threads : " + httpMaxSpareThreads);
        NotApplicableHelper.notApplicable("HTTP.maxSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsPort(final String httpsPort) {
        Jonas4.logger.finest("[Web configuration] setting https port : " + httpsPort);
        this.https.setPort(httpsPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsMaxThreads(final String httpsMaxThreads) {
        Jonas4.logger.finest("[Web configuration] setting max https threads : " + httpsMaxThreads);
        NotApplicableHelper.notApplicable("HTTPS.maxThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsMinSpareThreads(final String httpsMinSpareThreads) {
        Jonas4.logger.finest("[Web configuration] setting min https spare threads : " + httpsMinSpareThreads);
        NotApplicableHelper.notApplicable("HTTPS.minSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsMaxSpareThreads(final String httpsMaxSpareThreads) {
        Jonas4.logger.finest("[Web configuration] setting max https spare threads : " + httpsMaxSpareThreads);
        NotApplicableHelper.notApplicable("HTTPS.maxSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpPort(final String ajpPort) {
        Jonas4.logger.finest("[Web configuration] setting ajp port : " + ajpPort);
        this.ajp.setPort(ajpPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpMaxThreads(final String ajpMaxThreads) {
        Jonas4.logger.finest("[Web configuration] setting max ajp threads : " + ajpMaxThreads);
        NotApplicableHelper.notApplicable("AJP.maxThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpMinSpareThreads(final String ajpMinSpareThreads) {
        Jonas4.logger.finest("[Web configuration] setting min ajp spare threads : " + ajpMinSpareThreads);
        NotApplicableHelper.notApplicable("AJP.minSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpMaxSpareThreads(final String ajpMaxSpareThreads) {
        Jonas4.logger.finest("[Web configuration] setting max ajp spare threads : " + ajpMaxSpareThreads);
        NotApplicableHelper.notApplicable("AJP.maxSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setJvmRoute(final String jvmRoute) {
        Jonas4.logger.finest("[Web configuration] setting jvm route : " + jvmRoute);
        try {
            this.tomcat.setJvmRoute(jvmRoute);
        } catch (NoSuchMethodError e) {
            this.addReplacement("server.xml", "<Engine name=\"jonas\" defaultHost=\"localhost\">",
                "<Engine name=\"jonas\" defaultHost=\"localhost\" jvmRoute=\"" + jvmRoute + "\">");
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationClusterName(final String clusterName) {
        Jonas4.logger.finest("[Web configuration] setting cluster name : " + clusterName);
        this.cluster.setName(clusterName);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationMulticastAddress(final String mCastAddr) {
        Jonas4.logger.finest("[Web configuration] setting multi cast address : " + mCastAddr);
        this.cluster.setMcastAddr(mCastAddr);

    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationMulticastPort(final String mCastPort) {
        Jonas4.logger.finest("[Web configuration] setting multi cast port : " + mCastPort);
        this.cluster.setMcastPort(mCastPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationListenPort(final String listenPort) {
        Jonas4.logger.finest("[Web configuration] setting listen port : " + listenPort);
        this.cluster.setListenPort(listenPort);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service db configuration */
    /**
     * {@inheritDoc}
     */
    public void setDbPort(final String port) {
        Jonas4.logger.finest("[Database configuration] setting db port: " + port);
        this.isDbActivated = true;
        this.db.setPort(port);
        this.addReplacement("HSQL1.properties", ":9001/", ":" + port + "/");
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service discovery configuration */
    /**
     * {@inheritDoc}
     */
    public void setDiscoveryMasterActivated(final Boolean masterActivated) {
        Jonas4.logger.finest("[Discovery configuration] setting masterActivated : " + masterActivated);
        this.isDiscoveryMaster = masterActivated;
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoverySourcePort(final String sourcePort) {
        Jonas4.logger.finest("[Discovery configuration] setting source port : " + sourcePort);
        this.discovery.setSourcePort(sourcePort);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryDomainName(final String domainName) {
        Jonas4.logger.finest("[Discovery configuration] setting domain name : " + domainName);

        if (!this.isJonasRootSet) {
            Jonas4.logger.severe("[Discovery configuration] As jonas root isn't set, domain name skipped : ");
        } else {
            this.discovery.setDomainName(domainName);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryGreetingPort(final String greetingPort) {
        Jonas4.logger.finest("[Discovery configuration] setting greeting port : " + greetingPort);
        this.discovery.setGreetingPort(greetingPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryMulticastAddress(final String multicastAddress) {
        Jonas4.logger.finest("[Discovery configuration] setting multicast address : " + multicastAddress);
        this.discovery.setMcastAddr(multicastAddress);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryMulticastPort(final String multicastPort) {
        Jonas4.logger.finest("[Discovery configuration] setting multicast port : " + multicastPort);
        this.discovery.setMcastPort(multicastPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryTTL(final String ttl) {
        Jonas4.logger.finest("[Discovery configuration] setting tlt : " + ttl);
        NotApplicableHelper.notApplicable("Discovery.TTL");
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service mail configuration */
    /**
     * {@inheritDoc}
     */
    public void setMailFactoryType(final String factoryType) {
        Jonas4.logger.finest("[Mail Service Configuration] setting mail factory type : " + factoryType);
        this.mail.setType(factoryType);
        this.isMailActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setMailFactoryName(final String factoryName) {
        Jonas4.logger.finest("[Mail Service Configuration] setting mail factory name : " + factoryName);
        this.mail.setName(factoryName);
        this.isMailActivated = true;
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service WS configuration */
    /**
     * {@inheritDoc}
     */
    public void setWsdlPublisherFileName(final String fileName) {
        Jonas4.logger.finest("[WS Service configuration] setting wsdlPublisherFile name : " + fileName);
        this.wsdlPublisherFile.setName(fileName);
        this.isWsdlPublishActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setWsdlPublisherFileDirectory(final String fileDir) {
        Jonas4.logger.finest("[WS Service configuration] setting wsdlPublisherFile directory : " + fileDir);
        this.wsdlPublisherFile.setDir(fileDir);
        this.isWsdlPublishActivated = true;
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service HA configuration */
    /**
     * {@inheritDoc}
     */
    public void setHaActivated(final Boolean activated) {
        Jonas4.logger.finest("[HA Service configuration] setting ha activated : " + activated);
        this.isHaActivated = activated;
    }

    /**
     * {@inheritDoc}
     */
    public void setHaMulticastAddress(final String multicastAddr) {
        Jonas4.logger.finest("[HA Service configuration] setting multicast address : " + multicastAddr);
        this.ha.setMcastAddr(multicastAddr);
    }

    /**
     * {@inheritDoc}
     */
    public void setHaMulticastPort(final String multicastPort) {
        Jonas4.logger.finest("[HA Service configuration] setting multicast port : " + multicastPort);
        this.ha.setMcastPort(multicastPort);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Security manager configuration */
    /**
     * {@inheritDoc}
     */
    public void setSecurityManagerActivated(final Boolean activation) {
        Jonas4.logger.finest("[Security Manager configuration] setting isActivated : " + activation);
        if (this.jonasProperties != null) {
            this.jonasProperties.setSecurityManager(activation);
        } else {
            this.addReplacement("jonas.properties", "jonas.security.manager    true", "jonas.security.manager    "
                + activation.toString());
        }
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Ejb clustering configuration */
    /**
     * {@inheritDoc}
     */
    public void setEjbClusteringActivated(final Boolean activated) {
        Jonas4.logger.finest("[Ejb Clustering configuration] setting ejb clustering activated activated : " + activated);
        this.isEjbClusteringActivated = activated;
    }

    /**
     * {@inheritDoc}
     */
    public void setEjbClusteringMulticastAddress(final String multicastAddr) {
        Jonas4.logger.finest("[Ejb Clustering configuration] setting multicast address : " + multicastAddr);
        this.saveCmiMCastAddr = multicastAddr;
    }

    /**
     * {@inheritDoc}
     */
    public void setEjbClusteringMulticastPort(final String multicastPort) {
        Jonas4.logger.finest("[Ejb Clustering configuration] setting multicast port : " + multicastPort);
        this.saveCmiMCastPort = multicastPort;
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Jms configuration */
    /**
     * {@inheritDoc}
     */
    public void setJmsPort(final String port) {
        Jonas4.logger.finest("[Jms configuration] setting port : " + port);
        this.jms.setPort(port);
    }

    /**
     * {@inheritDoc}
     */
    public void setJmsQueues(final String queue) {
        Jonas4.logger.finest("[Jms configuration] setting queue : " + queue);
        this.jms.setInitialQueues(queue);
    }

    /**
     * {@inheritDoc}
     */
    public void setJmsTopics(final String topic) {
        Jonas4.logger.finest("[Jms configuration] setting topic : " + topic);
        this.jms.setInitialTopics(topic);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Jdbc configuration */
    /**
     * {@inheritDoc}
     */
    public void setJdbcDriverName(final String driver) {
        Jonas4.logger.finest("[Jdbc configuration] setting drivers: " + driver);
        this.jdbcRa.setDriverName(driver);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcDatasourceClass(final String dsClass) {
        Jonas4.logger.finest("[Jdbc configuration] setting ds class : " + dsClass);
        NotApplicableHelper.notApplicable("JDBC.datasourceClass");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcResourceAdapterName(final String raName) {
        Jonas4.logger.finest("[Jdbc configuration] setting RA name : " + raName);
        this.jdbcRa.setName(raName);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcCheckLevel(final String checkLevel) {
        Jonas4.logger.finest("[Jdbc configuration] setting check level: " + checkLevel);
        NotApplicableHelper.notApplicable("JDBC.checkLevel");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcTestStatement(final String testStatement) {
        Jonas4.logger.finest("[Jdbc configuration] setting test statement: " + testStatement);
        NotApplicableHelper.notApplicable("JDBC.testStatement");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcJndiName(final String jndiname) {
        Jonas4.logger.finest("[Jdbc configuration] setting jndi name: " + jndiname);
        this.jdbcRa.setJndiName(jndiname);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcMappername(final String mappername) {
        Jonas4.logger.finest("[Jdbc configuration] setting mappername: " + mappername);
        this.jdbcRa.setMapperName(mappername);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPassword(final String password) {
        Jonas4.logger.finest("[Jdbc configuration] setting password: " + password);
        this.jdbcRa.setPassword(password);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcRarlink(final String rarLink) {
        Jonas4.logger.finest("[Jdbc configuration] setting rar link: " + rarLink);
        NotApplicableHelper.notApplicable("JDBC.rarLink");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcUrl(final String url) {
        Jonas4.logger.finest("[Jdbc configuration] setting url: " + url);
        this.jdbcRa.setUrl(url);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcUser(final String user) {
        Jonas4.logger.finest("[Jdbc configuration] setting user: " + user);
        this.jdbcRa.setUser(user);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolInitialSize(final String poolInit) {
        Jonas4.logger.finest("[Jdbc configuration] setting pool init : " + poolInit);
        NotApplicableHelper.notApplicable("JDBC.poolInitialSize");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumSize(final String poolMax) {
        Jonas4.logger.finest("[Jdbc configuration] setting pool max: " + poolMax);
        this.jdbcRa.setMaxPoolSize(poolMax);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumAgeMinutes(final String poolMaxAgeMinutes) {
        Jonas4.logger.finest("[Jdbc configuration] setting pool max age minutes: " + poolMaxAgeMinutes);
        try {
            this.jdbcRa.setConnMaxAge(poolMaxAgeMinutes);
        } catch (NoSuchMethodError e) {
            NotApplicableHelper.notApplicable("JDBC.poolMaximumAgeMinutes");
        }
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumOpenTime(final String poolMaxOpenTime) {
        Jonas4.logger.finest("[Jdbc configuration] setting pool max open time: " + poolMaxOpenTime);
        try {
            this.jdbcRa.setMaxOpenTime(poolMaxOpenTime);
        } catch (NoSuchMethodError e) {
            NotApplicableHelper.notApplicable("JDBC.poolMaximumOpenTime");
        }
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumNumberOfWaiters(final String poolMaxWaiters) {
        Jonas4.logger.finest("[Jdbc configuration] setting max waiters: " + poolMaxWaiters);
        try {
            this.jdbcRa.setMaxWaiters(poolMaxWaiters);
        } catch (NoSuchMethodError e) {
            NotApplicableHelper.notApplicable("JDBC.poolMaximumNumberOfWaiters");
        }
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumWaitTime(final String poolMaxWaittime) {
        Jonas4.logger.finest("[Jdbc configuration] setting pool max wait time: " + poolMaxWaittime);
        try {
            this.jdbcRa.setMaxWaitTime(poolMaxWaittime);
        } catch (NoSuchMethodError e) {
            NotApplicableHelper.notApplicable("JDBC.poolMaximumWaitTime");
        }
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMinimumSize(final String poolMin) {
        Jonas4.logger.finest("[Jdbc configuration] setting pool min: " + poolMin);
        NotApplicableHelper.notApplicable("JDBC.poolMinimumSize");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolSamplingPeriod(final String poolSamplingPeriod) {
        Jonas4.logger.finest("[Jdbc configuration] setting pool sampling period : " + poolSamplingPeriod);
        NotApplicableHelper.notApplicable("JDBC.poolSamplingPeriod");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumNumberOfPreparedStatements(final String pstmtMax) {
        Jonas4.logger.finest("[Jdbc configuration] setting max prepared statements: " + pstmtMax);
        try {
            this.jdbcRa.setMaxPreparedStatementsSize(pstmtMax);
        } catch (NoSuchMethodError e) {
            NotApplicableHelper.notApplicable("JDBC.poolMaximumNumberOfPreparedStatements");
        }
        this.isJdbcRaActivated = true;
    }

    /* ------------------------------- */

    /**
     * {@inheritDoc}
     */
    public void execute() {
        if (!this.isJonasRootSet) {
            throw new IllegalArgumentException("Jonas root is not set.");
        }
        if (!this.isJonasBaseSet) {
            throw new IllegalArgumentException("Jonas base is not set.");
        }

        if (this.isHttpActivated) {
            this.tomcat.addConfiguredHttp(this.http);
        }
        if (this.isHttpsActivated) {
            this.tomcat.addConfiguredHttps(this.https);
        }
        if (this.isAjpActivated) {
            this.tomcat.addConfiguredAjp(this.ajp);
        }
        if (this.isHttpReplicationActivated) {
            this.tomcat.addConfiguredCluster(this.cluster);
        }

        this.webContainer.addConfiguredTomcat(this.tomcat);

        if (this.isEjbClusteringActivated) {
            this.carol.setCmiMcastAddr(this.saveCmiMCastAddr);
            this.carol.setCmiMcastPort(this.saveCmiMCastPort);
        }
        if (!this.isProtocolsListSet) {
            this.carol.setProtocols("jrmp");
        }
        this.jonasBase.addConfiguredCarol(this.carol);
        if (this.isDbActivated) {
            this.jonasBase.addConfiguredDb(this.db);
        }
        if (this.isDiscoveryMaster) {
            this.discovery.setSourcePort("-1");
        }
        this.jonasBase.addConfiguredDiscovery(this.discovery);
        if (this.isJdbcRaActivated) {
            this.jonasBase.addConfiguredJdbcRa(this.jdbcRa);
        }
        this.jonasBase.addConfiguredJms(this.jms);
        if (this.jonasProperties != null) {
            this.jonasBase.addConfiguredJonasProperties(this.jonasProperties);
        } else {
            this.jonasBase.addConfiguredServices(this.services);
        }
        if (this.isMailActivated) {
            this.jonasBase.addConfiguredMail(this.mail);
        }
        this.jonasBase.addConfiguredWebContainer(this.webContainer);
        if (this.isWsdlPublishActivated) {
            this.wsdlPublish.addConfiguredFile(this.wsdlPublisherFile);
            this.jonasBase.addConfiguredWsdlPublish(this.wsdlPublish);
        }
        if (this.isHaActivated) {
            this.jonasBase.addTasks(this.ha);
        }

        this.jonasBase.execute();
        this.doReplacements(new File(this.jonasFileBase, "conf"));
    }

    /* --------------------------------- */
    /* Cluster daemon configuration part */

    /**
     * {@inheritDoc}
     */
    public void setCDClusterName(final String clusterName) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting cluster name : " + clusterName);
        this.clusterDaemon.setName(clusterName);
        this.clusterDaemon.updateName();
    }

    /**
     * {@inheritDoc}
     */
    public void setCDClusterDomain(final String clusterDomain) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting cluster domain name : " + clusterDomain);
        this.clusterDaemon.setDomainName(clusterDomain);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDProtocol(final String protocol) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting protocol : " + protocol);
        this.clusterDaemon.setProtocol(protocol);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDPort(final String port) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting port : " + port);
        this.clusterDaemon.setPort(port);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDDestDirPrefix(final String destDirPrefix) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting dest dir prefix : " + destDirPrefix);
        this.clusterDaemon.setDestDirPrefix(destDirPrefix);
        this.clusterDaemonTask.setDestDirPrefix(destDirPrefix);
        this.clusterDaemon.setCdDir(destDirPrefix);
        this.clusterDaemonTask.setCdDir(destDirPrefix);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDInteractionMode(final String interactionMode) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting interaction mode : " + interactionMode);
        this.clusterDaemon.setInteractionMode(interactionMode);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDJavaHome(final String javaHome) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting java home : " + javaHome);
        this.clusterDaemon.setJdk(javaHome);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDJonasRoot(final String jonasRoot) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting jonas root : " + jonasRoot);
        this.clusterDaemon.setJonasRoot(jonasRoot);
        this.clusterDaemonTask.setJonasRoot(new File(jonasRoot));
    }

    /**
     * {@inheritDoc}
     */
    public void setCDServerNamePrefix(final String serverNamePrefix) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting server name prefix : " + serverNamePrefix);
        this.clusterDaemon.setClusterNodesName(serverNamePrefix);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDXparam(final String xparam) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting xparam : " + xparam);
        this.clusterDaemon.setXprm(xparam);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDAutoBoot(final boolean autoBoot) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting auto boot : " + autoBoot);
        this.clusterDaemon.setAutoBoot(new Boolean(autoBoot).toString());
    }

    /**
     * {@inheritDoc}
     */
    public void setCDNbInstances(final int nbInsts) {
        Jonas4.logger.finest("[Cluster daemon configuration] setting nb instance : " + nbInsts);
        this.clusterDaemon.setInstNb(nbInsts);
        this.clusterDaemonTask.setWebInstNb(nbInsts);
        this.clusterDaemonTask.setEjbInstNb(0);
    }

    /**
     * {@inheritDoc}
     */
    public void executeCDConf() {
        this.clusterDaemonTask.addConfiguredClusterDaemon(this.clusterDaemon);
        this.clusterDaemonTask.execute();
    }
    /* --------------------------------- */
}
