/**
 * JOnAS Configurator
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Jonas5.java 18711 2009-09-22 12:51:31Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.tools.configurator.impl;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.logging.Logger;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.listener.TimestampedLogger;
import org.ow2.jonas.ant.JOnASBaseTask;
import org.ow2.jonas.ant.JOnASClusterConfigTask;
import org.ow2.jonas.ant.cluster.ClusterDaemon;
import org.ow2.jonas.ant.jonasbase.Carol;
import org.ow2.jonas.ant.jonasbase.Cmi;
import org.ow2.jonas.ant.jonasbase.Db;
import org.ow2.jonas.ant.jonasbase.Discovery;
import org.ow2.jonas.ant.jonasbase.Ha;
import org.ow2.jonas.ant.jonasbase.JdbcRa;
import org.ow2.jonas.ant.jonasbase.Jms;
import org.ow2.jonas.ant.jonasbase.JonasProperties;
import org.ow2.jonas.ant.jonasbase.Mail;
import org.ow2.jonas.ant.jonasbase.Services;
import org.ow2.jonas.ant.jonasbase.WebContainer;
import org.ow2.jonas.ant.jonasbase.web.Ajp;
import org.ow2.jonas.ant.jonasbase.web.Cluster;
import org.ow2.jonas.ant.jonasbase.web.Http;
import org.ow2.jonas.ant.jonasbase.web.Https;
import org.ow2.jonas.ant.jonasbase.web.Tomcat;
import org.ow2.jonas.ant.jonasbase.wsdl.WsdlPublish;
import org.ow2.jonas.tools.configurator.api.JonasConfigurator;

/**
 * JonasConfigurator implementation for Jonas 5.x
 * 
 * @author Remy Bresson
 */
public class Jonas5 extends FileReplacerHelper implements JonasConfigurator {

    /**
     * JONAS_BASE
     */
    private File jonasFileBase;

    /**
     * Link to the jonas ant task
     */
    private JOnASBaseTask jonasBase;

    /**
     * The cluster daemon configuration task
     */
    private JOnASClusterConfigTask clusterDaemonTask;

    /**
     * The cluster daemon configuration
     */
    private ClusterDaemon clusterDaemon;

    /**
     * Configuration logger
     */
    private static Logger logger = Logger.getLogger(Jonas5.class.getName());

    /**
     * To configure protocols
     */
    private Carol carol;

    /**
     * Has JOnAS protocols list been set?
     */
    private boolean isProtocolsListSet = false;

    /**
     * To configure databases service
     */
    private Db db;

    /**
     * Whether the optional db service has been activated
     */
    private boolean isDbActivated = false;

    /**
     * To configure discovery sevice
     */
    private Discovery discovery;

    /**
     * To configure jdbc
     */
    private JdbcRa jdbcRa;

    /**
     * Whether the optional jdbc adapter has been activated
     */
    private boolean isJdbcRaActivated = false;

    /**
     * To configure jms
     */
    private Jms jms;

    /**
     * To configure jonas global properties
     */
    private JonasProperties jonasProperties;

    /**
     * To configure mail service
     */
    private Mail mail;

    /**
     * Whether the optional mail service has been activated
     */
    private boolean isMailActivated = false;

    /**
     * To configure active services
     */
    private Services services;

    /**
     * To configure web container
     */
    private WebContainer webContainer;

    /**
     * To configure the tomcat (in the web container)
     */
    private Tomcat tomcat;

    /**
     * To configure ajp
     */
    private Ajp ajp;

    /**
     * To configure http
     */
    private Http http;

    /**
     * To configure https
     */
    private Https https;

    /**
     * To configure cluster
     */
    private Cluster cluster;

    /**
     * To configure wsdl wsdlPublisherFile
     */
    private org.ow2.jonas.ant.jonasbase.wsdl.File wsdlPublisherFile;

    /**
     * To configure WSDL publishing
     */
    private WsdlPublish wsdlPublish;

    /**
     * Whether the optional WSDL publishing service has been activated
     */
    private boolean isWsdlPublishActivated = false;

    /**
     * To configure ejb clustering
     */
    private Cmi cmi;

    /**
     * To configure ha cluster
     */
    private Ha ha;

    /**
     * The project, not really usefull, only to have a project to pass to ant
     * task.
     */
    private Project project;

    /**
     * Is jonas root set
     */
    private boolean isJonasRootSet = false;

    /**
     * Is jonas base set
     */
    private boolean isJonasBaseSet = false;

    /**
     * Is http webservice activated
     */
    private Boolean isHttpActivated = true;

    /**
     * Is https webservice activated
     */
    private Boolean isHttpsActivated = false;

    /**
     * Is ajp webservice activated
     */
    private Boolean isAjpActivated = false;

    /**
     * Is http replication webservice activated
     */
    private Boolean isHttpReplicationActivated = false;

    /**
     * Is ha activated
     */
    private Boolean isHaActivated = false;

    /**
     * Is ejb clustering activated
     */
    private Boolean isEjbClusteringActivated = false;

    /**
     * Derfault constructor
     */
    public Jonas5() {
        this.jonasBase = new JOnASBaseTask();
        this.createProject();
        this.jonasBase.setProject(this.project);

        this.clusterDaemonTask = new JOnASClusterConfigTask();
        this.clusterDaemonTask.setProject(this.project);

        this.carol = new Carol();

        this.db = new Db();
        this.discovery = new Discovery();
        this.jdbcRa = new JdbcRa();
        this.jms = new Jms();

        this.jonasProperties = new JonasProperties();

        this.mail = new Mail();
        this.services = new Services();
        this.webContainer = new WebContainer();
        this.tomcat = new Tomcat();
        this.ajp = new Ajp();
        this.http = new Http();
        this.https = new Https();
        this.cluster = new Cluster();
        this.wsdlPublish = new WsdlPublish();
        this.wsdlPublisherFile = new org.ow2.jonas.ant.jonasbase.wsdl.File();
        this.cmi = new Cmi();
        this.ha = new Ha();

        this.clusterDaemon = new ClusterDaemon();
        this.clusterDaemon.setProtocol("jrmp");
        this.clusterDaemon.setPort("1099");

        Jonas5.logger.finest("Created the JOnAS 5 configurator instance");
    }

    /**
     * Create the project
     */
    private void createProject() {
        this.project = new Project();
        this.project.init();
        this.project.initProperties();

        TimestampedLogger logger = new TimestampedLogger();
        logger.setMessageOutputLevel(Project.MSG_DEBUG);

        logger.setOutputPrintStream(new PrintStream(new OutputStream() {
            StringBuffer buf = new StringBuffer();

            @Override
            public void write(final int b) throws IOException {
                if (b == '\n' || b == '\r') {
                    if (this.buf.length() > 0) {
                        Jonas5.logger.finest(this.buf.toString());
                        this.buf.setLength(0);
                    }
                } else {
                    this.buf.append((char) b);
                }
            }
        }));

        this.project.addBuildListener(logger);
    }

    /* ------------------------------- */
    /* Jonas global parameters */
    /**
     * {@inheritDoc}
     */
    public void setJdk(final String jdk) {
        Jonas5.logger.finest("[GlobalParameters] setting jdk : " + jdk);
        NotApplicableHelper.notApplicable("GlobalParameters.Jdk");
    }

    /**
     * {@inheritDoc}
     */
    public void setJavaOpts(final String javaOpts) {
        Jonas5.logger.finest("[GlobalParameters] setting java opts : " + javaOpts);
        this.carol.setJvmopts(javaOpts);
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasRoot(final String jonasRoot) {
        Jonas5.logger.finest("[GlobalParameters] setting jonas root : " + jonasRoot);
        File jonasFileRoot = new File(jonasRoot);
        this.jonasBase.setJonasRoot(jonasFileRoot);
        this.discovery.setJonasRoot(jonasRoot);

        this.isJonasRootSet = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasBase(final String jonasBase) {
        Jonas5.logger.finest("[GlobalParameters] setting jonas base : " + jonasBase);
        this.jonasFileBase = new File(jonasBase);
        this.jonasBase.setDestDir(this.jonasFileBase);
        this.project.setProperty("jonas.base", jonasBase);

        this.isJonasBaseSet = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasName(final String jonasName) {
        Jonas5.logger.finest("[GlobalParameters] setting jonas name : " + jonasName);
        this.carol.setServerName(jonasName);
        this.addReplacement("jonas.properties", "jonas.name    jonas", "jonas.name    " + jonasName);
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasDomain(final String jonasDomain) {
        Jonas5.logger.finest("[GlobalParameters] setting jonas domain : " + jonasDomain);
        this.carol.setDomainName(jonasDomain);
        this.addReplacement("jonas.properties", "domain.name    jonas", "domain.name    " + jonasDomain);
    }

    /**
     * {@inheritDoc}
     */
    public void setHost(final String jonasHost) {
        Jonas5.logger.finest("[GlobalParameters] setting host : " + jonasHost);
        this.carol.setHost(jonasHost);
        try {
            this.jms.setHost(jonasHost);
        } catch (NoSuchMethodError e) {
            NotApplicableHelper.notApplicable("GlobalParameters.JMSHost");
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasDevelopment(final boolean development) {
        logger.finest("[GlobalParameters] setting jonas.development : " + development);
        this.addReplacement("jonas.properties", "jonas.development    true", "jonas.development    "
            + Boolean.toString(development));
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Jonas protocols configuation */
    /**
     * {@inheritDoc}
     */
    public void setProtocolsList(final String protocolsList) {
        Jonas5.logger.finest("[Protocols configuration] setting protocols list : " + protocolsList);
        this.carol.setProtocols(protocolsList);
        this.isProtocolsListSet = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsIrmiPort(final String irmiPort) {
        Jonas5.logger.finest("[Protocols configuration] setting irmi port : " + irmiPort);
        this.carol.setIrmiPort(irmiPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsIiopPort(final String iiopPort) {
        Jonas5.logger.finest("[Protocols configuration] setting iiop port : " + iiopPort);
        this.carol.setIiopPort(iiopPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsJrmpPort(final String jrmpPort) {
        Jonas5.logger.finest("[Protocols configuration] setting jrmp port : " + jrmpPort);
        this.carol.setJrmpPort(jrmpPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsCmiPort(final String cmiPort) {
        Jonas5.logger.finest("[Protocols configuration] setting cmi port : " + cmiPort);
        NotApplicableHelper.notApplicable("Protocols.CmiPort");
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsLocalCallOptimized(final Boolean localCallOptimized) {
        Jonas5.logger.finest("[Protocols configuration] setting jndi local call optimisation : " + localCallOptimized);
        this.carol.setJrmpOptimization(localCallOptimized);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Active services */
    /**
     * {@inheritDoc}
     */
    public void setServices(final String services) {
        Jonas5.logger.finest("setting actives services list : " + services);
        this.services.setNames(services);
        this.jonasProperties.setServices(services);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service web configuration */
    /**
     * {@inheritDoc}
     */
    public void setHttpsConnectorActivation(final Boolean activation) {
        Jonas5.logger.finest("[Web configuration] setting isHttpsActivated : " + activation);
        this.isHttpsActivated = activation;
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpConnectorActivation(final Boolean activation) {
        Jonas5.logger.finest("[Web configuration] setting isAjpActivated : " + activation);
        this.isAjpActivated = activation;
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpSessionReplicationActivation(final Boolean activation) {
        Jonas5.logger.finest("[Web configuration] setting isHttpReplicationActivated : " + activation);
        this.isHttpReplicationActivated = activation;
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpPort(final String httpPort) {
        Jonas5.logger.finest("[Web configuration] setting http port : " + httpPort);
        this.http.setPort(httpPort);
        this.addReplacement("uddi.properties", ":9000/", ":" + httpPort + "/");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpMaxThreads(final String httpMaxThreads) {
        Jonas5.logger.finest("[Web configuration] setting max http threads : " + httpMaxThreads);
        NotApplicableHelper.notApplicable("HTTP.maxThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpMinSpareThreads(final String httpMinSpareThreads) {
        Jonas5.logger.finest("[Web configuration] setting min http spare threads : " + httpMinSpareThreads);
        NotApplicableHelper.notApplicable("HTTP.minSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpMaxSpareThreads(final String httpMaxSpareThreads) {
        Jonas5.logger.finest("[Web configuration] setting max http spare threads : " + httpMaxSpareThreads);
        NotApplicableHelper.notApplicable("HTTP.maxSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsPort(final String httpsPort) {
        Jonas5.logger.finest("[Web configuration] setting https port : " + httpsPort);
        this.https.setPort(httpsPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsMaxThreads(final String httpsMaxThreads) {
        Jonas5.logger.finest("[Web configuration] setting max https threads : " + httpsMaxThreads);
        NotApplicableHelper.notApplicable("HTTPS.maxThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsMinSpareThreads(final String httpsMinSpareThreads) {
        Jonas5.logger.finest("setting min https spare threads : " + httpsMinSpareThreads);
        NotApplicableHelper.notApplicable("HTTPS.minSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsMaxSpareThreads(final String httpsMaxSpareThreads) {
        Jonas5.logger.finest("setting max https spare threads : " + httpsMaxSpareThreads);
        NotApplicableHelper.notApplicable("HTTPS.maxSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpPort(final String ajpPort) {
        Jonas5.logger.finest("[Web configuration] setting ajp port : " + ajpPort);
        this.ajp.setPort(ajpPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpMaxThreads(final String ajpMaxThreads) {
        Jonas5.logger.finest("[Web configuration] setting max ajp threads : " + ajpMaxThreads);
        NotApplicableHelper.notApplicable("AJP.maxThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpMinSpareThreads(final String ajpMinSpareThreads) {
        Jonas5.logger.finest("[Web configuration] setting min ajp spare threads : " + ajpMinSpareThreads);
        NotApplicableHelper.notApplicable("AJP.minSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpMaxSpareThreads(final String ajpMaxSpareThreads) {
        Jonas5.logger.finest("[Web configuration] setting max ajp spare threads : " + ajpMaxSpareThreads);
        NotApplicableHelper.notApplicable("AJP.maxSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setJvmRoute(final String jvmRoute) {
        Jonas5.logger.finest("[Web configuration] setting jvm route : " + jvmRoute);
        this.tomcat.setJvmRoute(jvmRoute);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationClusterName(final String clusterName) {
        Jonas5.logger.finest("[Web configuration] setting cluster name : " + clusterName);
        this.cluster.setName(clusterName);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationMulticastAddress(final String mCastAddr) {
        Jonas5.logger.finest("[Web configuration] setting multi cast address : " + mCastAddr);
        this.cluster.setMcastAddr(mCastAddr);

    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationMulticastPort(final String mCastPort) {
        Jonas5.logger.finest("[Web configuration] setting multi cast port : " + mCastPort);
        this.cluster.setMcastPort(mCastPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationListenPort(final String listenPort) {
        Jonas5.logger.finest("[Web configuration] setting listen port : " + listenPort);
        this.cluster.setListenPort(listenPort);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service db configuration */
    /**
     * {@inheritDoc}
     */
    public void setDbPort(final String port) {
        Jonas5.logger.finest("[Database configuration] setting db port: " + port);
        this.isDbActivated = true;
        this.db.setPort(port);
        this.addReplacement("HSQL1.properties", ":9001/", ":" + port + "/");
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service discovery configuration */
    /**
     * {@inheritDoc}
     */
    public void setDiscoveryMasterActivated(final Boolean masterActivated) {
        Jonas5.logger.finest("[Discovery configuration] setting masterActivated : " + masterActivated);
        this.jonasProperties.setMaster(masterActivated);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoverySourcePort(final String sourcePort) {
        Jonas5.logger.finest("[Discovery configuration] setting source port : " + sourcePort);
        this.discovery.setSourcePort(sourcePort);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryDomainName(final String domainName) {
        Jonas5.logger.finest("[Discovery configuration] setting domain name : " + domainName);
        if (!this.isJonasRootSet) {
            Jonas5.logger.severe("[Discovery configuration] As jonas root isn't set, domain name skipped : ");
        } else {
            // JOnAS 5.1.0+ uses the domain.name from jonas.properties
            // see:
            // http://forge.ow2.org/tracker/index.php?func=detail&aid=313970&group_id=5&atid=100005
            this.addReplacement("jonas.properties", "domain.name    jonas", "domain.name    " + domainName);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryGreetingPort(final String greetingPort) {
        Jonas5.logger.finest("[Discovery configuration] setting greeting port : " + greetingPort);
        this.discovery.setGreetingPort(greetingPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryMulticastAddress(final String multicastAddress) {
        Jonas5.logger.finest("[Discovery configuration] setting multicast address : " + multicastAddress);
        this.discovery.setMcastAddr(multicastAddress);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryMulticastPort(final String multicastPort) {
        Jonas5.logger.finest("[Discovery configuration] setting multicast port : " + multicastPort);
        this.discovery.setMcastPort(multicastPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryTTL(final String ttl) {
        Jonas5.logger.finest("[Discovery configuration] setting tlt : " + ttl);
        this.discovery.setTtl(ttl);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service mail configuration */
    /**
     * {@inheritDoc}
     */
    public void setMailFactoryType(final String factoryType) {
        Jonas5.logger.finest("[Mail Service Configuration] setting mail factory type : " + factoryType);
        this.mail.setType(factoryType);
        this.isMailActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setMailFactoryName(final String factoryName) {
        Jonas5.logger.finest("[Mail Service Configuration] setting mail factory name : " + factoryName);
        this.mail.setName(factoryName);
        this.isMailActivated = true;
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service WS configuration */
    /**
     * {@inheritDoc}
     */
    public void setWsdlPublisherFileName(final String fileName) {
        Jonas5.logger.finest("[WS Service configuration] setting wsdlPublisherFile name : " + fileName);
        this.wsdlPublisherFile.setName(fileName);
        this.isWsdlPublishActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setWsdlPublisherFileDirectory(final String fileDir) {
        Jonas5.logger.finest("[WS Service configuration] setting wsdlPublisherFile directory : " + fileDir);
        this.wsdlPublisherFile.setDir(fileDir);
        this.isWsdlPublishActivated = true;
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service HA configuration */
    /**
     * {@inheritDoc}
     */
    public void setHaActivated(final Boolean activated) {
        Jonas5.logger.finest("[HA Service configuration] setting ha activated : " + activated);
        this.isHaActivated = activated;
    }

    /**
     * {@inheritDoc}
     */
    public void setHaMulticastAddress(final String multicastAddr) {
        Jonas5.logger.finest("[HA Service configuration] setting multicast address : " + multicastAddr);
        this.ha.setMcastAddr(multicastAddr);
    }

    /**
     * {@inheritDoc}
     */
    public void setHaMulticastPort(final String multicastPort) {
        Jonas5.logger.finest("[HA Service configuration] setting multicast port : " + multicastPort);
        this.ha.setMcastPort(multicastPort);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Security manager configuration */
    /**
     * {@inheritDoc}
     */
    public void setSecurityManagerActivated(final Boolean activation) {
        Jonas5.logger.finest("[Security Manager configuration] setting isActivates : " + activation);
        this.jonasProperties.setSecurityManager(activation);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Ejb clustering configuration */
    /**
     * {@inheritDoc}
     */
    /**
     * {@inheritDoc}
     */
    public void setEjbClusteringActivated(final Boolean activated) {
        Jonas5.logger.finest("[Ejb Clustering configuration] setting ejb clustering activated activated : " + activated);
        this.isEjbClusteringActivated = activated;
    }

    public void setEjbClusteringMulticastAddress(final String multicastAddr) {
        Jonas5.logger.finest("[Ejb Clustering configuration] setting multicast address : " + multicastAddr);
        this.cmi.setMcastAddr(multicastAddr);
    }

    /**
     * {@inheritDoc}
     */
    public void setEjbClusteringMulticastPort(final String multicastPort) {
        Jonas5.logger.finest("[Ejb Clustering configuration] setting multicast port : " + multicastPort);
        this.cmi.setMcastPort(multicastPort);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Jms configuration */
    /**
     * {@inheritDoc}
     */
    public void setJmsPort(final String port) {
        Jonas5.logger.finest("[Jms configuration] setting port : " + port);
        this.jms.setPort(port);
    }

    /**
     * {@inheritDoc}
     */
    public void setJmsQueues(final String queue) {
        Jonas5.logger.finest("[Jms configuration] setting queue : " + queue);
        this.jms.setInitialQueues(queue);
    }

    /**
     * {@inheritDoc}
     */
    public void setJmsTopics(final String topic) {
        Jonas5.logger.finest("[Jms configuration] setting topic : " + topic);
        this.jms.setInitialTopics(topic);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Jdbc configuration */
    /**
     * {@inheritDoc}
     */
    public void setJdbcDriverName(final String driver) {
        Jonas5.logger.finest("[Jdbc configuration] setting drivers: " + driver);
        this.jdbcRa.setDriverName(driver);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcDatasourceClass(final String dsClass) {
        Jonas5.logger.finest("[Jdbc configuration] setting ds class : " + dsClass);
        NotApplicableHelper.notApplicable("JDBC.datasourceClass");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcResourceAdapterName(final String raName) {
        Jonas5.logger.finest("[Jdbc configuration] setting RA name : " + raName);
        this.jdbcRa.setName(raName);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcCheckLevel(final String checkLevel) {
        Jonas5.logger.finest("[Jdbc configuration] setting check level: " + checkLevel);
        NotApplicableHelper.notApplicable("JDBC.checkLevel");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcTestStatement(final String testStatement) {
        Jonas5.logger.finest("[Jdbc configuration] setting test statement: " + testStatement);
        NotApplicableHelper.notApplicable("JDBC.testStatement");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcJndiName(final String jndiname) {
        Jonas5.logger.finest("[Jdbc configuration] setting jndi name: " + jndiname);
        this.jdbcRa.setJndiName(jndiname);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcMappername(final String mappername) {
        Jonas5.logger.finest("[Jdbc configuration] setting mappername: " + mappername);
        this.jdbcRa.setMapperName(mappername);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPassword(final String password) {
        Jonas5.logger.finest("[Jdbc configuration] setting password: " + password);
        this.jdbcRa.setPassword(password);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcRarlink(final String rarLink) {
        Jonas5.logger.finest("[Jdbc configuration] setting rar link: " + rarLink);
        NotApplicableHelper.notApplicable("JDBC.rarLink");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcUrl(final String url) {
        Jonas5.logger.finest("[Jdbc configuration] setting url: " + url);
        this.jdbcRa.setUrl(url);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcUser(final String user) {
        Jonas5.logger.finest("[Jdbc configuration] setting user: " + user);
        this.jdbcRa.setUser(user);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolInitialSize(final String poolInit) {
        Jonas5.logger.finest("[Jdbc configuration] setting pool init : " + poolInit);
        NotApplicableHelper.notApplicable("JDBC.poolInitialSize");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumSize(final String poolMax) {
        Jonas5.logger.finest("[Jdbc configuration] setting pool max: " + poolMax);
        this.jdbcRa.setMaxPoolSize(poolMax);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumAgeMinutes(final String poolMaxAgeMinutes) {
        Jonas5.logger.finest("[Jdbc configuration] setting pool max age minutes: " + poolMaxAgeMinutes);
        this.jdbcRa.setConnMaxAge(poolMaxAgeMinutes);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumOpenTime(final String poolMaxOpenTime) {
        Jonas5.logger.finest("[Jdbc configuration] setting pool max open time: " + poolMaxOpenTime);
        this.jdbcRa.setMaxOpenTime(poolMaxOpenTime);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumNumberOfWaiters(final String poolMaxWaiters) {
        Jonas5.logger.finest("[Jdbc configuration] setting max waiters: " + poolMaxWaiters);
        this.jdbcRa.setMaxWaiters(poolMaxWaiters);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumWaitTime(final String poolMaxWaittime) {
        Jonas5.logger.finest("[Jdbc configuration] setting pool max wait time: " + poolMaxWaittime);
        this.jdbcRa.setMaxWaitTime(poolMaxWaittime);
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMinimumSize(final String poolMin) {
        Jonas5.logger.finest("[Jdbc configuration] setting pool min: " + poolMin);
        NotApplicableHelper.notApplicable("JDBC.poolMinimumSize");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolSamplingPeriod(final String poolSamplingPeriod) {
        Jonas5.logger.finest("[Jdbc configuration] setting pool sampling period : " + poolSamplingPeriod);
        NotApplicableHelper.notApplicable("JDBC.poolSamplingPeriod");
        this.isJdbcRaActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJdbcPoolMaximumNumberOfPreparedStatements(final String pstmtMax) {
        Jonas5.logger.finest("[Jdbc configuration] setting max prepared statements: " + pstmtMax);
        this.jdbcRa.setMaxPreparedStatementsSize(pstmtMax);
        this.isJdbcRaActivated = true;
    }

    /* ------------------------------- */

    /**
     * {@inheritDoc}
     */
    public void execute() {
        if (!this.isJonasRootSet) {
            throw new IllegalArgumentException("Jonas root is not set.");
        }
        if (!this.isJonasBaseSet) {
            throw new IllegalArgumentException("Jonas base is not set.");
        }

        if (this.isHttpActivated) {
            this.tomcat.addConfiguredHttp(this.http);
        }
        if (this.isHttpsActivated) {
            this.tomcat.addConfiguredHttps(this.https);
        }
        if (this.isAjpActivated) {
            this.tomcat.addConfiguredAjp(this.ajp);
        }
        if (this.isHttpReplicationActivated) {
            this.tomcat.addConfiguredCluster(this.cluster);
        }

        this.webContainer.addConfiguredTomcat(this.tomcat);

        if (!this.isProtocolsListSet) {
            this.carol.setProtocols("jrmp");
        }
        this.jonasBase.addConfiguredCarol(this.carol);
        if (this.isDbActivated) {
            this.jonasBase.addConfiguredDb(this.db);
        }
        this.jonasBase.addConfiguredDiscovery(this.discovery);
        if (this.isJdbcRaActivated) {
            this.jonasBase.addConfiguredJdbcRa(this.jdbcRa);
        }
        this.jonasBase.addConfiguredJms(this.jms);
        this.jonasBase.addConfiguredJonasProperties(this.jonasProperties);
        if (this.isMailActivated) {
            this.jonasBase.addConfiguredMail(this.mail);
        }
        this.jonasBase.addConfiguredServices(this.services);
        this.jonasBase.addConfiguredWebContainer(this.webContainer);
        if (this.isWsdlPublishActivated) {
            this.wsdlPublish.addConfiguredFile(this.wsdlPublisherFile);
            this.jonasBase.addConfiguredWsdlPublish(this.wsdlPublish);
        }
        if (this.isHaActivated) {
            this.jonasBase.addTasks(this.ha);
        }
        if (this.isEjbClusteringActivated) {
            this.jonasBase.addTasks(this.cmi);
        }

        this.jonasBase.execute();
        this.doReplacements(new File(this.jonasFileBase, "conf"));
    }

    /* --------------------------------- */
    /* Cluster daemon configuration part */

    /**
     * {@inheritDoc}
     */
    public void setCDClusterName(final String clusterName) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting cluster name : " + clusterName);
        this.clusterDaemon.setName(clusterName);
        this.clusterDaemon.updateName();
    }

    /**
     * {@inheritDoc}
     */
    public void setCDClusterDomain(final String clusterDomain) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting cluster domain name : " + clusterDomain);
        this.clusterDaemon.setDomainName(clusterDomain);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDProtocol(final String protocol) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting protocol : " + protocol);
        this.clusterDaemon.setProtocol(protocol);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDPort(final String port) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting port : " + port);
        this.clusterDaemon.setPort(port);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDDestDirPrefix(final String destDirPrefix) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting dest dir prefix : " + destDirPrefix);
        this.clusterDaemon.setDestDirPrefix(destDirPrefix);
        this.clusterDaemonTask.setDestDirPrefix(destDirPrefix);
        this.clusterDaemon.setCdDir(destDirPrefix);
        this.clusterDaemonTask.setCdDir(destDirPrefix);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDInteractionMode(final String interactionMode) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting interaction mode : " + interactionMode);
        this.clusterDaemon.setInteractionMode(interactionMode);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDJavaHome(final String javaHome) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting java home : " + javaHome);
        this.clusterDaemon.setJdk(javaHome);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDJonasRoot(final String jonasRoot) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting jonas root : " + jonasRoot);
        this.clusterDaemon.setJonasRoot(jonasRoot);
        this.clusterDaemonTask.setJonasRoot(new File(jonasRoot));
    }

    /**
     * {@inheritDoc}
     */
    public void setCDServerNamePrefix(final String serverNamePrefix) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting server name prefix : " + serverNamePrefix);
        this.clusterDaemon.setClusterNodesName(serverNamePrefix);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDXparam(final String xparam) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting xparam : " + xparam);
        this.clusterDaemon.setXprm(xparam);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDAutoBoot(final boolean autoBoot) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting auto boot : " + autoBoot);
        this.clusterDaemon.setAutoBoot(new Boolean(autoBoot).toString());
    }

    /**
     * {@inheritDoc}
     */
    public void setCDNbInstances(final int nbInsts) {
        Jonas5.logger.finest("[Cluster daemon configuration] setting nb instance : " + nbInsts);
        this.clusterDaemon.setWebInstNb(nbInsts);
        this.clusterDaemon.setEjbInstNb(0);
        this.clusterDaemonTask.setWebInstNb(nbInsts);
        this.clusterDaemonTask.setEjbInstNb(0);
    }

    /**
     * {@inheritDoc}
     */
    public void executeCDConf() {
        this.clusterDaemonTask.addConfiguredClusterDaemon(this.clusterDaemon);
        this.clusterDaemonTask.execute();
    }
    /* --------------------------------- */
}
