/**
 * JOnAS Configurator
 * Copyright (C) 2008-2010 Bull S.A.S.
 * Copyright (C) 2008-2010 France Telecom R&D
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Jonas53.java 22044 2012-01-27 19:20:26Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.tools.configurator.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.logging.Logger;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.listener.TimestampedLogger;
import org.ow2.jonas.antmodular.cluster.clusterdaemon.ClusterDaemon;
import org.ow2.jonas.antmodular.cmi.Cmi;
import org.ow2.jonas.antmodular.jonasbase.bootstrap.JOnASBaseTask;
import org.ow2.jonas.antmodular.jonasbase.bootstrap.JonasProperties;
import org.ow2.jonas.antmodular.jonasbase.bootstrap.Services;
import org.ow2.jonas.antmodular.jonasbase.carol.Carol;
import org.ow2.jonas.antmodular.jonasbase.db.base.Db;
import org.ow2.jonas.antmodular.jonasbase.db.h2.DbH2;
import org.ow2.jonas.antmodular.jonasbase.discovery.multicast.DiscoveryMulticast;
import org.ow2.jonas.antmodular.jonasbase.ha.Ha;
import org.ow2.jonas.antmodular.jonasbase.jms.Jms;
import org.ow2.jonas.antmodular.jonasbase.mail.Mail;
import org.ow2.jonas.antmodular.jonasbase.resource.JdbcXml;
import org.ow2.jonas.antmodular.jonasbase.wsdl.WsdlPublish;
import org.ow2.jonas.antmodular.web.base.Ajp;
import org.ow2.jonas.antmodular.web.base.Cluster;
import org.ow2.jonas.antmodular.web.base.Http;
import org.ow2.jonas.antmodular.web.base.Https;
import org.ow2.jonas.antmodular.web.base.WebContainer;
import org.ow2.jonas.tools.configurator.api.JDBCConfiguration;
import org.ow2.jonas.tools.configurator.api.JonasConfigurator;

/**
 * JonasConfigurator implementation for Jonas 5.3.x
 * 
 * @author S. Ali Tokmen
 */
public class Jonas53 extends FileReplacerHelper implements JonasConfigurator {

    /**
     * JONAS_BASE
     */
    private File jonasFileBase;

    /**
     * Link to the jonas ant task
     */
    private JOnASBaseTask jonasBase;

    /**
     * The cluster daemon configuration
     */
    private ClusterDaemon clusterDaemon;

    /**
     * Configuration logger
     */
    private static Logger logger = Logger.getLogger(Jonas53.class.getName());

    /**
     * To configure protocols
     */
    private Carol carol;

    /**
     * Has JOnAS protocols list been set?
     */
    private boolean isProtocolsListSet = false;

    /**
     * To configure databases service
     */
    private Db db;

    /**
     * Whether the optional db service has been activated
     */
    private boolean isDbActivated = false;

    /**
     * To configure discovery sevice
     */
    private DiscoveryMulticast discovery;

    /**
     * To configure jdbc
     */
    private Map<String, JdbcXml> jdbcXml;

    /**
     * To configure jms
     */
    private Jms jms;

    /**
     * To configure jonas global properties
     */
    private JonasProperties jonasProperties;

    /**
     * To configure mail service
     */
    private Mail mail;

    /**
     * Whether the optional mail service has been activated
     */
    private boolean isMailActivated = false;

    /**
     * To configure active services
     */
    private Services services;

    /**
     * To configure web container
     */
    private String webContainer;

    /**
     * To configure web container
     */
    private String jvmRoute;

    /**
     * To configure ajp
     */
    private Ajp ajp;

    /**
     * To configure http
     */
    private Http http;

    /**
     * To configure https
     */
    private Https https;

    /**
     * To configure cluster
     */
    private Cluster cluster;

    /**
     * To configure wsdl wsdlPublisherFile
     */
    private org.ow2.jonas.antmodular.jonasbase.wsdl.File wsdlPublisherFile;

    /**
     * To configure WSDL publishing
     */
    private WsdlPublish wsdlPublish;

    /**
     * Whether the optional WSDL publishing service has been activated
     */
    private boolean isWsdlPublishActivated = false;

    /**
     * To configure ejb clustering
     */
    private Cmi cmi;

    /**
     * To configure ha cluster
     */
    private Ha ha;

    /**
     * The project, not really usefull, only to have a project to pass to ant
     * task.
     */
    private Project project;

    /**
     * Is jonas root set
     */
    private boolean isJonasRootSet = false;

    /**
     * Is jonas base set
     */
    private boolean isJonasBaseSet = false;

    /**
     * Is http webservice activated
     */
    private Boolean isHttpActivated = true;

    /**
     * Is https webservice activated
     */
    private Boolean isHttpsActivated = false;

    /**
     * Is ajp webservice activated
     */
    private Boolean isAjpActivated = false;

    /**
     * Is http replication webservice activated
     */
    private Boolean isHttpReplicationActivated = false;

    /**
     * Is ha activated
     */
    private Boolean isHaActivated = false;

    /**
     * Is ejb clustering activated
     */
    private Boolean isEjbClusteringActivated = false;

    /**
     * Derfault constructor
     */
    public Jonas53() {
        this.jonasBase = new JOnASBaseTask();
        this.createProject();
        this.jonasBase.setProject(this.project);

        this.carol = new Carol();
        this.carol.setProject(this.project);

        this.db = new DbH2();
        this.db.setProject(this.project);

        this.discovery = new DiscoveryMulticast();
        this.discovery.setProject(this.project);

        this.jdbcXml = new HashMap<String, JdbcXml>();

        this.jms = new Jms();
        this.jms.setProject(this.project);

        this.jonasProperties = new JonasProperties();
        this.jonasProperties.setProject(this.project);

        this.mail = new Mail();
        this.mail.setProject(this.project);

        this.services = new Services();
        this.services.setProject(this.project);

        this.ajp = new Ajp();
        this.http = new Http();
        this.https = new Https();
        this.cluster = new Cluster();

        this.wsdlPublish = new WsdlPublish();
        this.wsdlPublish.setProject(this.project);
        this.wsdlPublisherFile = new org.ow2.jonas.antmodular.jonasbase.wsdl.File();

        this.cmi = new Cmi();
        this.ha = new Ha();

        this.clusterDaemon = new ClusterDaemon();
        this.clusterDaemon.setProtocol("jrmp");
        this.clusterDaemon.setPort("1099");
        this.clusterDaemon.setProject(this.project);

        Jonas53.logger.finest("Created the JOnAS 5 configurator instance");
    }

    /**
     * Create the project
     */
    private void createProject() {
        this.project = new Project();
        this.project.init();
        this.project.initProperties();

        TimestampedLogger logger = new TimestampedLogger();
        logger.setMessageOutputLevel(Project.MSG_DEBUG);

        logger.setOutputPrintStream(new PrintStream(new OutputStream() {
            StringBuffer buf = new StringBuffer();

            @Override
            public void write(final int b) throws IOException {
                if (b == '\n' || b == '\r') {
                    if (this.buf.length() > 0) {
                        Jonas53.logger.finest(this.buf.toString());
                        this.buf.setLength(0);
                    }
                } else {
                    this.buf.append((char) b);
                }
            }
        }));

        this.project.addBuildListener(logger);
    }

    /* ------------------------------- */
    /* Jonas global parameters */
    /**
     * {@inheritDoc}
     */
    public void setJdk(final String jdk) {
        Jonas53.logger.finest("[GlobalParameters] setting jdk : " + jdk);
        NotApplicableHelper.notApplicable("GlobalParameters.Jdk");
    }

    /**
     * {@inheritDoc}
     */
    public void setJavaOpts(final String javaOpts) {
        Jonas53.logger.finest("[GlobalParameters] setting java opts : " + javaOpts);
        NotApplicableHelper.notApplicable("GlobalParameters.JavaOpts");
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasRoot(final String jonasRoot) {
        Jonas53.logger.finest("[GlobalParameters] setting jonas root : " + jonasRoot);
        this.project.setProperty("jonas.root", jonasRoot);

        this.isJonasRootSet = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasBase(final String jonasBase) {
        Jonas53.logger.finest("[GlobalParameters] setting jonas base : " + jonasBase);
        this.jonasFileBase = new File(jonasBase);
        this.project.setProperty("jonas.base", jonasBase);

        this.isJonasBaseSet = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasName(final String jonasName) {
        Jonas53.logger.finest("[GlobalParameters] setting jonas name : " + jonasName);
        this.addReplacement("jonas.properties", "jonas.name    jonas", "jonas.name    " + jonasName);
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasDomain(final String jonasDomain) {
        Jonas53.logger.finest("[GlobalParameters] setting jonas domain : " + jonasDomain);
        this.addReplacement("jonas.properties", "domain.name    jonas", "domain.name    " + jonasDomain);
    }

    /**
     * {@inheritDoc}
     */
    public void setHost(final String jonasHost) {
        Jonas53.logger.finest("[GlobalParameters] setting host : " + jonasHost);
        this.carol.setHost(jonasHost);
        this.jms.setHost(jonasHost);
    }

    /**
     * {@inheritDoc}
     */
    public void setJonasDevelopment(final boolean development) {
        logger.finest("[GlobalParameters] setting jonas.development : " + development);
        this.jonasProperties.setDevelopment(development);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Jonas protocols configuation */
    /**
     * {@inheritDoc}
     */
    public void setProtocolsList(final String protocolsList) {
        if (protocolsList != null && protocolsList.contains("cmi")) {
            throw new IllegalArgumentException("In JOnAS 5, CMI has become a service. To enable CMI, follow these steps:"
                + "\n\t1) In the CAROL protocols list, use JRMP, IIOP and/or IRMI as usual"
                + "\n\t2) In the JOnAS services list, add the \"cmi\" service");
        }

        Jonas53.logger.finest("[Protocols configuration] setting protocols list : " + protocolsList);
        this.carol.setProtocols(protocolsList);
        this.isProtocolsListSet = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsIrmiPort(final String irmiPort) {
        Jonas53.logger.finest("[Protocols configuration] setting irmi port : " + irmiPort);
        this.carol.setIrmiPort(irmiPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsIiopPort(final String iiopPort) {
        Jonas53.logger.finest("[Protocols configuration] setting iiop port : " + iiopPort);
        this.carol.setIiopPort(iiopPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsJrmpPort(final String jrmpPort) {
        Jonas53.logger.finest("[Protocols configuration] setting jrmp port : " + jrmpPort);
        this.carol.setJrmpPort(jrmpPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsCmiPort(final String cmiPort) {
        Jonas53.logger.finest("[Protocols configuration] setting cmi port : " + cmiPort);
        NotApplicableHelper.notApplicable("Protocols.CmiPort");
    }

    /**
     * {@inheritDoc}
     */
    public void setProtocolsLocalCallOptimized(final Boolean localCallOptimized) {
        Jonas53.logger.finest("[Protocols configuration] setting jndi local call optimisation : " + localCallOptimized);
        this.carol.setJrmpOptimization(localCallOptimized);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Active services */
    /**
     * {@inheritDoc}
     */
    public void setServices(final String services) {
        Jonas53.logger.finest("setting actives services list : " + services);
        this.services.setNames(services);
        this.jonasProperties.setServices(services);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service web configuration */
    /**
     * {@inheritDoc}
     */
    public void setWebcontainer(final String webContainer) {
        Jonas53.logger.finest("[Web configuration] setting webContainer : " + webContainer);
        this.webContainer = webContainer;
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsConnectorActivation(final Boolean activation) {
        Jonas53.logger.finest("[Web configuration] setting isHttpsActivated : " + activation);
        this.isHttpsActivated = activation;
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpConnectorActivation(final Boolean activation) {
        Jonas53.logger.finest("[Web configuration] setting isAjpActivated : " + activation);
        this.isAjpActivated = activation;
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpSessionReplicationActivation(final Boolean activation) {
        Jonas53.logger.finest("[Web configuration] setting isHttpReplicationActivated : " + activation);
        this.isHttpReplicationActivated = activation;
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpPort(final String httpPort) {
        Jonas53.logger.finest("[Web configuration] setting http port : " + httpPort);
        this.http.setPort(httpPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpMaxThreads(final String httpMaxThreads) {
        Jonas53.logger.finest("[Web configuration] setting max http threads : " + httpMaxThreads);
        NotApplicableHelper.notApplicable("HTTP.maxThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpMinSpareThreads(final String httpMinSpareThreads) {
        Jonas53.logger.finest("[Web configuration] setting min http spare threads : " + httpMinSpareThreads);
        NotApplicableHelper.notApplicable("HTTP.minSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpMaxSpareThreads(final String httpMaxSpareThreads) {
        Jonas53.logger.finest("[Web configuration] setting max http spare threads : " + httpMaxSpareThreads);
        NotApplicableHelper.notApplicable("HTTP.maxSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsPort(final String httpsPort) {
        Jonas53.logger.finest("[Web configuration] setting https port : " + httpsPort);
        this.https.setPort(httpsPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsMaxThreads(final String httpsMaxThreads) {
        Jonas53.logger.finest("[Web configuration] setting max https threads : " + httpsMaxThreads);
        NotApplicableHelper.notApplicable("HTTPS.maxThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsMinSpareThreads(final String httpsMinSpareThreads) {
        Jonas53.logger.finest("setting min https spare threads : " + httpsMinSpareThreads);
        NotApplicableHelper.notApplicable("HTTPS.minSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpsMaxSpareThreads(final String httpsMaxSpareThreads) {
        Jonas53.logger.finest("setting max https spare threads : " + httpsMaxSpareThreads);
        NotApplicableHelper.notApplicable("HTTPS.maxSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpPort(final String ajpPort) {
        Jonas53.logger.finest("[Web configuration] setting ajp port : " + ajpPort);
        this.ajp.setPort(ajpPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpMaxThreads(final String ajpMaxThreads) {
        Jonas53.logger.finest("[Web configuration] setting max ajp threads : " + ajpMaxThreads);
        NotApplicableHelper.notApplicable("AJP.maxThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpMinSpareThreads(final String ajpMinSpareThreads) {
        Jonas53.logger.finest("[Web configuration] setting min ajp spare threads : " + ajpMinSpareThreads);
        NotApplicableHelper.notApplicable("AJP.minSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setAjpMaxSpareThreads(final String ajpMaxSpareThreads) {
        Jonas53.logger.finest("[Web configuration] setting max ajp spare threads : " + ajpMaxSpareThreads);
        NotApplicableHelper.notApplicable("AJP.maxSpareThreads");
    }

    /**
     * {@inheritDoc}
     */
    public void setJvmRoute(final String jvmRoute) {
        Jonas53.logger.finest("[Web configuration] setting jvm route : " + jvmRoute);
        this.jvmRoute = jvmRoute;
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationClusterName(final String clusterName) {
        Jonas53.logger.finest("[Web configuration] setting cluster name : " + clusterName);
        this.cluster.setName(clusterName);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationMulticastAddress(final String mCastAddr) {
        Jonas53.logger.finest("[Web configuration] setting multi cast address : " + mCastAddr);
        this.cluster.setMcastAddr(mCastAddr);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationMulticastPort(final String mCastPort) {
        Jonas53.logger.finest("[Web configuration] setting multi cast port : " + mCastPort);
        this.cluster.setMcastPort(mCastPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setHttpReplicationListenPort(final String listenPort) {
        Jonas53.logger.finest("[Web configuration] setting listen port : " + listenPort);
        this.cluster.setListenPort(listenPort);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service db configuration */
    /**
     * {@inheritDoc}
     */
    public void setDbPort(final String port) {
        Jonas53.logger.finest("[Database configuration] setting db port: " + port);
        this.isDbActivated = true;
        this.db.setPort(port);
    }

    /**
     * {@inheritDoc}
     */
    public void setDbName(final String name) {
        Jonas53.logger.finest("[Database configuration] setting db name: " + name);
        this.isDbActivated = true;
        this.db.setDbName(name);
    }

    /**
     * {@inheritDoc}
     */
    public void setDbUsers(final String users) {
        Jonas53.logger.finest("[Database configuration] setting db users: " + users);
        this.isDbActivated = true;
        this.db.setUsers(users);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service discovery configuration */
    /**
     * {@inheritDoc}
     */
    public void setDiscoveryMasterActivated(final Boolean masterActivated) {
        Jonas53.logger.finest("[Discovery configuration] setting masterActivated : " + masterActivated);
        this.jonasProperties.setMaster(masterActivated);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoverySourcePort(final String sourcePort) {
        Jonas53.logger.finest("[Discovery configuration] setting source port : " + sourcePort);
        this.discovery.setSourcePort(sourcePort);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryDomainName(final String domainName) {
        Jonas53.logger.finest("[Discovery configuration] setting domain name : " + domainName);
        this.addReplacement("jonas.properties", "domain.name    jonas", "domain.name    " + domainName);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryGreetingPort(final String greetingPort) {
        Jonas53.logger.finest("[Discovery configuration] setting greeting port : " + greetingPort);
        this.discovery.setGreetingPort(greetingPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryMulticastAddress(final String multicastAddress) {
        Jonas53.logger.finest("[Discovery configuration] setting multicast address : " + multicastAddress);
        this.discovery.setMcastAddr(multicastAddress);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryMulticastPort(final String multicastPort) {
        Jonas53.logger.finest("[Discovery configuration] setting multicast port : " + multicastPort);
        this.discovery.setMcastPort(multicastPort);
    }

    /**
     * {@inheritDoc}
     */
    public void setDiscoveryTTL(final String ttl) {
        Jonas53.logger.finest("[Discovery configuration] setting ttl : " + ttl);
        this.discovery.setTtl(ttl);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service mail configuration */
    /**
     * {@inheritDoc}
     */
    public void setMailFactoryType(final String factoryType) {
        Jonas53.logger.finest("[Mail Service Configuration] setting mail factory type : " + factoryType);
        this.mail.setType(factoryType);
        this.isMailActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setMailFactoryName(final String factoryName) {
        Jonas53.logger.finest("[Mail Service Configuration] setting mail factory name : " + factoryName);
        this.mail.setName(factoryName);
        this.isMailActivated = true;
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service WS configuration */
    /**
     * {@inheritDoc}
     */
    public void setWsdlPublisherFileName(final String fileName) {
        Jonas53.logger.finest("[WS Service configuration] setting wsdlPublisherFile name : " + fileName);
        this.wsdlPublisherFile.setName(fileName);
        this.isWsdlPublishActivated = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setWsdlPublisherFileDirectory(final String fileDir) {
        Jonas53.logger.finest("[WS Service configuration] setting wsdlPublisherFile directory : " + fileDir);
        this.wsdlPublisherFile.setDir(fileDir);
        this.isWsdlPublishActivated = true;
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Service HA configuration */
    /**
     * {@inheritDoc}
     */
    public void setHaActivated(final Boolean activated) {
        Jonas53.logger.finest("[HA Service configuration] setting ha activated : " + activated);
        this.isHaActivated = activated;
    }

    /**
     * {@inheritDoc}
     */
    public void setHaMulticastAddress(final String multicastAddr) {
        Jonas53.logger.finest("[HA Service configuration] setting multicast address : " + multicastAddr);
        this.ha.setMcastAddr(multicastAddr);
    }

    /**
     * {@inheritDoc}
     */
    public void setHaMulticastPort(final String multicastPort) {
        Jonas53.logger.finest("[HA Service configuration] setting multicast port : " + multicastPort);
        this.ha.setMcastPort(multicastPort);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Security manager configuration */
    /**
     * {@inheritDoc}
     */
    public void setSecurityManagerActivated(final Boolean activation) {
        Jonas53.logger.finest("[Security Manager configuration] setting isActivates : " + activation);
        this.jonasProperties.setSecurityManager(activation);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Ejb clustering configuration */
    /**
     * {@inheritDoc}
     */
    /**
     * {@inheritDoc}
     */
    public void setEjbClusteringActivated(final Boolean activated) {
        Jonas53.logger.finest("[Ejb Clustering configuration] setting ejb clustering activated activated : " + activated);
        this.isEjbClusteringActivated = activated;
    }

    public void setEjbClusteringMulticastAddress(final String multicastAddr) {
        Jonas53.logger.finest("[Ejb Clustering configuration] setting multicast address : " + multicastAddr);
        this.cmi.setMcastAddr(multicastAddr);
    }

    /**
     * {@inheritDoc}
     */
    public void setEjbClusteringMulticastPort(final String multicastPort) {
        Jonas53.logger.finest("[Ejb Clustering configuration] setting multicast port : " + multicastPort);
        this.cmi.setMcastPort(multicastPort);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Jms configuration */
    /**
     * {@inheritDoc}
     */
    public void setJmsPort(final String port) {
        Jonas53.logger.finest("[Jms configuration] setting port : " + port);
        this.jms.setPort(port);
    }

    /**
     * {@inheritDoc}
     */
    public void setJmsQueues(final String queue) {
        Jonas53.logger.finest("[Jms configuration] setting queue : " + queue);
        this.jms.setInitialQueues(queue);
    }

    /**
     * {@inheritDoc}
     */
    public void setJmsTopics(final String topic) {
        Jonas53.logger.finest("[Jms configuration] setting topic : " + topic);
        this.jms.setInitialTopics(topic);
    }

    /* ------------------------------- */

    /* ------------------------------- */
    /* Jdbc configuration */
    /**
     * {@inheritDoc}
     */
    public void addJdbcRA(final String raName, final JDBCConfiguration configuration) {
        JdbcXml jdbcXml = new JdbcXml();
        jdbcXml.setProject(this.project);

        if (configuration.driverName != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting driverName: " + configuration.driverName);
            NotApplicableHelper.notApplicable("JDBC.driverName");
        }

        if (configuration.datasourceClass != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting datasourceClass: " + configuration.datasourceClass);
            jdbcXml.setClassName(configuration.datasourceClass);
        }

        Jonas53.logger.finest("[Jdbc configuration] setting raName: " + raName);
        NotApplicableHelper.notApplicable("JDBC.raName");

        if (configuration.checkLevel != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting checkLevel: " + configuration.checkLevel);
            jdbcXml.setConCheckLevel(configuration.checkLevel);
        }

        if (configuration.testStatement != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting testStatement: " + configuration.testStatement);
            jdbcXml.setConTestStmt(configuration.testStatement);
        }

        if (configuration.jndiName != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting jndiName: " + configuration.jndiName);
            jdbcXml.setName(configuration.jndiName);
        }

        if (configuration.mappername != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting mappername: " + configuration.mappername);
            jdbcXml.setMapper(configuration.mappername);
        }

        if (configuration.password != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting password: " + configuration.password);
            jdbcXml.setPassword(configuration.password);
        }

        if (configuration.rarLink != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting rarLink: " + configuration.rarLink);
            NotApplicableHelper.notApplicable("JDBC.rarLink");
        }

        if (configuration.url != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting url: " + configuration.url);
            jdbcXml.setUrl(configuration.url);
        }

        if (configuration.user != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting user: " + configuration.user);
            jdbcXml.setUserName(configuration.user);
        }

        if (configuration.poolInitialSize != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting poolInitialSize: " + configuration.poolInitialSize);
            jdbcXml.setInitConPool(configuration.poolInitialSize);
        }

        if (configuration.poolMaximumSize != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting poolMaximumSize: " + configuration.poolMaximumSize);
            jdbcXml.setMaxConPool(configuration.poolMaximumSize);
        }

        if (configuration.poolMaximumAgeMinutes != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting poolMaximumAgeMinutes: " + configuration.poolMaximumAgeMinutes);
            jdbcXml.setConnMaxAge(configuration.poolMaximumAgeMinutes);
        }

        if (configuration.poolMaximumOpenTime != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting poolMaximumOpenTime: " + configuration.poolMaximumOpenTime);
            jdbcXml.setMaxOpenTime(configuration.poolMaximumOpenTime);
        }

        if (configuration.poolMaximumNumberOfWaiters != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting poolMaximumNumberOfWaiters: "
                + configuration.poolMaximumNumberOfWaiters);
            jdbcXml.setMaxWaiters(configuration.poolMaximumNumberOfWaiters);
        }

        if (configuration.poolMaximumWaitTime != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting poolMaximumWaitTime: " + configuration.poolMaximumWaitTime);
            jdbcXml.setMaxWaitTime(configuration.poolMaximumWaitTime);
        }

        if (configuration.poolMinimumSize != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting poolMinimumSize: " + configuration.poolMinimumSize);
            jdbcXml.setMinConPool(configuration.poolMinimumSize);
        }

        if (configuration.poolSamplingPeriod != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting poolSamplingPeriod: " + configuration.poolSamplingPeriod);
            jdbcXml.setSamplingPeriod(configuration.poolSamplingPeriod);
        }

        if (configuration.poolMaximumNumberOfPreparedStatements != null) {
            Jonas53.logger.finest("[Jdbc configuration] setting poolMaximumNumberOfPreparedStatements: "
                + configuration.poolMaximumNumberOfPreparedStatements);
            jdbcXml.setPstmtMax(configuration.poolMaximumNumberOfPreparedStatements);
        }

        this.jdbcXml.put(raName, jdbcXml);
    }

    /* ------------------------------- */

    /**
     * {@inheritDoc}
     */
    public void execute() {
        if (!this.isJonasRootSet) {
            throw new IllegalArgumentException("Jonas root is not set.");
        }
        if (!this.isJonasBaseSet) {
            throw new IllegalArgumentException("Jonas base is not set.");
        }

        this.jonasBase.addConfiguredServices(this.services);
        if (this.isHaActivated) {
            this.jonasBase.addTasks(this.ha);
        }
        if (this.isEjbClusteringActivated) {
            this.jonasBase.addTasks(this.cmi);
        }
        this.jonasBase.execute();
        this.jonasProperties.execute();
        this.doReplacements(new File(this.jonasFileBase, "conf"));

        WebContainer webContainer = null;
        try {
            if (this.webContainer == null) {
                // If not set, get the webContainer from jonas.properties
                //
                // jonas.service.web.class is for example org.ow2.jonas.web.tomcat7.Tomcat7Service,
                // we therefore split it at each for and get the length - 2 (in our example, tomcat7).
                Properties jonasProperties = new Properties();
                InputStream fis = new FileInputStream(new File(this.jonasFileBase, "conf/jonas.properties"));
                try {
                    jonasProperties.load(fis);
                } finally {
                    fis.close();
                    fis = null;
                    System.gc();
                }
                String[] defaultWebcontainerClass = jonasProperties.getProperty("jonas.service.web.class").split("\\.");
                this.webContainer = defaultWebcontainerClass[defaultWebcontainerClass.length - 2];
            }
            for (String webContainerClassName : getAllClasses((URLClassLoader) this.getClass().getClassLoader(),
                WebContainer.class.getName())) {
                if (webContainerClassName.contains(this.webContainer)) {
                    webContainer = (WebContainer)
                        this.getClass().getClassLoader().loadClass(webContainerClassName).newInstance();
                    webContainer.setProject(this.project);
                }
            }
        } catch (Exception e) {
            throw new IllegalStateException("Cannot create Web container", e);
        }
        if (webContainer == null) {
            throw new IllegalStateException("Unknown Web container: " + this.webContainer);
        }

        if (this.jvmRoute != null) {
            Method m = null;
            try {
                m = webContainer.getClass().getMethod("setJvmRoute", String.class);
            } catch (Exception e) {
                NotApplicableHelper.notApplicable("Web.jvmRoute");
            }

            if (m != null) {
                try {
                    m.invoke(webContainer, this.jvmRoute);
                } catch (Exception e) {
                    throw new IllegalArgumentException("[Web configuration] cannot set jvm route : " + jvmRoute, e);
                }
            }
        }

        Object connector = null;
        Method connectorMethod = null;
        for (Method method : webContainer.getClass().getMethods()) {
            if (method.getName().equals("addConfiguredConnector") &&
                method.getParameterTypes().length == 1) {
                try {
                    connector = method.getParameterTypes()[0].newInstance();
                } catch (Exception e) {
                    throw new IllegalStateException("Cannot generate Web connector to configure", e);
                }
                connectorMethod = method;
                break;
            }
        }
        if (connector == null) {
            throw new IllegalStateException("Cannot find method for adding configured Web connector");
        }
        if (this.isHttpActivated) {
            try {
                Method m = connector.getClass().getMethod("addConfiguredHttp", this.http.getClass());
                m.invoke(connector, this.http);
            } catch (Exception e) {
                NotApplicableHelper.notApplicable("Web.HTTP");
            }
        }
        if (this.isHttpsActivated) {
            try {
                Method m = connector.getClass().getMethod("addConfiguredHttps", this.https.getClass());
                m.invoke(connector, this.https);
            } catch (Exception e) {
                NotApplicableHelper.notApplicable("Web.HTTPS");
            }
        }
        if (this.isAjpActivated) {
            try {
                Method m = connector.getClass().getMethod("addConfiguredAjp", this.ajp.getClass());
                m.invoke(connector, this.ajp);
            } catch (Exception e) {
                NotApplicableHelper.notApplicable("Web.AJP");
            }
        }
        if (this.isHttpReplicationActivated) {
            try {
                Method m = webContainer.getClass().getMethod("addConfiguredCluster", this.cluster.getClass());
                m.invoke(webContainer, this.cluster);
            } catch (Exception e) {
                NotApplicableHelper.notApplicable("Web.Cluster");
            }
        }
        try {
            connectorMethod.invoke(webContainer, connector);
        } catch (Exception e) {
            throw new IllegalStateException("Cannot add configured Web connector", e);
        }
        webContainer.execute();

        if (!this.isProtocolsListSet) {
            this.carol.setProtocols("jrmp");
        }
        this.carol.execute();

        if (this.isDbActivated) {
            this.db.execute();
        }

        this.discovery.execute();

        boolean first = true;
        for (JdbcXml jdbcXml : this.jdbcXml.values()) {
            // The JdbcXml task will write all DS definitions to a single file
            // called JONAS_BASE/deploy/jdbc-ds.xml
            //
            // Since the file does not exist at the first call, we need to set
            // it as "new file"; after we tell JdbcXml to merge
            jdbcXml.setNewFile(Boolean.toString(first));
            jdbcXml.execute();
            first = false;
        }

        this.jms.execute();

        if (this.isMailActivated) {
            this.mail.execute();
        }

        if (this.isWsdlPublishActivated) {
            this.wsdlPublish.addConfiguredFile(this.wsdlPublisherFile);
            this.wsdlPublish.execute();
        }
    }

    /* --------------------------------- */
    /* Cluster daemon configuration part */

    /**
     * {@inheritDoc}
     */
    public void setCDClusterName(final String clusterName) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting cluster name : " + clusterName);
        this.clusterDaemon.setName(clusterName);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDClusterDomain(final String clusterDomain) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting cluster domain name : " + clusterDomain);
        this.clusterDaemon.setDomainName(clusterDomain);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDProtocol(final String protocol) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting protocol : " + protocol);
        this.clusterDaemon.setProtocol(protocol);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDPort(final String port) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting port : " + port);
        this.clusterDaemon.setPort(port);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDDestDirPrefix(final String destDirPrefix) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting dest dir prefix : " + destDirPrefix);
        this.clusterDaemon.setDestDirPrefix(destDirPrefix);
        this.clusterDaemon.setCdDir(destDirPrefix);
        this.project.setProperty("jonas.base", destDirPrefix);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDInteractionMode(final String interactionMode) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting interaction mode : " + interactionMode);
        this.clusterDaemon.setInteractionMode(interactionMode);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDJavaHome(final String javaHome) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting java home : " + javaHome);
        this.clusterDaemon.setJdk(javaHome);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDJonasRoot(final String jonasRoot) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting jonas root : " + jonasRoot);
        this.project.setProperty("jonas.root", jonasRoot);

        this.isJonasRootSet = true;
    }

    /**
     * {@inheritDoc}
     */
    public void setCDServerNamePrefix(final String serverNamePrefix) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting server name prefix : " + serverNamePrefix);
        this.clusterDaemon.setClusterNodesName(serverNamePrefix);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDXparam(final String xparam) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting xparam : " + xparam);
        this.clusterDaemon.setXprm(xparam);
    }

    /**
     * {@inheritDoc}
     */
    public void setCDAutoBoot(final boolean autoBoot) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting auto boot : " + autoBoot);
        this.clusterDaemon.setAutoBoot(new Boolean(autoBoot).toString());
    }

    /**
     * {@inheritDoc}
     */
    public void setCDNbInstances(final int nbInsts) {
        Jonas53.logger.finest("[Cluster daemon configuration] setting nb instance : " + nbInsts);
        this.clusterDaemon.setInstNb(nbInsts);
    }

    /**
     * {@inheritDoc}
     */
    public void executeCDConf() {
        this.clusterDaemon.execute();
    }
    /* --------------------------------- */

    /**
     * Gets all classes in a given URLClassLoader that are children of a given class or interface.
     * @param urlCL URLClassLoader to scan.
     * @param parent Parent class or interface that returned classes should implement.
     * @return All classes in <code>urlCl</code> that are children of class or interface <code>parent</code>.
     * @throws IOException If the URLClassLoader cannot be read.
     * @throws URISyntaxException If the URLClassLoader cannot be read.
     * @throws ClassNotFoundException If class or interface <code>parent</code> cannot be loaded.
     */
    private SortedSet<String> getAllClasses(URLClassLoader urlCL, String parent)
        throws IOException, URISyntaxException, ClassNotFoundException {

        SortedSet<String> classes = new TreeSet<String>();

        Class<?> parentClass;
        try {
            parentClass = urlCL.loadClass(parent);
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException("Class " + parent + " not found in " + urlCL);
        }

        for (URL url : urlCL.getURLs()) {
            classes.addAll(getAllClasses(url, urlCL, parentClass));
        }

        return classes;
    }

    /**
     * Gets all classes in a given JAR file or a URLClassLoader that are children of a given class or interface.
     * @param jarUrl JAR file's URL.
     * @param urlCL URLClassLoader to scan.
     * @param parentClass Parent class or interface that returned classes should implement.
     * @return All classes in <code>jar</code> that are member of <code>urlCl</code> that are children of class or interface
     * <code>parent</code>.
     * @throws IOException If the JAR file cannot be read.
     * @throws URISyntaxException If the JAR file cannot be read.
     * @throws ClassNotFoundException If class or interface <code>parent</code> cannot be loaded.
     */
    private SortedSet<String> getAllClasses(URL jarUrl, URLClassLoader urlCL, Class<?> parentClass)
        throws IOException, URISyntaxException, ClassNotFoundException {

        SortedSet<String> classes = new TreeSet<String>();

        File jarFile = new File(jarUrl.toURI());
        if (!jarFile.isFile()) {
            return classes;
        }

        JarFile jar = new JarFile(jarFile);
        Enumeration<JarEntry> entries = jar.entries();
        while (entries.hasMoreElements()) {
            JarEntry entry = entries.nextElement();
            String entryName = entry.getName();
            if (entryName.endsWith(".class") && !entryName.contains("$")) {
                entryName = entryName.substring(0, entryName.length() - 6);
                entryName = entryName.replace('/', '.');
                entryName = entryName.replace('\\', '.');

                Class<?> entryClass;
                try {
                    entryClass = urlCL.loadClass(entryName);
                } catch (NoClassDefFoundError ignored) {
                    continue;
                }
                if (!entryClass.isInterface() && ! Modifier.isAbstract(entryClass.getModifiers())
                    && parentClass.isAssignableFrom(entryClass)) {

                    classes.add(entryName);
                }
            }
        }

        return classes;
    }
}
