/**
 * JOnAS Configurator
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: FileReplacerHelper.java 18553 2009-09-09 11:36:48Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.tools.configurator.impl;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Helps doing replacements in files
 * 
 * @author S. Ali Tokmen
 */
public class FileReplacerHelper {

    /**
     * FileReplacer logger
     */
    private static Logger logger = Logger.getLogger(FileReplacerHelper.class.getName());

    /**
     * A replacement task.
     */
    private class Task {
        /**
         * File to replace in.
         */
        String file;

        /**
         * Token to replace.
         */
        String from;

        /**
         * String to replace with.
         */
        String to;

        /**
         * Create a new replacement task.
         * 
         * @param file File to replace in.
         * @param from Token to replace.
         * @param to String to replace with.
         */
        public Task(final String file, final String from, final String to) {
            this.file = file;
            this.from = from;
            this.to = to;
        }
    }

    /**
     * Replacement tasks.
     */
    private List<Task> tasks = new ArrayList<Task>();

    /**
     * Adds a replacement task.
     * 
     * @param file File in which to replace.
     * @param from Original string.
     * @param to String to replace with.
     */
    protected void addReplacement(final String file, final String from, final String to) {
        this.tasks.add(new Task(file, from, to));
    }

    /**
     * Does all the replacement tasks as added in
     * {@link FileReplacerHelper#addReplacement(String, String, String)}.
     * 
     * @param parent Parent folder in which to look for all files.
     */
    protected void doReplacements(final File parent) {
        for (Task task : this.tasks) {
            File file = new File(parent, task.file);
            if (file.isFile()) {
                try {
                    StringBuffer sb = new StringBuffer();
                    String line;
                    BufferedReader br = new BufferedReader(new FileReader(file));
                    while ((line = br.readLine()) != null) {
                        sb.append(line.replace(task.from, task.to));
                        sb.append(System.getProperty("line.separator"));
                    }
                    br.close();
                    FileWriter fw = new FileWriter(file);
                    fw.write(sb.toString());
                    fw.close();
                } catch (Exception e) {
                    logger.log(Level.SEVERE, "Failed writing file " + file, e);
                }
            } else {
                logger.severe("Skipped file " + file);
            }
        }
    }
}
