/**
 * JOnAS: Java(TM) Open Application Server
 * Copyright (C) 2012 Bull S.A.S.
 * Contact: jonas-team@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: Wrapper.java 22224 2012-03-08 19:46:27Z cazauxj $
 * --------------------------------------------------------------------------
 */
package org.ow2.jonas.antmodular.jonasbase.wrapper;

import java.io.File;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.taskdefs.Copy;
import org.apache.tools.ant.types.FileSet;
import org.ow2.jonas.antmodular.bootstrap.JOnASAntTool;
import org.ow2.jonas.antmodular.jonasbase.bootstrap.AbstractJOnASBaseAntTask;
import org.ow2.jonas.antmodular.jonasbase.bootstrap.JReplace;

/**
 * Wrapper ANT task
 * @author Jeremy Cazaux
 */
public class Wrapper extends AbstractJOnASBaseAntTask {

    /**
     * JOnAS wrapper start conf
     */
    public static final String JONAS_WRAPPER_START_CONF = "jonas_start.conf";

    /**
     * JOnAS wrapper stop conf
     */
    public static final String JONAS_WRAPPER_STOP_CONF = "jonas_stop.conf";

    /**
     * JOnAS wrapper base conf
     */
    public static final String JONAS_WRAPPER_BASE_CONF = "jonas_wrapper_base.conf";

    /**
     * Delimiter
     */
    public static final String DELIMITER = "=";

    /**
     * JAVA_HOME property
     */
    public static final String JAVA_COMMAND_PROPERTY = "wrapper.java.command";

    /**
     * JAVA_HOME token
     */
    public static final String JAVA_COMMAND_TOKEN = JAVA_COMMAND_PROPERTY + DELIMITER + "/path/to/java";

    /**
     * JONAS_ROOT property
     */
    public static final String JONAS_ROOT_PROPERTY = "jonas.root";
    
    /**
     * JONAS_ROOT token
     */
    public static final String JONAS_ROOT_TOKEN = JONAS_ROOT_PROPERTY + DELIMITER + "/path/to/jonas-root";

    /**
     * JONAS_BASE property
     */
    public static final String JONAS_BASE_PROPERTY = "jonas.base";

    /**
     * JONAS_BASE token
     */
    public static final String JONAS_BASE_TOKEN = JONAS_BASE_PROPERTY + DELIMITER + "/path/to/jonas-base";

    /**
     * logfile property
     */
    public static final String LOGFILE_PROPERTY = "wrapper.logfile";

    /**
     * logfile token
     */
    public static final String LOGFILE_TOKEN = LOGFILE_PROPERTY + DELIMITER + "${wrapper.working.dir}/logs/wrapper.log";

    /**
     * logfile rollmode property
     */
    public static final String LOGFILE_ROLLMODE_PROPERTY = "wrapper.logfile.rollmode";

    /**
     * logfile's rollmode token
     */
    public static final String LOGFILE_ROLLMODE_TOKEN = LOGFILE_ROLLMODE_PROPERTY + DELIMITER + "DATE";

    /**
     * OS service name property
     */
    public static final String SERVICE_NAME_PROPERTY = "wrapper.ntservice.name";

    /**
     * OS service name token
     */
    public static final String SERVICE_NAME_TOKEN = SERVICE_NAME_PROPERTY + DELIMITER + "yajsw";

    /**
     * OS service display name property
     */
    public static final String SERVICE_DISPLAYNAME = "wrapper.ntservice.displayname";

    /**
     * OS service display name token
     */
    public static final String SERVICE_DISPLAYNAME_TOKEN = SERVICE_DISPLAYNAME + DELIMITER + "yajsw";

    /**
     * OS service description property
     */
    public static final String SERVICE_DESCRIPTION_PROPERTY = "wrapper.ntservice.description";
    
    /**
     * OS service description token
     */
    public static final String SERVICE_DESCRIPTION_TOKEN = SERVICE_DESCRIPTION_PROPERTY + DELIMITER + "yajsw";

    /**
     * Wrapper debug property
     */
    public static final String VERBOSITY_PROPERTY = "wrapper.debug";
    
    /**
     * Wrapper debug token
      */
    public static final String VERBOSITY_TOKEN = VERBOSITY_PROPERTY + DELIMITER + "false";

    /**
     * Default wrapper directory name
     */
    public static final String DEFAULT_WRAPPER_HOME_DIRECTORY_NAME = "wrapper";

    /**
     * Wrapper Setenv SH file
     */
    public static final String JONAS_WRAPPER_SETENV_SH = "setenv.sh";

    /**
     * Wrapper base include property
     */
    public static final String WRAPPER_BASE_INCLUDE_PROPERTY = "include";

    /**
     * Wrapper conf directory
     */
    public static final String WRAPPER_CONF_DIR = "conf";

    /**
     * Wrapper bin directory
     */
    public static final String WRAPPER_BIN_DIR = "bat";

    /**
     * Wrapper base include token
     */
    public static final String WRAPPER_BASE_INCLUDE_TOKEN = WRAPPER_BASE_INCLUDE_PROPERTY + DELIMITER +
            WRAPPER_CONF_DIR + File.separator + JONAS_WRAPPER_BASE_CONF;

    /**
     * jonas_stop include file
     *
     */
    public static final String WRAPPER_STOP_INCLUDE_FILE = "wrapper.stop.conf";

    /**
     * jonas stop include token
     */
    public static final String WRAPPER_STOP_INCLUDE_TOKEN = WRAPPER_STOP_INCLUDE_FILE + DELIMITER +
            WRAPPER_CONF_DIR + File.separator + JONAS_WRAPPER_STOP_CONF;
    /**
     * wrapper_root setenv property
      */
    public static final String WRAPPER_ROOT_PROPERTY = "wrapper_root";

    /**
     * wrapper_root setenv token
     */
    public static final String WRAPPER_ROOT_TOKEN = WRAPPER_ROOT_PROPERTY + DELIMITER + "\"$PRGDIR\"/..";

    /**
     * wrapper_base setenv property
     */
    public static final String SETENV_WRAPPER_BASE_PROPERTY = "wrapper_base";

    /**
     * wrapper_base setenv token
     */
    public static final String SETENV_WRAPPER_BASE_TOKEN = SETENV_WRAPPER_BASE_PROPERTY + DELIMITER + "\"$wrapper_root\"";

    public static final String WRAPPER_BASE_VAR_PROPERTY = "wrapper.base";
    
    public static final String WRAPPER_BASE_VAR_TOKEN = WRAPPER_BASE_VAR_PROPERTY + DELIMITER + "/path/to/wrapper-base";
   
    /**
     * JAVA HOME path
     */
    private String javaHome;

    /**
     * Wrapper home
     */
    private String wrapperHome;

    /**
     * Name of the OS service
     */
    private String serviceName;

    /**
     * Service description
     */
    private String serviceDescription;

    /**
     * True if the wrapper is start in verbosity mode
     */
    private Boolean verbosity;

    /**
     * Wrapper log file
     */
    private String logfile;
    
    /**
     * Log file rollmode
     */
    private LogfileRollmode logfileRollmode;

    /**
     * Default constructor
     */
    public Wrapper() {
         super();
    }

    /**
     * @param javaHome The path to the JAVA_HOME to set
     */
    public void setJavaHome(final String javaHome) {
        this.javaHome  = javaHome;        
    }

    /**
     * @param wrapperHome wrapper
     */
    public void setWrapperHome(final String wrapperHome) {
        this.wrapperHome = wrapperHome;
    }

    /**
     * @param serviceName The name of the OS service
     */
    public void setServiceName(final String serviceName) {
        this.serviceName = serviceName;
    }

    /**
     * @param logfileRollmode The rollmode of the logfile to set
     */
    public void setLogfileRollmode(final LogfileRollmode logfileRollmode) {
        this.logfileRollmode = logfileRollmode;
    }

    /**
     * @param logfileRollmode The rollmode of the logfile to set
     */
    public void setLogfileRollmode(final String logfileRollmode) {
        if ("JVM".equals(logfileRollmode)) {
            this.logfileRollmode = LogfileRollmode.JVM;
        } else if ("DATE".equals(logfileRollmode)) {
            this.logfileRollmode = LogfileRollmode.DATE;
        }
    }

    /**
     * @param verbosity Verbosity mode
     */
    public void setVerbosity(final Boolean verbosity) {
        this.verbosity = verbosity;
    }

    /**
     * @param serviceDescription Service description to set
     */
    public void setServiceDescription(String serviceDescription) {
        this.serviceDescription = serviceDescription;
    }

    /**
     * @param logfile log file to set
     */
    public void setLogfile(String logfile) {
        this.logfile = logfile;
    }

    /**
     * Execute this task.
     */
    @Override
    public void execute() {
        super.execute();

        //get wrapper_root
        File wrapperRoot;
        final File wrapperBase = new File(this.getJOnASBase(), DEFAULT_WRAPPER_HOME_DIRECTORY_NAME);
        
        if (this.wrapperHome == null || !new File(this.wrapperHome).isDirectory()) {
            wrapperRoot = new File(this.getJOnASRoot(), DEFAULT_WRAPPER_HOME_DIRECTORY_NAME);
            if (wrapperRoot == null || !wrapperRoot.isDirectory()) {
                throw new BuildException("Cannot found a valid wrapper home directory");
            }
        } else {
            wrapperRoot = new File(this.wrapperHome);
        }

        //copy root templates to the wrapper base conf & bin directories
        Copy copy = new Copy();
        JOnASAntTool.configure(this, copy);
        copy.setTodir(wrapperBase);
        FileSet confFileSet = new FileSet();
        confFileSet.setDir(new File(new File(wrapperRoot, "templates"), WRAPPER_CONF_DIR));
        FileSet binFileSet = new FileSet();
        binFileSet.setDir(new File(new File(wrapperRoot, "templates"), WRAPPER_BIN_DIR));
        copy.addFileset(confFileSet);
        copy.addFileset(binFileSet);
        copy.setOverwrite(true);
        copy.execute();

        File jonasWrapperBaseConfFile = new File(new File(wrapperBase, WRAPPER_CONF_DIR), JONAS_WRAPPER_BASE_CONF);
        if (!jonasWrapperBaseConfFile.exists() || !jonasWrapperBaseConfFile.isFile()) {
            throw new BuildException("File " + jonasWrapperBaseConfFile.getAbsolutePath() + " doesn't exist");
        }

        if(this.javaHome == null || !new File(this.javaHome).exists()) {
            //get JAVA_HOME env property if JAVA_HOME is not set
            this.javaHome = System.getenv("JAVA_HOME");
        }

        //update jonas_wrapper_base conf file
        File java = new File(new File(this.javaHome, "bin"), "java");
        JReplace javaHomeReplace = new JReplace();
        javaHomeReplace.setFile(jonasWrapperBaseConfFile);
        javaHomeReplace.setToken(JAVA_COMMAND_TOKEN);
        javaHomeReplace.setValue(JAVA_COMMAND_PROPERTY + DELIMITER + java.getAbsolutePath());
        javaHomeReplace.setLogInfo("Setting JAVA to " + java.getAbsolutePath());
        addTask(javaHomeReplace);

        JReplace jonasRootReplace = new JReplace();
        jonasRootReplace.setFile(jonasWrapperBaseConfFile);
        jonasRootReplace.setToken(JONAS_ROOT_TOKEN);
        jonasRootReplace.setValue(JONAS_ROOT_PROPERTY + DELIMITER + this.jonasRoot);
        jonasRootReplace.setLogInfo("Setting JONAS_ROOT to " + this.jonasRoot);
        addTask(jonasRootReplace);

        JReplace jonasBaseReplace = new JReplace();
        jonasBaseReplace.setFile(jonasWrapperBaseConfFile);
        jonasBaseReplace.setToken(JONAS_BASE_TOKEN);
        jonasBaseReplace.setValue(JONAS_BASE_PROPERTY + DELIMITER + this.getJOnASBase());
        jonasBaseReplace.setLogInfo("Setting JONAS_BASE to "  + this.getJOnASBase());
        addTask(jonasBaseReplace);

        if (this.verbosity != null) {
            JReplace verbosityReplace = new JReplace();
            verbosityReplace.setFile(jonasWrapperBaseConfFile);
            verbosityReplace.setToken(VERBOSITY_TOKEN);
            verbosityReplace.setValue(VERBOSITY_PROPERTY + DELIMITER + String.valueOf(this.verbosity));
            verbosityReplace.setLogInfo("Setting " + VERBOSITY_PROPERTY + " to " + this.verbosity);
            addTask(verbosityReplace);
        }

        if (this.serviceName != null) {
            JReplace serviceNameReplace = new JReplace();
            serviceNameReplace.setFile(jonasWrapperBaseConfFile);
            serviceNameReplace.setToken(SERVICE_NAME_TOKEN);
            serviceNameReplace.setValue(SERVICE_NAME_PROPERTY + DELIMITER + this.serviceName);
            serviceNameReplace.setLogInfo("Setting " + SERVICE_NAME_PROPERTY + " to " + this.serviceName);
            addTask(serviceNameReplace);

            JReplace serviceDisplayNameReplace = new JReplace();
            serviceDisplayNameReplace.setFile(jonasWrapperBaseConfFile);
            serviceDisplayNameReplace.setToken(SERVICE_DISPLAYNAME_TOKEN);
            serviceDisplayNameReplace.setValue(SERVICE_DISPLAYNAME + DELIMITER + this.serviceName);
            serviceDisplayNameReplace.setLogInfo("Setting " + SERVICE_DISPLAYNAME + " to " + this.serviceName);
            addTask(serviceDisplayNameReplace);
        }

        if (this.serviceDescription != null) {
            JReplace serviceDescriptionReplace = new JReplace();
            serviceDescriptionReplace.setFile(jonasWrapperBaseConfFile);
            serviceDescriptionReplace.setToken(SERVICE_DESCRIPTION_TOKEN);
            serviceDescriptionReplace.setValue(SERVICE_DESCRIPTION_PROPERTY + DELIMITER + this.serviceDescription);
            serviceDescriptionReplace.setLogInfo("Setting " + SERVICE_DESCRIPTION_PROPERTY + " to " + this.serviceDescription);
            addTask(serviceDescriptionReplace);
        }
        
        if (this.logfile != null) {
            JReplace logfileReplace = new JReplace();
            logfileReplace.setFile(jonasWrapperBaseConfFile);
            logfileReplace.setToken(LOGFILE_TOKEN);
            logfileReplace.setValue(LOGFILE_PROPERTY + DELIMITER + this.logfile);
            logfileReplace.setLogInfo("Setting " + LOGFILE_PROPERTY + " to " + this.logfile);
            addTask(logfileReplace);
        }
        
        if (this.logfileRollmode != null) {
            JReplace logfileRollmodeReplace = new JReplace();
            logfileRollmodeReplace.setFile(jonasWrapperBaseConfFile);
            logfileRollmodeReplace.setToken(LOGFILE_ROLLMODE_TOKEN);
            logfileRollmodeReplace.setValue(LOGFILE_ROLLMODE_PROPERTY + DELIMITER + this.logfileRollmode);
            logfileRollmodeReplace.setLogInfo("Setting " + LOGFILE_ROLLMODE_PROPERTY + " to " + this.logfileRollmode);
            addTask(logfileRollmodeReplace);
        }

        //get Wrapper bin directory
        File wrapperBinDir = new File(wrapperBase, WRAPPER_BIN_DIR);
        
        //replace path to configuration files in binaries scripts
        JReplace setenvReplace1 = new JReplace();
        setenvReplace1.setFile(new File(wrapperBinDir, JONAS_WRAPPER_SETENV_SH));
        setenvReplace1.setToken(WRAPPER_ROOT_TOKEN);
        setenvReplace1.setValue(WRAPPER_ROOT_PROPERTY + DELIMITER + wrapperRoot.getAbsolutePath());
        addTask(setenvReplace1);

        if (!wrapperRoot.getAbsolutePath().equals(wrapperBase.getAbsolutePath())) {
            JReplace setenvReplace2 = new JReplace();
            setenvReplace2.setFile(new File(wrapperBinDir, JONAS_WRAPPER_SETENV_SH));
            setenvReplace2.setToken(SETENV_WRAPPER_BASE_TOKEN);
            setenvReplace2.setValue(SETENV_WRAPPER_BASE_PROPERTY + DELIMITER + wrapperBase);
            addTask(setenvReplace2);
        }

        //TODO same thing for setenv bat file

        //replace the link to other configuration files from the jonas_start_script
        File jonasWrapperStartConf = new File(new File(wrapperBase, WRAPPER_CONF_DIR), JONAS_WRAPPER_START_CONF);
        JReplace jonasWrapperStartReplace = new JReplace();
        jonasWrapperStartReplace.setFile(jonasWrapperStartConf);
        jonasWrapperStartReplace.setToken(WRAPPER_BASE_VAR_TOKEN);
        jonasWrapperStartReplace.setValue(WRAPPER_BASE_VAR_PROPERTY + DELIMITER + wrapperBase);
        addTask(jonasWrapperStartReplace);

        //replace the link to other configuration file from the jonas_stop_script
        File jonasWrapperStopConf = new File(new File(wrapperBase, WRAPPER_CONF_DIR), JONAS_WRAPPER_STOP_CONF);
        JReplace jonasWrapperStopReplace = new JReplace();
        jonasWrapperStopReplace.setFile(jonasWrapperStopConf);
        jonasWrapperStopReplace.setToken(WRAPPER_BASE_VAR_TOKEN);
        jonasWrapperStopReplace.setValue(WRAPPER_BASE_VAR_PROPERTY + DELIMITER + wrapperBase);
        addTask(jonasWrapperStopReplace);

        //execute all tasks
        super.executeAllTask();
    }

}
