/*
 * @(#) ResourceManagerEventListener.java
 *
 * JOTM: Java Open Transaction Manager
 * 
 * This module was originally developed by 
 *  - INRIA inside the ObjectWeb Consortium(http://www.objectweb.org)
 *    
 * The original code and portions created by INRIA are 
 * Copyright (C) 2002 - INRIA (www.inria.fr)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 * -Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer. 
 *
 * -Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * --------------------------------------------------------------------------
 * $Id: JTAClientTransactionInterceptor.java,v 1.5 2003/12/05 20:24:18 degolf Exp $
 * --------------------------------------------------------------------------
 */
package org.objectweb.transaction.jta;

/**
 * A <code>ResourceManagerEventListener</code> allows late enrolment of
 * connections that are opened in the thread before the transaction starts.
 * 
 * <p>Resource Managers that support registration of
 * <code>ResourceManagerEventListener</code> will be able to notify the
 * transaction manager of connections opened without transaction context and get
 * a callback from the transaction manager when the transaction begins.<br /> In
 * a  typical Resource Manager, the registration of the listener is made by the
 * application server via mutator method such as <code>ResourceManager.
 * setResourceManagerEventListener(listener)</code>.</p>
 * 
 * @author Christophe Ney cney@batisseurs.com for Lutris Technologies
 */
public interface ResourceManagerEventListener {

    /**
     * This method is called by the resource manager when a 
     * <code>getConnection</code> call is made without a transaction 
     * context. The <code>ResourceManagerEvent</code> is to be used 
     * by the listener to keep track of those events and perform a late
     * enrolment of those logical connection when the transaction begins.
     * 
     * @param    event event sent to the listener
     */
    void connectionOpened(ResourceManagerEvent event);

    /**
     * This method is called by the resource manager when a logical connection
     * that was opened without transaction context is closed. 
     * It is to be used by the listener to keep track of those events and 
     * perform clean-up.
     * 
     * @param   event event sent to the listener
     */
    void connectionClosed(ResourceManagerEvent event);

    /**
     * This method is called by the resource manager when a logical connection
     * that was opened without transaction context is about to throw an 
     * exception.
     * It is to be used by the listener to keep track of those events and 
     * perform clean-up.
     * 
     * @param event event sent to the listener
     */
    void connectionErrorOccured(ResourceManagerEvent event);
}
