/*
 * @(#) TransactionManager.java
 *
 * JOTM: Java Open Transaction Manager
 * 
 * This module was originally developed by 
 *  - INRIA inside the ObjectWeb Consortium(http://www.objectweb.org)
 *    
 * The original code and portions created by INRIA are 
 * Copyright (C) 2002 - INRIA (www.inria.fr)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 * -Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer. 
 *
 * -Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * --------------------------------------------------------------------------
 * $Id: TransactionManager.java,v 1.4 2003/12/05 20:24:18 degolf Exp $
 * --------------------------------------------------------------------------
 */
package org.objectweb.transaction.jta;

import java.util.List;

/**
 * A <code>TransactionManager</code> extends JTA
 * <code>javax.transaction.TransactionManager</code> with an event API that
 * allows late enlistment of connections opened by resource managers before the
 * transaction has begun. <br />
 * 
 * To enable late enlistment of connections, the <code>TransactionManager</code>
 * must be registered as a <code>ResourceManagerEventListener</code> with the
 * resource manager.<br/> 
 * When the transaction manager receives an event from a
 * resource manager, it stores the events in a thread local list. The thread
 * local lists can be stacked/unstacked with the two methods
 * <code>pushThreadLocalRMEventList</code> and
 * <code>popThreadLocalRMEventList</code>. This is typically used when different
 * event lists are to be nested in the same thread, or when a list spams several
 * threads.
 * 
 *  Created on Feb 7, 2002
 *  @author   Christophe Ney, cney@batisseurs. com
 */
public interface TransactionManager
    extends javax.transaction.TransactionManager, ResourceManagerEventListener {

    /**
     * Push a new event list on the stack of thread local resource event sets.
     * The list must contain only <code>ResourceManagerEvent</code> objects.
     * 
     * @param eventList the possibly null list of events to store forecoming
     * <code>ResourceManagerEvent</code> events occuring in the current thread.
     */
    void pushThreadLocalRMEventList(List eventList);

    /**
     * Pop the current set from the stack of thread local resource event sets
     * The list contains <code>ResourceManagerEvent</code> objects.
     * 
     * @return The possibly null <code>ResourceManagerEvent</code> 
     * list of events that have occured in the  current thread since the last 
     * call of <code>pushThreadLocalRMEventList</code> or since the thread 
     * started.
     */
    List popThreadLocalRMEventList();
}