/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.objectweb.util.monolog.api;

import java.util.Enumeration;

/**
 *<P>A TopicalLogger dispatches events to a set of Handlers. A TopicalLogger
 * is a sort of message router.</P>
 *
 * <P>A topic is associated with each TopicalLogger. A topic
 * is represented by a dotted string, which is used to build a hierarchical
 * namespace. The latter should typically be aligned with the Java packaging
 * namespace.</P>
 *
 * <P>The name hierarchy of TopicalLogger allows adding properties
 * inheritance. For example, a TopicalLogger with the "a.b.c" name can inherit
 * of the Handlers list and the level from the "a.b" parent. </P>
 *
 *<P>Another property for a TopicalLogger is the capacity to have several
 * topics. This is important when a component is used by several other
 * components. This will allow events logged by the shared component to
 * appear for each component using this shared component. A consequence
 *  of this property is that a Logger may have several parents</P>
 *
 * <P>The additivity flag indicates if the current TopicalLogger inherits the
 * handlers of its parents. The default value is true.</P>
 */
public interface TopicalLogger extends Logger {

	// HANDLER MANAGEMENT //
	//--------------------//

	/**
	 * A TopicalLogger manages a list of Handler instances. This method
	 * allows adding a handler to this list. The addHandler method returns
	 * true only if the Handler did not exist
	 */
	void addHandler(Handler h) throws Exception;

	/**
	 * It returns the list of the handler associated to this logger.
	 * @return an array of Handler or an empty array.
	 */
	Handler[] getHandler();

	/**
	 * It returns the handler which the name is equals to the parameter
	 * @hn is the handler name
	 * @return an Handler or a null value.
	 */
	Handler getHandler(String hn);

	/**
	* A TopicalLogger manages a list of Handler instances. This method
	* allows removing a handler to this list.
	*/
   void removeHandler(Handler h) throws Exception;

	/**
	* A TopicalLogger manages a list of Handler instances. This method
	* allows removing all handler.
	*/
   void removeAllHandlers() throws Exception;

	/**
	 * It assigns the additivity flag for this logger instance.
	 */
	void setAdditivity(boolean a);

	/**
	 * It retrieves the additivity flag for this logger instance.
	 */
	boolean getAdditivity();



	// TOPIC MANAGEMENT //
	//------------------//

	/**
	 *This method allows adding a topic to a TopicalLogger. This actions change
	 * the hierarchical structure, but also the list of handlers. The list of handlers
	 * of a TopicalLogger is composed of its handlers and all handlers inherited
	 * from its parents. Adding a topic changes the inherited handlers list.
	 */
	void addTopic(String topic)  throws Exception ;

	/**
	 *This method allows getting a topic list of this TopicalLogger.
	 */
	String[] getTopic();

	/**
	 * This method allows getting a topic list of this TopicalLogger.
	 * Only kept for the backward compatibility.
	 */
	Enumeration getTopics();

	/**
	 *This method allows removing a topic to a TopicalLogger. This actions change
	 * the hierarchical structure, but also the list of handlers. The list of handlers
	 * of a TopicalLogger is composed of its handlers and all handlers inherited
	 * from its parents. Removing a topic changes the inherited handlers list.
	 */
	void removeTopic(String topic)  throws Exception ;
}
