/**
 * Copyright (C) 2001-2004 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/**
 * Copyright (C) 2001-2005 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.objectweb.util.monolog.wrapper.remote.api;

import java.rmi.RemoteException;
import java.util.Map;
import java.util.Properties;

import org.objectweb.util.monolog.api.Level;

/**
 * Defines a MBean for managing the 3 Monolog concepts: Logger, Handler and 
 * Level
 * 
 * @see org.objectweb.util.monolog.wrapper.remote.api.LoggerInfo
 * @author S.Chassande-Barrioz
 */
public interface MonologFactoryProxy {

	/**
	 * It defines a new Level with a name and an integer value.
	 * @param name is the name of the new level
	 * @param value is the integer value of the new level
	 * @return true if the level has been created.
	 */
	boolean defineLevel(String name, int value) throws RemoteException;

	/**
	 * It defines a new Level with a name and a string value. The string value
	 * is analyzed to obtain the integer value.
	 * @param name is the name of the new level
	 * @param value is the string value of the new level
	 * @return true if the level has been created.
	 */
	boolean defineLevel(String name, String value) throws RemoteException;

	/**
	 * It removes a Level instance to this manager.
	 */
	void removeLevel(String name) throws RemoteException;

	/**
	 * It retrieves a Level instance which the name is equals to the parameter.
	 * @param name is the name of request Level
	 * @return a Leve instance or a null value if the level does not exist.
	 */
	Level getLevel(String name) throws RemoteException;

	/**
	 * It retrieves a Level instance which the integer value is equals to the
	 * parameter.
	 * @param value is the integer value of request Level
	 * @return a Leve instance or a null value if the level does not exist. As
	 * it is possible to define several Levels which have the same integer value
	 * this methods returns the Level instance of first name found in the list.
	 */
	Level getLevel(int value) throws RemoteException;

	/**
	 * It retrieves all Level instances defined in this manager.
	 */
	Level[] getLevels() throws RemoteException;

	/**
	 * Compares two levels.
	 * @param levelname1 is the name of the first level
	 * @param levelname2 is the name of the second level
	 * @returns a 
	 * negative integer, zero, or a positive integer as the first level  is less than,
	 * equal to, or greater than the second level.
	 */
	int compareTo(String levelname1,  String levelname2) throws RemoteException;

    /**
	 * Creates a new handler
	 * @param hn is the name of the handler to create
	 * @param 	handlertype is the type of the parameter. The possible value are
	 * 			defined in this interface by the XXX_HANDLER_TYPE constants.
	 * @return	true if the handler has been created
	 */
	boolean createHandler(String hn, String handlertype) throws RemoteException;

	/**
	 * It removes the handler which the name is specified by the parameter
	 * @param handlername is the name of the handler
	 * @return true if the handler has been removed.
	 */
	boolean removeHandler(String handlername) throws RemoteException;

	/**
	 * It retrieves name of all handler managed by this factory.
	 */
	String[] getHandlerNames() throws RemoteException;

	/**
	 * It retrieves the attributes of an handler
	 * @param handlername is the name of the handler
	 * @return a map containing the association between an attribute name 
	 * (String) and an attribute value (String).
	 */
	Map getHandlerAttributes(String handlername) throws RemoteException;

	/**
	 * It retrieves the attributes of all handlers
	 * @return Map(
	 * 		String handlername, 
	 * 		Map(String attributename, String attributevalue)
	 * 		)
	 */
	Map getAllHandlerAttributes() throws RemoteException;
	
	/**
	 * Assignes a value to an handler attribute.
	 * @param handlername is the name of the handler
	 * @param attributeName is the name of the attribute
	 * @param value is the new value of the attribute
	 */
	void setHandlerAttribute(String handlername, 
	        String attributeName, 
	        String value) throws RemoteException;
	
	
	/**
	 * Creates a logger if it does not exist.
	 * @param loggername is the name of the logger 
	 */
	LoggerInfo getLogger(String loggername) throws RemoteException;
	
	/**
	 * Creates a logger if it does not exist.
	 * @param loggername is the name of the logger 
	 * @param resourceBundleName allows specifying the name of a 
	 * resource bundle in order to internationalise the logging. 
	 */
	LoggerInfo getLogger(String key, String resourceBundleName) throws RemoteException;
	
	/**
	 * Accessors to a resource bundle name associated to a LoggerFactory.
	 */
	String getResourceBundleName() throws RemoteException;
	
	/**
	 * Accessors to a resource bundle name associated to a LoggerFactory.
	 */
	void setResourceBundleName(String resourceBundleName) throws RemoteException;

	/**
	 * It retrieves a list of all loggers.
	 */
	LoggerInfo[] getLoggers() throws RemoteException;

	/**
	 * A TopicalLogger manages a list of Handler instances. This method
	 * allows adding a handler to this list. The addHandler method returns
	 * true only if the Handler did not exist
	 */
	void addHandlerToLogger(String handlerName, String loggerName) throws RemoteException;

	/**
	* A TopicalLogger manages a list of Handler instances. This method
	* allows removing a handler to this list.
	*/
	void removeHandlerFromLogger(String handlerName, String loggerName) throws RemoteException;

	/**
	* A TopicalLogger manages a list of Handler instances. This method
	* allows removing all handler.
	*/
   void removeAllHandlersFromLogger(String loggerName) throws RemoteException;

	/**
	 * It assigns the additivity flag for this logger instance.
	 */
	void setAdditivity(boolean a, String loggerName) throws RemoteException;

	/**
	 * It assigns a level to a logger.
	 * @param level is an int value corresponding to a level
	 * @param loggerName is the name of logger which the level must be set.
	 */
	void setLoggerLevel(int level, String loggerName) throws RemoteException;

	/**
	 * It assigns a level to a logger. If the level name does not match any 
	 * existing level, the level is not set on the logger.
	 * @param level is a string value corresponding to a level defined into the
	 * LevelFactory
	 * @param loggerName is the name of logger which the level must be set.
	 */
	void setLoggerLevel(String levelName, String loggerName) throws RemoteException;

	/**
	 * This method allows adding a topic to a TopicalLogger. This actions change
	 * the hierarchical structure, but also the list of handlers. The list of handlers
	 * of a TopicalLogger is composed of its handlers and all handlers inherited
	 * from its parents. Adding a topic changes the inherited handlers list.
	 */
	void addTopicToLogger(String topic, String loggerName)  throws RemoteException ;

	/**
	 *This method allows removing a topic to a TopicalLogger. This actions change
	 * the hierarchical structure, but also the list of handlers. The list of handlers
	 * of a TopicalLogger is composed of its handlers and all handlers inherited
	 * from its parents. Removing a topic changes the inherited handlers list.
	 */
	void removeTopicFromLogger(String topic, String loggerName)  throws RemoteException ;
	
	/**
	 * Retrieves the properties corresponding to the current configuration.
	 */
	Properties getMonologProperties() throws RemoteException;

	/**
	 * Configure Monolog with properties.
	 */
	void setMonologProperties(Properties p) throws RemoteException;
}
