/**
 * Copyright (C) 2005-2005 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.objectweb.util.monolog.lib;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Loggable;
import org.objectweb.util.monolog.api.Logger;
import org.objectweb.util.monolog.api.LoggerFactory;

/**
 * This class is a basic implementation of Loggable interface. This class holds
 * reference to the logger and the logger factory.
 *
 * @author S.Chassande-Barrioz
 */
public class LoggableImpl implements Loggable {
 
    protected boolean debug = false;
    protected Logger logger = null;
    protected LoggerFactory loggerFactory = null;
    
    
    public Logger getLogger() {
        return logger;
    }
    
    public LoggerFactory getLoggerFactory() {
        return loggerFactory;
    }
    
    public void setLogger(Logger l) {
        logger = l;
        debug = logger != null && logger.isLoggable(BasicLevel.DEBUG);
    }
    
    public void setLoggerFactory(LoggerFactory lf) {
        loggerFactory = lf;
        String eln = getExpectedLoggerName();
        if (eln != null) {
            logger = loggerFactory.getLogger(eln);
        }
        if (logger != null) {
	        String lns = getLoggerNameSuffix();
	        if (lns != null) {
	            logger = loggerFactory.getLogger(logger.getName() + "." + lns);
	        }
        }
    }
    
    public String getLoggerName() {
        return logger == null ? null : logger.getName();
    }
    /**
     * @return a string defining the expected name of the logger. By default
     * this method retrieves a null value meaning that no particular name is
     * expected for the logger. Typically this method has to be override in
     * order to specify an expected logger name if wanted. 
     */
    public String getExpectedLoggerName() {
        return null;
    }
    
    /**
     * @return a string defining a suffix of the expected name of the logger.
     * logger.name = logger.name + "." + suffix
     */
    public String getLoggerNameSuffix() {
        return null;
    }
    
    /**
     * Indicates if the DEBUG level is enabled for the current logger. 
     */
    public boolean isDebug() {
        return logger != null && logger.isLoggable(BasicLevel.DEBUG);
    }
    
    /**
     * Assign the current logger and the current loggerFactory to an object 
     * or to each element if o is a structure. This method supports three 
     * structures: Map, Collection and Array. 
     */
    public void assignLogger(Object o) {
        assignLogger(o, logger);
    }
    
    /**
     * Assign a logger and the current loggerFactory to an object or to each 
     * element if o is a structure. This method supports three structures: Map, 
     * Collection and Array. 
     */
    public void assignLogger(Object o, Logger l) {
        if (o == null) {
        } else if (o instanceof Loggable) {
            ((Loggable) o).setLogger(l);
            ((Loggable) o).setLoggerFactory(loggerFactory);
        } else if (o instanceof Collection) {
            for (Iterator it = ((Collection) o).iterator(); it.hasNext();) {
                assignLogger(it.next(), l);
            }
        } else if (o.getClass().isArray()) {
            Object[] array = (Object[]) o;
            for (int i = 0; i < array.length; i++) {
                assignLogger(array[i], l);
            }
        } else if (o instanceof Map) {
            for (Iterator it = ((Map) o).entrySet().iterator(); it.hasNext();) {
                Map.Entry me = (Map.Entry) it.next();
                assignLogger(me.getKey(), l);
                assignLogger(me.getValue(), l);
            }
        }
    }
}