/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.objectweb.util.monolog.wrapper.printwriter;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Vector;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Handler;
import org.objectweb.util.monolog.api.Level;
import org.objectweb.util.monolog.api.LogInfo;
import org.objectweb.util.monolog.api.Logger;
import org.objectweb.util.monolog.api.MonologFactory;
import org.objectweb.util.monolog.api.MonologFactoryListener;
import org.objectweb.util.monolog.api.TopicalLogger;
import org.objectweb.util.monolog.wrapper.common.EnumrationImpl;
import org.objectweb.util.monolog.wrapper.common.LevelImpl;

/**
 * This class is a simple implementation of the Logger interface provided by
 * the monolog specification.
 *
 *@author     sebastien.chassande@inrialpes.fr
 */
public class LoggerImpl
	implements TopicalLogger, MonologFactory {

	public final static String PRINT_WRITER = "printwriter";

	/**
	 * The current level of this logger
	 */
	private int level = BasicLevel.DEBUG;
	private Level lev = null;

	protected String name = null;
	protected Vector topics = null;
	protected Hashtable handlers = null;
	protected boolean additivity = true;
	protected Map levels = null;

	/**
	 * The printwriter which messages are written.
	 */
	private PrintWriter pw = null;

	private boolean isOn = true;

	/**
	 * This constructor permits to specify the printWriter linked to this
	 * logger
	 *
	 */
	public LoggerImpl() {
		this(null);
	}

	/**
	 * This constructor permits to specify the printWriter linked to this
	 * logger
	 *
	 *@param  _pw  the printwriter
	 */
	public LoggerImpl(PrintWriter _pw) {
		this("org.objectweb.util.monolog.wrapper.printwriter", _pw);
	}

	/**
	 * This constructor permits to specify the printWriter linked to this
	 * logger
	 *
	 *@param  _pw  the printwriter
	 */
	public LoggerImpl(String n, PrintWriter _pw) {
		name = n;
		pw = _pw;
	}

	public PrintWriter getPrintWriter() {
		return pw;
	}

	public void addMonologFactoryListener(MonologFactoryListener mfl) {
	}

	public void removeMonologFactoryListener(MonologFactoryListener mfl) {
	}

	// IMPLEMENTATION OF THE HandlerFactory INTERFACE //
	//-----------------------------------------------//
	public void configure(Properties prop) throws Exception {
	}

	// IMPLEMENTATION OF THE HandlerFactory INTERFACE //
	//-----------------------------------------------//
	public Handler createHandler(String hn, String handlertype) {
		return this;
	}

	public Handler removeHandler(String handlername) {
		return this;
	}

    public Handler[] getHandlers() {
        return new Handler[] {this};
    }


	// IMPLEMENTATION OF THE LevelFactory INTERFACE //
	//-----------------------------------------------//

	public synchronized Level defineLevel(String name, int value) {
		Level l =  new LevelImpl(name, value);
		if (levels == null) {
			levels = new HashMap();
		}
		levels.put(new Integer(value), l);
		return l;
	}

	public Level defineLevel(String name, String value) {
		Level l =  new LevelImpl(name, value, this);
		if (levels == null) {
			levels = new HashMap();
		}
		levels.put(new Integer(value), l);
		return l;
	}

	public Level getLevel(String name) {
		Iterator it = levels.values().iterator();
		while(it.hasNext()) {
			Level l = (Level) it.next();
			if (l.getName().equals(name)) {
				return l;
			}
		}
		return null;
	}

	/**
	 * This method is not synchronized because the configuration is rarely
	 */
	public Level getLevel(int value) {
		return (Level) levels.get(new Integer(value));
	}

	/**
	 * This method is not synchronized because the configuration is rarely
	 */
	public Level[] getLevels() {
		return (Level[]) levels.values().toArray(new Level[levels.size()]);
	}

	public synchronized void removeLevel(String name) {
		Iterator it = levels.values().iterator();
		while(it.hasNext()) {
			Level l = (Level) it.next();
			if (l.getName().equals(name)) {
				it.remove();
			}
		}
	}


	// IMPLEMENTATION OF THE LoggerFactory INTERFACE //
	//-----------------------------------------------//

	public Logger getLogger(String key) {
		return this;
	}

	public Logger getLogger(String key, String resourceBundleName) {
		return this;
	}

	public String getResourceBundleName() {
		return null;
	}

	public void setResourceBundleName(String resourceBundleName) {
	}

	public Logger[] getLoggers() {
		return new Logger[]{this};
	}

    public String getTopicPrefix() {
        return null;
    }
    


	// IMPLEMENTATION OF THE TopicalLogger INTERFACE //
	//-----------------------------------------------//

	/**
	 *  Sets the IntLevel attribute of the LoggerImpl object
	 *
	 *@param  l  The new IntLevel value
	 */
	public void setIntLevel(int l) {
		level = l;
	}

	/**
	 *  Sets the Level attribute of the LoggerImpl object
	 *
	 *@param  l  The new Level value
	 */
	public void setLevel(Level l) {
		lev = l;
		if (lev != null) {
			level = lev.getIntValue();
		}
	}

	/**
	 *  Gets the CurrentIntLevel attribute of the LoggerImpl object
	 *
	 *@return    The CurrentIntLevel value
	 */
	public int getCurrentIntLevel() {
		return level;
	}

	/**
	 *  Gets the CurrentLevel attribute of the LoggerImpl object
	 *
	 *@return    The CurrentLevel value
	 */
	public Level getCurrentLevel() {
		return (lev == null
			? lev = new LevelImpl(
				(level ==-1
					?"INHERIT"
				:"INTER"),
				level)
			: lev);
	}

	/**
	 *  Gets the Loggable attribute of the LoggerImpl object
	 *
	 *@param  l  Description of Parameter
	 *@return    The Loggable value
	 */
	public boolean isLoggable(int l) {
		return pw != null && l >= level;
	}

	/**
	 *  Gets the Loggable attribute of the LoggerImpl object
	 *
	 *@param  l  Description of Parameter
	 *@return    The Loggable value
	 */
	public boolean isLoggable(Level l) {
		return pw != null && l != null && l.getIntValue() >= level;
	}

	/**
	 *  Gets the On attribute of the LoggerImpl object
	 *
	 *@return    The On value
	 */
	public boolean isOn() {
		return isOn;
	}

	/**
	 *  Gets the Topics attribute of the LoggerImpl object
	 *
	 *@return    The Topics value
	 */
	public Enumeration getTopics() {
		if (topics == null) {
			String[] ar = {name};
			return new EnumrationImpl(ar);
		}
		else {
            String[] str=new String[topics.size()];
            for (int i=0; i<topics.size(); i++ ) {
                str[i]=(String)(topics.elementAt(i));
            }
			return new EnumrationImpl(str);
		}
	}

	/**
	 *  Log method
	 */
	public void log(int level, Object o) {
		if (pw == null || !isOn() || o == null || !isLoggable(level)) {
			return;
		}
		if (level == BasicLevel.ERROR) {
			pw.println("**" + format(o.toString(), 2));
		}
		else {
			pw.println(format(o.toString(), 2));
		}
	}

	/**
	 *  Log method
	 */
	public void log(Level l, Object o) {
	}

	/**
	 *  Log method
	 */
	public void log(int level, Object o, Throwable t) {
		if (pw == null || !isOn() || o == null || !isLoggable(level)) {
			return;
		}
		pw.println(o.toString() + ":" + t.toString());
	}

	/**
	 *  Log method
	 */
	public void log(Level l, Object o, Throwable t) {
	}

	/**
	 *  Log method
	 */
	public void log(int level, Object o, Object location, Object method) {
		if (pw == null || !isOn() || o == null || !isLoggable(level)) {
			return;
		}
		pw.println(location.toString() + "." + method.toString() + "(...) :"
			+ o.toString());
	}

	/**
	 *  Log method
	 */
	public void log(Level l, Object o, Object location,
					Object method) {
	}

	/**
	 *  Log method
	 */
	public void log(int level, Object o, Throwable t, Object location, Object method) {
		if (pw == null || !isOn() || o == null || !isLoggable(level)) {
			return;
		}
		pw.println(location.toString() + "." + method.toString() + "(...) :"
			+ o.toString() + " " + t.toString());
	}

	/**
	 *  Log method
	 */
	public void log(Level l, Object o, Throwable t, Object location,
					Object method) {
	}

	/**
	 * Turn on this logger
	 */
	public void turnOn() {
		isOn = true;
	}

	/**
	 * Turn off this logger
	 */
	public void turnOff() {
		isOn = false;
	}

	/**
	 *  The toString method is override to signal the logger imlementation
	 * fowards its messages to a printwriter
	 */
	public String toString() {
		return "Personnal implementation on PrintWriter object";
	}


	// IMPLEMENTATION OF THE TopicalLogger INTERFACE //
	//-----------------------------------------------//

	public void addHandler(Handler h) throws Exception {
		if (h != null) {
			if (handlers == null) {
				handlers = new Hashtable();
			}
			handlers.put(h.getName(), h);
		}
	}

	public void addTopic(String topic) throws Exception {
		if (topics == null) {
			topics = new Vector();
			topics.addElement(name);
		}
		topics.addElement(topic);
	}

	public void removeHandler(Handler h) throws Exception {
		if (h != null && handlers != null) {
			handlers.remove(h.getName());
		}
	}

	public void removeAllHandlers() throws Exception {
		if (handlers!=null) {
			handlers.clear();
		}
	}

	public void removeTopic(String topic) throws Exception {
		if (topics != null) {
			topics.removeElement(topic);
		}
	}

	public void setAdditivity(boolean a) {
		additivity = a;
	}

	public boolean getAdditivity() {
		return additivity;
	}

	public Handler[] getHandler() {
        Handler[] result;
        if (handlers == null)
            result = new Handler[0];
        else {
            result = new Handler[handlers.size()];
            int i =0;
            for (Enumeration e=handlers.elements(); e.hasMoreElements(); i++) {
                result[i] = (Handler)(e.nextElement());
            }
        }
        return (result);
	}

	public Handler getHandler(String hn) {
		return (handlers == null ? null : (Handler) handlers.get(hn));
	}

	public String[] getTopic() {
		if (topics == null) {
			String[] ar = {name};
			return ar;
		}
		else {
            String[] str=new String[topics.size()];
            for (int i=0; i<topics.size(); i++ ) {
                str[i]=(String)(topics.elementAt(i));
            }
			return str;
		}
	}

	// IMPLEMENTATION OF THE Handler INTERFACE //
	//-----------------------------------------//
	public String getName() {
		return name;
	}

	public void setName(String n) {
		name = n;
	}

	public String getType() {
		return "logger";
	}

	public String[] getAttributeNames() {
		String[] ar = {PRINT_WRITER};
		return ar;
	}

	public Object getAttribute(String name) {
		if (PRINT_WRITER.equalsIgnoreCase(name)) {
			return pw;
		}
		return null;
	}

	public Object setAttribute(String name, Object value) {
		if (PRINT_WRITER.equalsIgnoreCase(name)
			&& value != null
			&& value instanceof PrintWriter) {
			pw = (PrintWriter) value;
		}
		return null;
	}

	/**
	 * This method permits to format messages. More exatcly this method find
	 * the class name and the method name where the log call was done.
	 * In order to find the right class name and method name, this method is
	 * parametrable either the number of call done in this logger.
	 */
	public static String format(String msg, int removeTopStack) {
		Throwable t = new Throwable().fillInStackTrace();
		StringWriter sw = new StringWriter();
		t.printStackTrace(new PrintWriter(sw));
		String m = sw.getBuffer().toString();

		int deb = -1;

		int fin = 0;

		// take the right line
		deb = -1;
		for (int i = 0; i < (removeTopStack + 1); i++) {
			deb = m.indexOf("\n", deb + 1);
		}

		deb = m.indexOf("at ", deb);
		fin = m.indexOf("\n", deb);
		m = m.substring(deb + 3, fin);

		// remove param
		deb = m.indexOf("(");
		fin = m.indexOf(":");
		m = m.substring(0, deb + 1) + m.substring(fin + 1, m.length());

		// remove package name
		deb = m.indexOf("(");
		int c1 = 0;
		int c2 = 0;
		int current = m.indexOf(".");
		while (current != -1 && current < deb) {
			c1 = c2;
			c2 = current;
			current = m.indexOf(".", current + 1);
		}
		m = m.substring(c1 + 1, m.length());

		return m + ": " + msg;
	}

    /**
     * Set the extension
     */
    public void addLogInfo(Character pattern, LogInfo logInfoProvider) {}

    /**
     * Remove an extension
     * @param pattern log info pattern
     * @return whether the object removed is the right log info provider
     */
    public void removeLogInfo (char pattern) {}

    /**
     * Gets the provider of the extension
     * @param pattern
     * @return LogInfo provider
     */
    public LogInfo getLogInfo(Character pattern) {return null;}

	static {
		if (BasicLevel.FATAL <= 0
			|| BasicLevel.ERROR <= 0
			|| BasicLevel.WARN <= 0
			|| BasicLevel.INFO <= 0
			|| BasicLevel.DEBUG <= 0) {
			BasicLevel.FATAL = 50000;
			BasicLevel.ERROR = 40000;
			BasicLevel.WARN = 30000;
			BasicLevel.INFO = 20000;
			BasicLevel.DEBUG = 10000;
		}
	}
}

