/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.objectweb.util.monolog.wrapper.printwriter;

import java.io.PrintStream;

import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Loggable;
import org.objectweb.util.monolog.api.Logger;
import org.objectweb.util.monolog.api.LoggerFactory;

/**
 * This class is a PrintStream wrapper. It exports the Printstream methods but
 * fowards the message to a Logger. This implementation bufferizes the data when
 * a print method is used. The buffer and the data are always written when a
 * println method is used. No end of line are inserted by the println methods.
 * A line is equals to a monolog message.
 *
 * @author S.Chassande-Barrioz
 */
public class PrintStreamImpl
        extends PrintStream
        implements Loggable {


	protected Logger logger = null;
    protected LoggerFactory loggerFactory = null;

	/**
	 * This field is the buffer which represents the current line.
	 */
	protected String currentLine = "";

    protected int level;

	/**
	 * It builds a PrintWriterImpl instance. The default level is DEBUG
	 *
	 * @param l is the logger toward which the message must be send
	 * @throws NullPointerException if the parameter is null.
	 */
	public PrintStreamImpl(Logger l) throws NullPointerException {
		super(new EmptyOutputStream());
		if (l == null)
			throw new NullPointerException("Logger parameter is null");
		logger = l;
        level =  BasicLevel.DEBUG;
	}

	/**
	 * It builds a PrintWriterImpl instance. The default level is DEBUG
	 *
	 * @param l is the logger toward which the message must be send
     * @param level is the level used to log message.
	 * @throws NullPointerException if the parameter is null.
	 */
	public PrintStreamImpl(Logger l, int level) throws NullPointerException {
		super(new EmptyOutputStream());
		if (l == null)
			throw new NullPointerException("Logger parameter is null");
		logger = l;
        this.level =  level;
	}

    /**
     * Retrieves the level of the messages
     * @return an int value representing the message priority (BasicLevel.XXXX)
     */
    public int getLevel() {
        return level;
    }

    /**
     * Changes the level of the messages
     * @param level is the new level
     */
    public void setLevel(int level) {
        this.level = level;
    }

    // IMPLEMENTATION OF THE Loggable INTERFACE //
    //------------------------------------------//

    /**
     * Retrieves the logger instance used
     */
    public Logger getLogger() {
        return logger;
    }

    /**
     * Assigns the logger instance to use
     */
    public void setLogger(Logger logger) {
        this.logger = logger;
    }

    /**
     * Retrieves the logger factory instance used
     */
    public LoggerFactory getLoggerFactory() {
        return loggerFactory;
    }

    /**
     * Assigns the logger factory instance to use
     */
    public void setLoggerFactory(LoggerFactory lf) {
        this.loggerFactory = lf;
    }

    // IMPLEMENTATION OF THE PrintWriter CLASS //
    //-----------------------------------------//


	/**
     * Writes the byte[] as a string in the buffer
     */
    public void write(byte[] bytes) {
		currentLine += new String(bytes);
	}

	/**
     * Compare the inner loggers
     */
    public boolean equals(Object o) {
		return o instanceof PrintStreamImpl
			&& ((PrintStreamImpl) o).logger == logger;
	}

    /**
     * Do nothing
     */
	public void flush() {
	}

	/**
     * Always throws a CloneNotSupportedException
     */
    protected Object clone() throws CloneNotSupportedException {
		throw new CloneNotSupportedException();
	}

	/**
     * Do nothing
     */
    public void close() {
	}

	/**
     * Call the toString() method over the inner logger
     */
    public String toString() {
		return logger.toString();
	}

	/**
     * Always retrieves 'false'
     * @return
     */
    public boolean checkError() {
		return false;
	}


    /**
     * Do nothing
     */
    protected void setError() {
	}

    /**
     * Writes the int value in the buffer
     */
	public void write(int i) {
		currentLine += i;
	}

    /**
     * Writes the byte[] as a string in the buffer
     */
	public void write(byte[] bytes, int i, int i1) {
		currentLine += new String(bytes, i, i1);
	}

    /**
     * Writes the boolean value in the buffer
     */
	public void print(boolean b) {
		currentLine += b;
	}

    /**
     * Writes the char value in the buffer
     */
	public void print(char c) {
        currentLine += c;
	}

    /**
     * Writes the int value in the buffer
     */
	public void print(int i) {
		currentLine += i;
	}

    /**
     * Writes the long value in the buffer
     */
	public void print(long l) {
		currentLine += l;
	}

    /**
     * Writes the float value in the buffer
     */
	public void print(float v) {
		currentLine += v;
	}

    /**
     * Writes the double value in the buffer
     */
	public void print(double v) {
		currentLine += v;
	}

    /**
     * Writes the char[] value as a String in the buffer
     */
	public void print(char[] chars) {
		logger.log(level, currentLine + new String(chars));
		currentLine = "";
	}

    /**
     * Writes the String value in the buffer
     */
	public void print(String s) {
		currentLine += s;
	}

    /**
     * Writes the Object value as a String in the buffer
     */
	public void print(Object o) {
        currentLine += o;
	}

    /**
     * log the buffer
     */
	public void println() {
		logger.log(level, currentLine);
		currentLine = "";
	}

    /**
     * 1/ Write a boolean value in the buffer.
     * 2/ Log the buffer.
     * 3/ Clear the buffer.
     */
	public void println(boolean b) {
		logger.log(level, currentLine + b);
		currentLine = "";
	}

    /**
     * 1/ Write a char value in the buffer
     * 2/ Log the buffer
     * 3/ Clear the buffer
     */
	public void println(char c) {
		logger.log(level, currentLine + c);
		currentLine = "";
	}

    /**
     * 1/ Write an int value in the buffer
     * 2/ Log the buffer
     * 3/ Clear the buffer
     */
	public void println(int i) {
		logger.log(level, currentLine + i);
		currentLine = "";
	}

    /**
     * 1/ Write a long value in the buffer
     * 2/ Log the buffer
     * 3/ Clear the buffer
     */
	public void println(long l) {
		logger.log(level, currentLine + l);
		currentLine = "";
	}

    /**
     * 1/ Write a float value in the buffer
     * 2/ Log the buffer
     * 3/ Clear the buffer
     */
	public void println(float v) {
		logger.log(level, currentLine + v);
		currentLine = "";
	}

    /**
     * 1/ Write a double value in the buffer
     * 2/ Log the buffer
     * 3/ Clear the buffer
     */
	public void println(double v) {
		logger.log(level, currentLine + v);
		currentLine = "";
	}

    /**
     * 1/ Write a char[] value in the buffer
     * 2/ Log the buffer
     * 3/ Clear the buffer
     */
	public void println(char[] chars) {
		logger.log(level, currentLine + new String(chars));
		currentLine = "";
	}

    /**
     * 1/ Write a String value in the buffer
     * 2/ Log the buffer
     * 3/ Clear the buffer
     */
	public void println(String s) {
		logger.log(level, currentLine + s);
		currentLine = "";
	}

    /**
     * 1/ Write a object value in the buffer
     * 2/ Log the buffer
     * 3/ Clear the buffer
     */
	public void println(Object o) {
		logger.log(level, currentLine + o);
		currentLine = "";
	}
}
