/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.objectweb.util.monolog.wrapper.ant;

import org.apache.tools.ant.BuildEvent;
import org.apache.tools.ant.BuildListener;
import org.apache.tools.ant.Project;
import org.objectweb.util.monolog.Monolog;
import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Logger;
import org.objectweb.util.monolog.api.LoggerFactory;

/**
 * This class is an ant BuildListener which logs events in monolog. This 
 * listener uses the LoggerFactory initialized by the Monolog.class.
 * The topic of the message is concatenation (with dot separator) of the project
 * name, the target name and the task name. 
 *
 * @author S.Chassande-Barrioz
 */
public class MonologBuildListener implements BuildListener {

    /**
     * The logger factory used for logger allocation if no logger are specified
     */
    private LoggerFactory loggerFactory;

    /**
     * The logger use for logging event. A null value means the logger is
     * allocated each time. In this case the topic name depends on the project, 
     * target and task names.
     */
    private Logger log;
    
    public MonologBuildListener() {
        this(Monolog.initialize());
    }

    public MonologBuildListener(Logger log) {
        this.log = log;
        if (log == null) {
            throw new IllegalArgumentException("Non null Logger is required");
        }
    }

    public MonologBuildListener(LoggerFactory loggerFactory) {
        this.loggerFactory = loggerFactory;
        if (loggerFactory == null) {
            throw new IllegalArgumentException("Non null LoggerFactory is required");
        }
    }

    /**
     * Does the logging of the event
     */
    private void log(BuildEvent be) {
        Logger logger;
        if (log == null) {
            logger = loggerFactory.getLogger(
                be.getProject().getName()  
                + "." + be.getTarget().getName() 
                + "." + be.getTask().getTaskName());
        } else {
            logger = log;
        }
        int level;
        switch(be.getPriority()) {
        case Project.MSG_ERR:
            level = BasicLevel.ERROR;
        	break;
        case Project.MSG_WARN:
            level = BasicLevel.WARN;
        	break;
        case Project.MSG_INFO:
            level = BasicLevel.INFO;
        	break;
        case Project.MSG_DEBUG:
        case Project.MSG_VERBOSE:
        default:
            level = BasicLevel.DEBUG;
        	break;
        }
        if (be.getSource() != null) {
            if (be.getException() != null) {
                logger.log(level, be.getMessage(), be.getException(), be.getSource(), be.getSource());
            } else {
                logger.log(level, be.getMessage(), be.getSource(), be.getSource());
            }
        } else if (be.getException() != null) {
            logger.log(level, be.getMessage(), be.getException());
        } else {
            logger.log(level, be.getMessage());
        }
    }
    
    // IMPLEMENTATION OF THE BuildListener INTERFACE //
    //-----------------------------------------------//
    
    public void buildStarted(BuildEvent be) {
        log(be);
    }
    public void buildFinished(BuildEvent be) {
        log(be);
    }
    public void targetStarted(BuildEvent be) {
        log(be);
    }
    public void targetFinished(BuildEvent be) {
        log(be);
    }
    public void taskStarted(BuildEvent be) {
        log(be);
    }
    public void taskFinished(BuildEvent be) {
        log(be);
    }
    public void messageLogged(BuildEvent be) {
        log(be);
    }
}
