/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.objectweb.util.monolog.wrapper.config;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Properties;

import org.objectweb.util.monolog.api.Handler;
import org.objectweb.util.monolog.api.Level;
import org.objectweb.util.monolog.api.LogInfo;
import org.objectweb.util.monolog.api.Logger;
import org.objectweb.util.monolog.api.MonologFactory;
import org.objectweb.util.monolog.api.MonologFactoryListener;
import org.objectweb.util.monolog.api.TopicalLogger;
import org.objectweb.util.monolog.wrapper.common.LevelImpl;

/**
 * This class is a basic implementation of the monolog factories
 * (HandlerFactory, LoggerFactory and LevelFactory). It does not linked to an
 * any underlying log system.
 *
 * @author Sebastien Chassande-Barrioz
 */
public class BasicFactory
	implements MonologFactory, Serializable {

	/**
	 * This field references the level instances by their names.<br/>
	 * key = a level name<br/>
	 * value = the unique Level instance linked to the name.
	 */
	protected HashMap nameToLevel = null;

	/**
	 * This field reference the level names by their integer value.<br/>
	 * key = a java.lang.Integer which the value is the level<br/>
	 * value = a String or an ArrayList of String. The strings represent the
	 * name which match to the integer value. Indeed both name can be associated
	 * to the same integer value.
	 */
	protected HashMap intToNames = null;

	/**
	 * This field references the handler instance by their names.<br/>
	 * key = a String object which is an handler name.
	 * value = the unique handler instance which has the key for name.
	 */
	protected HashMap handlers = null;

	/**
	 * This field references the handler instance by their names.<br/>
	 * key = a String object which is a logger topic.
	 * value = the unique handler instance which has the key for name.
	 */
	protected HashMap loggers = null;

	/**
	 * The resource bundle linked to the LoggerFactory.
	 */
	protected String resourceBundleName = null;

	/**
	 * It intializes the data struture, defines the default level and the root
	 * logger.
	 */
	public BasicFactory() {
		intToNames = new HashMap();
		nameToLevel = new HashMap();
		handlers = new HashMap();
		loggers = new HashMap();
		defineDefaultLevels();
		defineRootLogger();
	}

	/**
	 * It initializes the default monolog level:
	 * <ul>
	 * <li>DEBUG: 10 000</li>
	 * <li>INFO: 20 000</li>
	 * <li>WARN: 30 000</li>
	 * <li>ERROR: 40 000</li>
	 * <li>FATAL: 50 000</li>
	 * </ul>
	 */
	protected void defineDefaultLevels() {
		defineLevel("INHERIT", -1);
		defineLevel("DEBUG", 10000);
		defineLevel("INFO", 20000);
		defineLevel("WARN", 30000);
		defineLevel("ERROR", 40000);
		defineLevel("FATAL", 50000);
	}

	/**
	 * It defines the level of the root logger to WARN.
	 */
	protected void defineRootLogger() {
		getLogger("").setLevel(getLevel("WARN"));
	}

	public void addMonologFactoryListener(MonologFactoryListener mfl) {
	}

	public void removeMonologFactoryListener(MonologFactoryListener mfl) {
	}

	// IMPLEMENTATION OF THE LevelFactory INTERFACE //
	//-----------------------------------------------//

	public Level defineLevel(String name, int value) {
		return defineLevel(new LevelImpl(name, value));
	}

	public Level defineLevel(String name, String value) {
		return defineLevel(new LevelImpl(name, value, this));
	}

	public Level getLevel(String name) {
		return (Level) nameToLevel.get(name);
	}

	public Level getLevel(int value) {
		Object temp = intToNames.get(new Integer(value));
		if (temp == null) {
			return null;
		}
		else if (temp instanceof String) {
			return getLevel((String) temp);
		}
		else if (temp instanceof ArrayList) {
			return getLevel((String) ((ArrayList) temp).get(0));
		}
		return null;
	}

	public Level[] getLevels() {
		return (Level[]) nameToLevel.values().toArray(new Level[0]);
	}

	public void removeLevel(String name) {
		Level removed = (Level) nameToLevel.remove(name);
		if (removed != null) {
			Integer i = new Integer(removed.getIntValue());
			Object temp = intToNames.get(i);
			if (temp instanceof String) {
				intToNames.remove(i);
			}
			else if (temp instanceof ArrayList) {
				((ArrayList) temp).remove(name);
			}
		}
	}

	// OTHER METHODS //
	//---------------//

	/**
	 * Insert a level into the data structure.<br/>
	 * If the level name is already used with other integer value, the null
	 * value is returned.<br/>
	 * If the level name is already used with the same integer value, the level
	 * found in the data structure is returned.<br/>
	 *
	 * @param l the Level instance which must be inserted.
	 * @return the Level instance or a null value.
	 */
	private Level defineLevel(Level l) {
		String name = l.getName();
		int value = l.getIntValue();
		Level res = (Level) nameToLevel.get(name);
		if (res != null) {
			// The name is already defined.
			return (res.getIntValue() == value ? res : null);
		}
		else {
			res = l;
			nameToLevel.put(name, res);
			Integer i = new Integer(value);
			Object temp = intToNames.get(i);
			if (temp != null) {
				if (temp instanceof String) {
					if (!((Level) temp).getName().equalsIgnoreCase(name)) {
						// The int value has already another name
						// Add the new name to the other
						ArrayList al = new ArrayList(5);
						al.add(temp);
						al.add(name);
						intToNames.put(i, al);
					}
				}
				else if (temp instanceof ArrayList) {
					// The int value has already several another name
					ArrayList al = (ArrayList) temp;
					if (!al.contains(name)) {
						// Add the new name to the others
						al.add(name);
					}
				}
			}
			else {
				// The int value does not have any name
				intToNames.put(i, name);
			}
		}
		return res;
	}

	public void configure(Properties prop) throws Exception {
	}
	
	// IMPLEMENTATION OF THE LoggerFactory INTERFACE //
	//-----------------------------------------------//

	public Logger getLogger(String _key) {
	    String key = _key;
	    if ("".equals(_key)) {
	        key = "root";
	    }
		synchronized (loggers) {
			Logger res = (Logger) loggers.get(key);
			if (res == null) {
				res = new BasicLogger(key, this);
				loggers.put(key, res);
			}
			return res;
		}
	}
	public void removeLogger(Logger logger) {
		synchronized (loggers) {
			loggers.remove(logger.getName());
		}
	}
	
	public Logger getLogger(String key, String resourceBundleName) {
		return getLogger(key);
	}

	public String getResourceBundleName() {
		return resourceBundleName;
	}

	public void setResourceBundleName(String rbn) {
		resourceBundleName = rbn;
	}

	public Logger[] getLoggers() {
		return (TopicalLogger[]) loggers.values().toArray(
		        new TopicalLogger[loggers.size()]);
	}
    public String getTopicPrefix() {
        return null;
    }
    

	// IMPLEMENTATION OF THE HandlerFactory INTERFACE //
	//------------------------------------------------//

	public Handler createHandler(String hn, String handlertype) {
		Handler res = (Handler) handlers.get(hn);
		if (res != null) {
			return null;
		}
		res = new BasicHandler(hn, handlertype);
		handlers.put(hn, res);
		return res;
	}

	public Handler[] getHandlers() {
		return (Handler[]) handlers.values().toArray(new Handler[0]);
	}

	public Handler getHandler(String hn) {
		return (Handler) handlers.get(hn);
	}

	public Handler removeHandler(String hn) {
		return (Handler) handlers.remove(hn);
	}

    /**
     * Add the extention
     * @param pattern Character corresponding to the info to add
     * @param logInfoProvider instance of info provider
     */
    public void addLogInfo(Character pattern, LogInfo logInfoProvider) {}

    /**
     * Remove an extension
     * @param pattern log info pattern
     * @return whether the object removed is the right log info provider
     */
    public void removeLogInfo (char pattern) {}
    /**
     * Gets the provider of the extension
     * @param pattern
     * @return LogInfo provider
     */
    public LogInfo getLogInfo(Character pattern) {return null;}
}
