/**
 * Copyright (C) 2001-2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.objectweb.util.monolog.wrapper.log4j;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Properties;
import java.util.ResourceBundle;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.LogManager;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.xml.DOMConfigurator;
import org.objectweb.util.monolog.Monolog;
import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.LogInfo;
import org.objectweb.util.monolog.api.Logger;
import org.objectweb.util.monolog.api.TopicalLogger;
import org.objectweb.util.monolog.wrapper.common.AbstractFactory;

/**
 * This class wraps the LoggerFactory, HandlerFactory, LevelFactory concepts
 * into the log4j world. This implementation supports also the Configurable
 * interface. Then it is possible to specify a log4j configuration file.
 *
 * @author Sebastien Chassande-Barrioz
 */
public class MonologLoggerFactory
        extends AbstractFactory {

    public static final String LOG4J_CF_PROP = "log4j.categoryFactory";
    public static final String LOG4J_CF_VALUE
        = "org.objectweb.util.monolog.wrapper.log4j.MonologCategoryFactory";

    /**
     * The root logger of the logger hierarchy
     */
    protected static Logger rootLogger = null;

    /**
     * It specifies the factory of MonologCategory instances.
     */
    private static org.apache.log4j.spi.LoggerFactory factory
        = new MonologCategoryFactory();

    /**
     * This static code initializes the BasicLevel fields.
     */
    static {
        debug("Initializing " + MonologLoggerFactory.class.getName());
        debug("set default level values");
        BasicLevel.INHERIT = -1;
        debug("\t-INHERIT= " + BasicLevel.INHERIT);
        BasicLevel.DEBUG = org.apache.log4j.Level.DEBUG.toInt();
        debug("\t-DEBUG= " + BasicLevel.DEBUG);
        BasicLevel.INFO = org.apache.log4j.Level.INFO.toInt();
        debug("\t-INFO= " + BasicLevel.INFO);
        BasicLevel.WARN = org.apache.log4j.Level.WARN.toInt();
        debug("\t-WARN= " + BasicLevel.WARN);
        BasicLevel.ERROR = org.apache.log4j.Level.ERROR.toInt();
        debug("\t-ERROR= " + BasicLevel.ERROR);
        BasicLevel.FATAL = org.apache.log4j.Level.FATAL.toInt();
        debug("\t-FATAL= " + BasicLevel.FATAL);

        BasicLevel.LEVEL_INHERIT = new LevelImpl("INHERIT", BasicLevel.INHERIT);
        BasicLevel.LEVEL_DEBUG = new LevelImpl("DEBUG", BasicLevel.DEBUG);
        BasicLevel.LEVEL_INFO = new LevelImpl("INFO", BasicLevel.INFO);
        BasicLevel.LEVEL_WARN = new LevelImpl("WARN", BasicLevel.WARN);
        BasicLevel.LEVEL_ERROR = new LevelImpl("ERROR", BasicLevel.ERROR);
        BasicLevel.LEVEL_FATAL = new LevelImpl("FATAL", BasicLevel.FATAL);

        if (!org.apache.log4j.Logger.getRootLogger().getAllAppenders().hasMoreElements()) {
            BasicConfigurator.configure();
        }
        if (classLoaderIsoltion) {
            debug("class loader isolation activated");
            String rootName = null;
            int i = 0;
            while (rootLogger == null) {
                rootName = "root" + i;
                try {
                    org.apache.log4j.Logger rlog =
                        org.apache.log4j.Logger.getLogger(rootName, factory);
                    rlog.setAdditivity(false);
                    rootLogger = (Logger) rlog;
                    Monolog.debug("Instanciate the root logger " + rootName);
                } catch (ClassCastException exc) {
                    // this name has already been reserved by another
                    // instance of this class from another class loader
                    i ++;
                }
            }
            rootLoggerName = rootName;
            rootLoggerPrefix = rootLoggerName + '.';
        }
    }

    /**
     * It intializes the data struture, defines the default level and the root
     * logger.
     */
    public MonologLoggerFactory() {
        super();
        if (!classLoaderIsoltion) {
            rootLogger = new MonologCategory(
                org.apache.log4j.Logger.getRootLogger());
        }
    }

    public String getWrapperName() {
        return "log4j";
    }

    protected String[][] getDefaultHandlerType2className() {
        return new String[][] {
            {handlerTypes[0], "org.objectweb.util.monolog.wrapper.log4j.ConsoleHandler"},
            {handlerTypes[1], "org.objectweb.util.monolog.wrapper.log4j.FileHandler"},
            {handlerTypes[2], "org.objectweb.util.monolog.wrapper.log4j.RollingFileHandler"},
            {handlerTypes[3], "org.objectweb.util.monolog.wrapper.log4j.NTEventLogHandler"},
            {handlerTypes[4], "org.objectweb.util.monolog.wrapper.log4j.JMXHandler"},
            {handlerTypes[5], "org.objectweb.util.monolog.wrapper.log4j.SocketHubHandler"},
            {handlerTypes[6], "org.objectweb.util.monolog.wrapper.log4j.GenericHandler"}
        };
    }

    // IMPLEMENTATION OF THE Configurable INTERFACE //

    /**
     * This method permits to configure the factory with tha specific
     * configuration file: like a log4j.properties
     */
    public void configure(Properties prop) throws Exception {
        debug("MonologLoggerFactory.configure(prop=" + prop + ")");
        if (prop == null) {
            if (!org.apache.log4j.Logger.getRootLogger().getAllAppenders().hasMoreElements()) {
                BasicConfigurator.configure();
            }
            return;
        }

        String conf = prop.getProperty(LOG_CONFIGURATION_TYPE,
            prop.getProperty("log4jConfiguration", DEFAULT));
        debug("MonologLoggerFactory.configure(): conf=" + conf);

        if (DEFAULT.equals(conf)) {
            if (!org.apache.log4j.Logger.getRootLogger().getAllAppenders().hasMoreElements()) {
                BasicConfigurator.configure();
            }
            return;
        }

        // Fetch the configuration file name
        String filename = prop.getProperty(LOG_CONFIGURATION_FILE,
            prop.getProperty("log4jConfigurationFile", ""));
        debug("MonologLoggerFactory.configure(): filename=" + filename);

        if (XML.equals(conf)) {
            if (prop.getProperty(LOG_CONFIGURATION_FILE_USE_CLASSPATH, "false")
                    .equalsIgnoreCase("true")
                    || prop.getProperty("findFileInClassPath", "false")
                            .equalsIgnoreCase("true")) {
                debug("MonologLoggerFactory.configure(): load from classpath");
                // valid into classpath
                URL configFileResource = getClass().getClassLoader().getResource(
                        filename);
                DOMConfigurator.configure(configFileResource.getFile());
            } else {
                debug("MonologLoggerFactory.configure(): load from file system");
                DOMConfigurator.configure(filename);
            }
        }
        else if (PROPERTY.equals(conf)) {
            Properties log4jfileprop = null;
            // Check if the file name designs a path valid into the
            // classpath, or designs a path valid into the file system.
            // In both case load the property file
            if (prop.getProperty(LOG_CONFIGURATION_FILE_USE_CLASSPATH, "false")
                .equalsIgnoreCase("true")
                || prop.getProperty("findFileInClassPath", "false")
                .equalsIgnoreCase("true")) {

                debug("MonologLoggerFactory.configure(): load from classpath");

                if (!LogManager.DEFAULT_CONFIGURATION_FILE.equals(filename)) {
                    debug("MonologLoggerFactory.configure(): not default config file");
                    // valid into the classpath
                    log4jfileprop = getProperties(filename);
                    log4jfileprop.setProperty(LOG4J_CF_PROP, LOG4J_CF_VALUE);
                    PropertyConfigurator.configure(log4jfileprop);
                }
                // else the file is already loaded by the Logger class
            }
            else {
                debug("MonologLoggerFactory.configure(): load from file system");
                // valid into the file system
                log4jfileprop = new Properties();
                log4jfileprop.load(new FileInputStream(filename));
                log4jfileprop.setProperty(LOG4J_CF_PROP, LOG4J_CF_VALUE);
                PropertyConfigurator.configure(log4jfileprop);
            }
        }
        else {
            throw new Exception("Unsupported configuration type: " + conf);
        }
        debug("MonologLoggerFactory.configure(): End");
    }

    // Return null if properties are not reachable
    private Properties getProperties(String name) throws IOException {
        InputStream is = getClass().getClassLoader().getResourceAsStream(name);
        if (is != null) {
            Properties props = new Properties();
            props.load(is);
            return props;
        }
        throw new FileNotFoundException("Not found in classpath: " + name);
    }


    // IMPLEMENTATION OF INTERFACE LoggerFactory //
    //-------------------------------------------//

    public Logger getLogger(String key) {
        if (key == null || key.length() == 0 || key.equalsIgnoreCase("root")) {
            return rootLogger;
        }
                // isolates the logger hierarchy
                key = monoLoggerName(key);
        if (resourceBundleName == null)
            return (Logger) org.apache.log4j.Logger.getLogger(key, factory);
        else
            return getLogger(key, resourceBundleName);
    }

    public synchronized Logger getLogger(String key, String rbn) {
        if (key == null || key.length() == 0 || key.equalsIgnoreCase("root")) {
            return rootLogger;
        }
                // isolates the logger hierarchy
                key = monoLoggerName(key);
        org.apache.log4j.Logger res =
            org.apache.log4j.Logger.getLogger(key, factory);
        res.setResourceBundle(ResourceBundle.getBundle(rbn));
        return (Logger) res;
    }

    public Logger[] getLoggers() {
        HashSet al = new HashSet();
        for (Enumeration e = org.apache.log4j.LogManager.getCurrentLoggers();
             e.hasMoreElements();) {
            Object o = e.nextElement();
            if (o instanceof Logger)
                al.add(o);
        }
        al.add(rootLogger);
        return (Logger[]) al.toArray(new TopicalLogger[0]);
    }

    /**
     * Add the extention
     * @param pattern Character corresponding to the info to add
     * @param logInfoProvider instance of info provider
     */
    public void addLogInfo(Character pattern, LogInfo logInfoProvider) {}

    /**
     * Remove an extension
     * @param pattern log info pattern
     * @return whether the object removed is the right log info provider
     */
    public void removeLogInfo (char pattern) {}
    /**
     * Gets the provider of the extension
     * @param pattern
     * @return LogInfo provider
     */
    public LogInfo getLogInfo(Character pattern) {return null;}
}
