/**
 * Open SUIT - Simple User Interface Toolkit
 * 
 * Copyright (c) 2009 EBM Websourcing, http://www.ebmwebsourcing.com/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.ow2.opensuit.cel;

import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.List;

/**
 * Interface for compilation context
 * @author Adrien Ruffie / Pierre Smeyers
 * @author http://opensuit.ow2.org/
 * @version 1.0
 */
public interface ICompilationContext
{
    /**
     * Allow to recover <Class> located into this 
     * context by providing a name of object.
     * Example: 
     *  - request return HttpServletRequest
     *  - locale return Locale
     * @param iName The name of the object
     * @return Return from the context, the 
     * <Class> type of the requested object
     * @throws UnresolvedVariableError
     */
	Class<?> getVariableType(String iName) throws UnresolvedVariableError;
	
	/**
     * Allow to recover generic <Type> located into this 
     * context by providing a name of object.
     * Example: 
     * @param iName The name of the object
     * @return Return from the context, the 
     * <Type> of the requested object
     * @throws UnresolvedVariableError
     */
	Type getVariableGenericType(String iName) throws UnresolvedVariableError;
	
	public static class UnresolvedVariableError extends Exception
	{

        /**
         * 
         */
        private static final long serialVersionUID = -8100490749123700524L;
	}
	
	/**
	 * May return a list of compilable functions
	 * @param prefix The prefix of the functions
	 * @param name The name of the functions
	 * @return List<IFunctionContext>
	 */
	List<IFunctionContext> getFunctions(String prefix, String name);
	
	public static interface IFunctionContext
	{
		Method getMethod();
		
		void compile(List<IExpression> arguments, ICompilationLogger logger);
		
		String getValuePattern(List<IExpression> arguments);
	}
	
	/**
	 * This interface declares functions for
	 * compilation logger, the static values
	 * of different severity levels
	 * @author Adrien Ruffie / Pierre Smeyers
	 * @author http://opensuit.ow2.org/
	 * @version 1.0
	 */
	public static interface ICompilationLogger
	{
		public final static int INFO_LEVEL = ICompilationMessage.INFO_LEVEL;
		public final static int WARNING_LEVEL = ICompilationMessage.WARNING_LEVEL;
		public final static int ERROR_LEVEL = ICompilationMessage.ERROR_LEVEL;

		void addMessage(int level, String message);
	}
}
