/*
 * Copyright 2006, 2007 Odysseus Software GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.opensuit.cel.impl.misc;

import java.math.BigDecimal;
import java.math.BigInteger;

import org.ow2.opensuit.cel.ITypeConverter;
import org.ow2.opensuit.cel.util.ConversionError;




/**
 * Arithmetic Operations as specified in chapter 1.7.
 * 
 * @author Christoph Beck
 */
public class NumberOperations {
	private final static Long LONG_ZERO = Long.valueOf(0L);
	
	private static final boolean isFloatOrDouble(Object value) {
		return value instanceof Float || value instanceof Double;
	}

	private static final boolean isBigDecimalOrBigInteger(Object value) {
		return value instanceof BigDecimal || value instanceof BigInteger;
	}

	private static final boolean isBigDecimalOrFloatOrDouble(Object value) {
//		return value instanceof BigDecimal || isFloatOrDoubleOrDotEe(value);
		return value instanceof BigDecimal || isFloatOrDouble(value);
	}

	public static boolean checkTypes(ITypeConverter converter, Class<?> c1, Class<?> c2)
	{
		return converter.isConvertible(c1, Number.class) && converter.isConvertible(c2, Number.class);
	}
	public static boolean checkType(ITypeConverter converter, Class<?> c)
	{
		return converter.isConvertible(c, Number.class);
	}
	public static Class<?> getAddReturnType(Class<?> c1, Class<?> c2)
	{
		if (BigDecimal.class.isAssignableFrom(c1) || BigDecimal.class.isAssignableFrom(c2)){
			return BigDecimal.class;
		}
		
		if (c1 == Float.class || c1 == Float.TYPE || c1 == Double.class || c1 == Double.TYPE)
		{
			if(BigInteger.class.isAssignableFrom(c2)){
				return BigDecimal.class;
			}
		}	else{
			return Double.class;
		}
		
		if (c2 == Float.class || c2 == Float.TYPE || c2 == Double.class || c2 == Double.TYPE)
		{
			if(BigInteger.class.isAssignableFrom(c1)){
				return BigDecimal.class;
			}
		}	else{
			return Double.class;
		}
		
		if (BigInteger.class.isAssignableFrom(c1) || BigInteger.class.isAssignableFrom(c2)){
			return BigInteger.class;
		}
		
		return Long.class;
	}
	public static final Number add(ITypeConverter converter, Object o1, Object o2) throws Exception {
		if (o1 == null && o2 == null) {
			return LONG_ZERO;
		}
		if (o1 instanceof BigDecimal || o2 instanceof BigDecimal) {
			return converter.convert(o1, BigDecimal.class).add(converter.convert(o2, BigDecimal.class));
		}
		if (isFloatOrDouble(o1) || isFloatOrDouble(o2)) {
			if (o1 instanceof BigInteger || o2 instanceof BigInteger) {
				return converter.convert(o1, BigDecimal.class).add(converter.convert(o2, BigDecimal.class));
			}
			return converter.convert(o1, Double.class) + converter.convert(o2, Double.class);
		}
		if (o1 instanceof BigInteger || o2 instanceof BigInteger) {
			return converter.convert(o1, BigInteger.class).add(converter.convert(o2, BigInteger.class));
		}
		return converter.convert(o1, Long.class) + converter.convert(o2, Long.class);
	}

	public static Class<?> getSubReturnType(Class<?> c1, Class<?> c2)
	{
		return getAddReturnType(c1, c2);
	}
	public static final Number sub(ITypeConverter converter, Object o1, Object o2) throws Exception {
		if (o1 == null && o2 == null) {
			return LONG_ZERO;
		}
		if (o1 instanceof BigDecimal || o2 instanceof BigDecimal) {
			return converter.convert(o1, BigDecimal.class).subtract(converter.convert(o2, BigDecimal.class));
		}
		if (isFloatOrDouble(o1) || isFloatOrDouble(o2)) {
			if (o1 instanceof BigInteger || o2 instanceof BigInteger) {
				return converter.convert(o1, BigDecimal.class).subtract(converter.convert(o2, BigDecimal.class));
			}
			return converter.convert(o1, Double.class) - converter.convert(o2, Double.class);
		}
		if (o1 instanceof BigInteger || o2 instanceof BigInteger) {
			return converter.convert(o1, BigInteger.class).subtract(converter.convert(o2, BigInteger.class));
		}
		return converter.convert(o1, Long.class) - converter.convert(o2, Long.class);
	}

	public static Class<?> getMulReturnType(Class<?> c1, Class<?> c2)
	{
		return getAddReturnType(c1, c2);
	}
	public static final Number mul(ITypeConverter converter, Object o1, Object o2) throws Exception {
		if (o1 == null && o2 == null) {
			return LONG_ZERO;
		}
		if (o1 instanceof BigDecimal || o2 instanceof BigDecimal) {
			return converter.convert(o1, BigDecimal.class).multiply(converter.convert(o2, BigDecimal.class));
		}
		if (isFloatOrDouble(o1) || isFloatOrDouble(o2)) {
			if (o1 instanceof BigInteger || o2 instanceof BigInteger) {
				return converter.convert(o1, BigDecimal.class).multiply(converter.convert(o2, BigDecimal.class));
			}
			return converter.convert(o1, Double.class) * converter.convert(o2, Double.class);
		}
		if (o1 instanceof BigInteger || o2 instanceof BigInteger) {
			return converter.convert(o1, BigInteger.class).multiply(converter.convert(o2, BigInteger.class));
		}
		return converter.convert(o1, Long.class) * converter.convert(o2, Long.class);
	}

	public static Class<?> getDivReturnType(Class<?> c1, Class<?> c2)
	{
		if (BigDecimal.class.isAssignableFrom(c1) || BigDecimal.class.isAssignableFrom(c2)){
			return BigDecimal.class;
		}
		
		if (BigInteger.class.isAssignableFrom(c1) || BigInteger.class.isAssignableFrom(c2)){
			return BigDecimal.class;
		}
		
		return Double.class;
	}
	public static final Number div(ITypeConverter converter, Object o1, Object o2) throws Exception {
		if (o1 == null && o2 == null) {
			return LONG_ZERO;
		}
		if (isBigDecimalOrBigInteger(o1) || isBigDecimalOrBigInteger(o2)) {
			return converter.convert(o1, BigDecimal.class).divide(converter.convert(o2, BigDecimal.class), BigDecimal.ROUND_HALF_UP);
		}
		return converter.convert(o1, Double.class) / converter.convert(o2, Double.class);
	}
	public static Class<?> getModReturnType(Class<?> c1, Class<?> c2)
	{
		return getDivReturnType(c1, c2);
	}

	public static final Number mod(ITypeConverter converter, Object o1, Object o2) throws Exception {
		if (o1 == null && o2 == null) {
			return LONG_ZERO;
		}
		if (isBigDecimalOrFloatOrDouble(o1) || isBigDecimalOrFloatOrDouble(o2)) {
			return converter.convert(o1, Double.class) % converter.convert(o2, Double.class);
		}
		if (o1 instanceof BigInteger || o2 instanceof BigInteger) {
			return converter.convert(o1, BigInteger.class).remainder(converter.convert(o2, BigInteger.class));
		}
		return converter.convert(o1, Long.class) % converter.convert(o2, Long.class);
	}

	public static final Number neg(ITypeConverter converter, Object value) throws ConversionError {
		if (value == null) {
			return LONG_ZERO;
		}
		if (value instanceof BigDecimal) {
			return ((BigDecimal)value).negate();
		}
		if (value instanceof BigInteger) {
			return ((BigInteger)value).negate();
		}
		if (value instanceof Double) {
			return Double.valueOf(-((Double)value).doubleValue());
		}
		if (value instanceof Float) {
			return Float.valueOf(-((Float)value).floatValue());
		}
		/*
		if (value instanceof String) {
			if (isDotEe((String)value)) {
				return Double.valueOf(-converter.convert(value, Double.class).doubleValue());
			}
			return Long.valueOf(-converter.convert(value, Long.class).longValue());
		}
		*/
		if (value instanceof Long) {
			return Long.valueOf(-((Long)value).longValue());
		}
		if (value instanceof Integer) {
			return Integer.valueOf(-((Integer)value).intValue());
		}
		if (value instanceof Short) {
			return Short.valueOf((short)-((Short)value).shortValue());
		}
		if (value instanceof Byte) {
			return Byte.valueOf((byte)-((Byte)value).byteValue());
		}
		throw new ConversionError(LocalMessages.get("error.negate", value.getClass()));
	}
}
