/*
 * Copyright 2006, 2007 Odysseus Software GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.opensuit.cel.impl.tree.impl;

import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.COLON;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.COMMA;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.EMPTY;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.END_EVAL;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.EOF;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.FALSE;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.FLOAT;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.IDENTIFIER;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.INTEGER;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.LPAREN;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.MINUS;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.NOT;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.NULL;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.QUESTION;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.RBRACK;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.RPAREN;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.START_EVAL_DEFERRED;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.START_EVAL_DYNAMIC;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.STRING;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.TEXT;
import static org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol.TRUE;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.ow2.opensuit.cel.impl.misc.LocalMessages;
import org.ow2.opensuit.cel.impl.tree.IExprNode;
import org.ow2.opensuit.cel.impl.tree.impl.Scanner.ScanException;
import org.ow2.opensuit.cel.impl.tree.impl.Scanner.Symbol;
import org.ow2.opensuit.cel.impl.tree.impl.Scanner.Token;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstBinaryOperation;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstBoolean;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstBracket;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstChoice;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstComposite;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstDot;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstEval;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstFunction;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstIdentifier;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstMethod;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstNested;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstNode;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstNull;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstNumber;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstString;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstText;
import org.ow2.opensuit.cel.impl.tree.impl.ast.AstUnaryOperation;



/**
 * Handcrafted top-down parser.
 *
 * @author Christoph Beck
 */
public class Parser {
	/**
	 * Parse exception type
	 */
	@SuppressWarnings("serial")
	public static class ParseException extends Exception {
		final int position;
		final String encountered;
		final String expected;
		
		public ParseException(int position, String encountered, String expected) {
			super(LocalMessages.get("error.parse", position, encountered, expected));
			this.position = position;
			this.encountered = encountered;
			this.expected = expected;
		}
		public int getPosition()
		{
			return position;
		}
		public String getEncountered()
		{
			return encountered;
		}
		public String getExpected()
		{
			return expected;
		}
	}

	/**
	 * Token type (used to store lookahead)
	 */
	private static final class LookaheadToken {
		final Token token;
		final int position;

		LookaheadToken(Token token, int position) {
			this.token = token;
			this.position = position;
		}
	}

	public enum ExtensionPoint {
		OR,
		AND,
		EQ,
		CMP,
		ADD,
		MUL,
		UNARY,
		LITERAL
	}

	/**
	 * Provide limited support for syntax extensions.
	 */
	public static abstract class ExtensionHandler {
		private final ExtensionPoint point;
		
		public ExtensionHandler(ExtensionPoint point) {
			this.point = point;
		}

		/**
		 * @return the extension point specifying where this syntax extension is active
		 */
		public ExtensionPoint getExtensionPoint() {
			return point;
		}
		
		/**
		 * Called by the parser if it handles a extended token associated with this handler
		 * at the appropriate extension point.
		 * @param children
		 * @return abstract syntax tree node
		 */
		public abstract AstNode createAstNode(AstNode... children);
	}

	private static final String EXPR_FIRST =
		IDENTIFIER + "|" + 
		STRING + "|" + FLOAT + "|" + INTEGER + "|" + TRUE + "|" + FALSE + "|" + NULL + "|" +
		MINUS + "|" + NOT + "|" + EMPTY + "|" +
		LPAREN;
	
	protected final Scanner scanner;

	private List<LookaheadToken> lookahead = Collections.emptyList();

	private Token token; // current token
	private int position;// current token's position
	
	protected Map<Scanner.ExtensionToken, ExtensionHandler> extensions = Collections.emptyMap();

	public Parser(String input) {
		this.scanner = createScanner(input);
	}

	protected Scanner createScanner(String expression) {
		return new Scanner(expression);
	}

	public void putExtensionHandler(Scanner.ExtensionToken token, ExtensionHandler extension) {
		if (extensions.isEmpty()) {
			extensions = new HashMap<Scanner.ExtensionToken, ExtensionHandler>(16);
		}
		extensions.put(token, extension);
	}
	
	protected ExtensionHandler getExtensionHandler(Token token) {
		return extensions.get(token);
	}
	
	/**
	 * Parse an integer literal.
	 * @param string string to parse
	 * @return <code>Long.valueOf(string)</code>
	 */
	protected Number parseInteger(String string) throws ParseException {
		try {
			return Integer.valueOf(string);
		} catch (NumberFormatException e) {
			try {
				return Long.valueOf(string);
			} catch (NumberFormatException e2) {
				fail(INTEGER);
				return null;
			}
		}
	}
	
	/**
	 * Parse a floating point literal.
	 * @param string string to parse
	 * @return <code>Double.valueOf(string)</code>
	 */
	protected Number parseFloat(String string) throws ParseException {
		try {
			return Float.valueOf(string);
		} catch (NumberFormatException e) {
			try {
				return Double.valueOf(string);
			} catch (NumberFormatException e2) {
				fail(FLOAT);
				return null;
			}
		}
	}

	protected AstBinaryOperation createAstBinary(AstNode left, AstNode right, AstBinaryOperation.Operator operator) {
		return new AstBinaryOperation(position, left, right, operator);
	}
	
	protected AstBracket createAstBracket(AstNode base, AstNode property, boolean lvalue) {
		return new AstBracket(position, base, property, lvalue);
	}
	
	protected AstChoice createAstChoice(AstNode question, AstNode yes, AstNode no) {
		return new AstChoice(position, question, yes, no);
	}
	
	protected AstComposite createAstComposite(List<AstNode> nodes) {
		return new AstComposite(position, nodes);
	}
	
	protected AstDot createAstDot(AstNode base, String property, boolean lvalue) {
		return new AstDot(position, base, property, lvalue);
	}
	
	protected AstFunction createAstFunction(String name, List<AstNode> args) {
		return new AstFunction(position, name, args);
	}

	protected AstIdentifier createAstIdentifier(String name) {
		return new AstIdentifier(position, name);
	}

	protected AstMethod createAstMethod(AstNode prefix, String name, List<AstNode> nodes) {
		return new AstMethod(position, prefix, name, nodes);
	}
	
	protected AstUnaryOperation createAstUnary(AstNode child, AstUnaryOperation.Operator operator) {
		return new AstUnaryOperation(position, child, operator);
	}

	protected final Token getToken() {
		return token;
	}

	/**
	 * throw exception
	 */
	protected void fail(String expected) throws ParseException {
		throw new ParseException(position, "'" + token.getImage() + "'", expected);
	}

	/**
	 * throw exception
	 */
	protected void fail(Symbol expected) throws ParseException {
		fail(expected.toString());
	}

	/**
	 * get lookahead symbol.
	 */
	protected final Token lookahead(int index) throws ScanException, ParseException {
		if (lookahead.isEmpty()) {
			lookahead = new LinkedList<LookaheadToken>();
		}
		while (index >= lookahead.size()) {
			lookahead.add(new LookaheadToken(scanner.next(), scanner.getPosition()));
		}
		return lookahead.get(index).token;
	}

	/**
	 * consume current token (get next token).
	 * @return the consumed token (which was the current token when calling this method)
	 */
	protected final Token consumeToken() throws ScanException, ParseException {
		Token result = token;
		if (lookahead.isEmpty()) {
			token = scanner.next();
			position = scanner.getPosition();
		} else {
			LookaheadToken next = lookahead.remove(0);
			token = next.token;
			position = next.position;
		}
		return result;
	}

	/**
	 * consume current token (get next token); throw exception if the current token doesn't
	 * match the expected symbol.
	 */
	protected final Token consumeToken(Symbol expected) throws ScanException, ParseException {
		if (token.getSymbol() != expected) {
			fail(expected);
		}
		return consumeToken();
	}
	
	/**
	 * tree := text? ((dynamic text?)+ | (deferred text?)+)? 
	 */
	public IExprNode parse() throws ScanException, ParseException {
		consumeToken();
		AstNode t = text();
		if (token.getSymbol() == EOF) {
			if (t == null) {
				t = new AstText(position, "");
			}
			return t;
		}
		AstEval e = eval();
		if (token.getSymbol() == EOF && t == null) {
			return e;
		}
		ArrayList<AstNode> list = new ArrayList<AstNode>();
		if (t != null) {
			list.add(t);
		}
		list.add(e);
		t = text();
		if (t != null) {
			list.add(t);
		}
		while (token.getSymbol() != EOF) {
			if (e.isDeferred()) {
				list.add(eval(true, true));
			} else {
				list.add(eval(true, false));
			}
			t = text();
			if (t != null) {
				list.add(t);
			}
		}
		return createAstComposite(list);
	}

	/**
	 * text := <TEXT>
	 */
	protected AstNode text() throws ScanException, ParseException {
		AstNode v = null;
		if (token.getSymbol() == TEXT) {
			v = new AstText(position, token.getImage());
			consumeToken();
		}
		return v;
	}

	/**
	 * eval := dynamic | deferred
	 */
	protected AstEval eval() throws ScanException, ParseException {
		AstEval e = eval(false, false);
		if (e == null) {
			e = eval(false, true);
			if (e == null) {
				fail(START_EVAL_DEFERRED + "|" + START_EVAL_DYNAMIC);
			}
		}
		return e;
	}

	/**
	 * dynmamic := <START_EVAL_DYNAMIC> expr <END_EVAL>
	 * deferred := <START_EVAL_DEFERRED> expr <END_EVAL>
	 */
	protected AstEval eval(boolean required, boolean deferred) throws ScanException, ParseException {
		AstEval v = null;
		Symbol start_eval = deferred ? START_EVAL_DEFERRED : START_EVAL_DYNAMIC;
		if (token.getSymbol() == start_eval) {
			consumeToken();
			v = new AstEval(position, expr(true), deferred);
			consumeToken(END_EVAL);
		} else if (required) {
			fail(start_eval);
		}
		return v;
	}

	/**
	 * expr := or (<QUESTION> expr <COLON> expr)?
	 */
	protected AstNode expr(boolean required) throws ScanException, ParseException {
		AstNode v = or(required);
		if (v == null) {
			return null;
		}
		if (token.getSymbol() == QUESTION) {
			consumeToken();
			AstNode a = expr(true);
			consumeToken(COLON);
			AstNode b = expr(true);
			v = createAstChoice(v, a, b);
		}
		return v;
	}

	/**
	 * or := and (<OR> and)*
	 */
	protected AstNode or(boolean required) throws ScanException, ParseException {
		AstNode v = and(required);
		if (v == null) {
			return null;
		}
		while (true) {
			switch (token.getSymbol()) {
				case OR:
					consumeToken();
					v = createAstBinary(v, and(true), AstBinaryOperation.OR);
					break;
				case EXTENSION:
					if (getExtensionHandler(token).getExtensionPoint() == ExtensionPoint.OR) {
						v = getExtensionHandler(consumeToken()).createAstNode(v, and(true));
						break;
					}
				default:
					return v;
			}
		}
	}

	/*
	 * and := eq (<AND> eq)*
	 */
	protected AstNode and(boolean required) throws ScanException, ParseException {
		AstNode v = eq(required);
		if (v == null) {
			return null;
		}
		while (true) {
			switch (token.getSymbol()) {
				case AND:
					consumeToken();
					v = createAstBinary(v, eq(true), AstBinaryOperation.AND);
					break;
				case EXTENSION:
					if (getExtensionHandler(token).getExtensionPoint() == ExtensionPoint.AND) {
						v = getExtensionHandler(consumeToken()).createAstNode(v, eq(true));
						break;
					}
				default:
					return v;
			}
		}
	}

	/**
	 * eq := cmp (<EQ> cmp | <NE> cmp)*
	 */
	protected AstNode eq(boolean required) throws ScanException, ParseException {
		AstNode v = cmp(required);
		if (v == null) {
			return null;
		}
		while (true) {
			switch (token.getSymbol()) {
				case EQ:
					consumeToken();
					v = createAstBinary(v, cmp(true), AstBinaryOperation.EQ);
					break;
				case NE:
					consumeToken();
					v = createAstBinary(v, cmp(true), AstBinaryOperation.NE);
					break;
				case EXTENSION:
					if (getExtensionHandler(token).getExtensionPoint() == ExtensionPoint.EQ) {
						v = getExtensionHandler(consumeToken()).createAstNode(v, cmp(true));
						break;
					}
				default:
					return v;
			}
		}
	}
	
	/**
	 * cmp := add (<LT> add | <LE> add | <GE> add | <GT> add)*
	 */
	protected AstNode cmp(boolean required) throws ScanException, ParseException {
		AstNode v = add(required);
		if (v == null) {
			return null;
		}
		while (true) {
			switch (token.getSymbol()) {
				case LT:
					consumeToken();
					v = createAstBinary(v, add(true), AstBinaryOperation.LT);
					break;
				case LE:
					consumeToken();
					v = createAstBinary(v, add(true), AstBinaryOperation.LE);
					break;
				case GE:
					consumeToken();
					v = createAstBinary(v, add(true), AstBinaryOperation.GE);
					break;
				case GT:
					consumeToken();
					v = createAstBinary(v, add(true), AstBinaryOperation.GT);
					break;
				case EXTENSION:
					if (getExtensionHandler(token).getExtensionPoint() == ExtensionPoint.CMP) {
						v = getExtensionHandler(consumeToken()).createAstNode(v, add(true));
						break;
					}
				default:
					return v;
			}
		}
	}

	/**
	 * add := add (<PLUS> mul | <MINUS> mul)*
	 */
	protected AstNode add(boolean required) throws ScanException, ParseException {
		AstNode v = mul(required);
		if (v == null) {
			return null;
		}
		while (true) {
			switch (token.getSymbol()) {
				case PLUS:
					consumeToken();
					v = createAstBinary(v, mul(true), AstBinaryOperation.ADD);
					break;
				case MINUS:
					consumeToken();
					v = createAstBinary(v, mul(true), AstBinaryOperation.SUB);
					break;
				case EXTENSION:
					if (getExtensionHandler(token).getExtensionPoint() == ExtensionPoint.ADD) {
						v = getExtensionHandler(consumeToken()).createAstNode(v, mul(true));
						break;
					}
				default:
					return v;
			}
		}
	}

	/**
	 * mul := unary (<MUL> unary | <DIV> unary | <MOD> unary)*
	 */
	protected AstNode mul(boolean required) throws ScanException, ParseException {
		AstNode v = unary(required);
		if (v == null) {
			return null;
		}
		while (true) {
			switch (token.getSymbol()) {
				case MUL:
					consumeToken();
					v = createAstBinary(v, unary(true), AstBinaryOperation.MUL);
					break;
				case DIV:
					consumeToken();
					v = createAstBinary(v, unary(true), AstBinaryOperation.DIV);
					break;
				case MOD:
					consumeToken();
					v = createAstBinary(v, unary(true), AstBinaryOperation.MOD);
					break;
				case EXTENSION:
					if (getExtensionHandler(token).getExtensionPoint() == ExtensionPoint.MUL) {
						v = getExtensionHandler(consumeToken()).createAstNode(v, unary(true));
						break;
					}
				default:
					return v;
			}
		}
	}

	/**
	 * unary := <NOT> unary | <MINUS> unary | <EMPTY> unary | value
	 */
	protected AstNode unary(boolean required) throws ScanException, ParseException {
		AstNode v = null;
		switch (token.getSymbol()) {
			case NOT:
				consumeToken();
				v = createAstUnary(unary(true), AstUnaryOperation.NOT);
				break;
			case MINUS:
				consumeToken();
				v = createAstUnary(unary(true), AstUnaryOperation.NEG);
				break;
			case EMPTY:
				consumeToken();
				v = createAstUnary(unary(true), AstUnaryOperation.EMPTY);
				break;
			case EXTENSION:
				if (getExtensionHandler(token).getExtensionPoint() == ExtensionPoint.UNARY) {
					v = getExtensionHandler(consumeToken()).createAstNode(unary(true));
					break;
				}
			default:
				v = value();
		}
		if (v == null && required) {
			fail(EXPR_FIRST);
		}
		return v;
	}

	/**
	 * value := (nonliteral | literal) (<DOT> <IDENTIFIER> | <LBRACK> expr <RBRACK>)*
	 */
	protected AstNode value() throws ScanException, ParseException {
		boolean lvalue = true;
		AstNode v = nonliteral();
		if (v == null) {
			v = literal();
			if (v == null) {
				return null;
			}
			lvalue = false;
		}
		while (true) {
			switch (token.getSymbol()) {
				case DOT:
					consumeToken();
					String name = consumeToken(IDENTIFIER).getImage();
					if (token.getSymbol() == LPAREN) {
						consumeToken();
						v = createAstMethod(v, name, list());
						consumeToken(RPAREN);
					} else {
						v = createAstDot(v, name, lvalue);
					}
					break;
				case LBRACK:
					consumeToken();
					AstNode property = expr(true);
					v = createAstBracket(v, property, lvalue);
					consumeToken(RBRACK);
					break;
				default:
					return v;
			}
		}
	}

	/**
	 * nonliteral := <IDENTIFIER> | function | <LPAREN> expr <RPAREN>
	 * function   := (<IDENTIFIER> <COLON>)? <IDENTIFIER> <LPAREN> list? <RPAREN>
	 */
	protected AstNode nonliteral() throws ScanException, ParseException {
		AstNode v = null;
		switch (token.getSymbol()) {
			case IDENTIFIER:
				String name = consumeToken().getImage();
				if (token.getSymbol() == COLON && lookahead(0).getSymbol() == IDENTIFIER && lookahead(1).getSymbol() == LPAREN) { // ns:f(...)
					consumeToken();
					name += ":" + token.getImage();
					consumeToken();
				}
				if (token.getSymbol() == LPAREN) { // function
					consumeToken();
					List<AstNode> args = list();
					consumeToken(RPAREN);
					v = function(name, args);
				} else { // identifier
					v = identifier(name);
				}
				break;
			case LPAREN:
				consumeToken();
				v = expr(true);
				consumeToken(RPAREN);
				v = new AstNested(position, v);
				break;
		}
		return v;
	}

	/**
	 * list := expr (<COMMA> expr)*
	 */
	protected List<AstNode> list() throws ScanException, ParseException {
		List<AstNode> l = Collections.emptyList();
		AstNode v = expr(false);
		if (v != null) {
			l = new ArrayList<AstNode>();
			l.add(v);
			while (token.getSymbol() == COMMA) {
				consumeToken();
				l.add(expr(true));
			}
		}
		return l;
	}
	
	/**
	 * literal := <TRUE> | <FALSE> | <STRING> | <INTEGER> | <FLOAT> | <NULL>
	 */
	protected AstNode literal() throws ScanException, ParseException {
		AstNode v = null;
		switch (token.getSymbol()) {
			case TRUE:
				v = new AstBoolean(position, true);
				consumeToken();
				break;
			case FALSE:
				v = new AstBoolean(position, false);
				consumeToken();
				break;
			case STRING:
				v = new AstString(position, token.getImage());
				consumeToken();
				break;
			case INTEGER:
				v = new AstNumber(position, parseInteger(token.getImage()));
				consumeToken();
				break;
			case FLOAT:
				v = new AstNumber(position, parseFloat(token.getImage()));
				consumeToken();
				break;			
			case NULL:
				v = new AstNull(position);
				consumeToken();
				break;
			case EXTENSION:
				if (getExtensionHandler(token).getExtensionPoint() == ExtensionPoint.LITERAL) {
					v = getExtensionHandler(consumeToken()).createAstNode();
					break;
				}
		}
		return v;
	}

	protected final AstFunction function(String name, List<AstNode> args) {
		AstFunction function = createAstFunction(name, args);
		return function;
	}
	
	protected final AstIdentifier identifier(String name) {
		AstIdentifier identifier = createAstIdentifier(name);
		return identifier;
	}
}
