package org.ow2.opensuit.cel.impl.tree.impl.ast;

import java.lang.reflect.Type;

import org.ow2.opensuit.cel.ICompilationContext;
import org.ow2.opensuit.cel.ICompilationMessage;
import org.ow2.opensuit.cel.IEvaluationContext;
import org.ow2.opensuit.cel.ITypeConverter;
import org.ow2.opensuit.cel.impl.ICompilationResultWriter;
import org.ow2.opensuit.cel.impl.misc.BooleanOperations;
import org.ow2.opensuit.cel.impl.misc.NumberOperations;
import org.ow2.opensuit.cel.impl.misc.RegExpBuilder;
import org.ow2.opensuit.cel.impl.tree.ExpressionEvaluationException;



public class AstBinaryOperation extends AstRightValue
{
	private ITypeConverter converter;
	private final Operator operator;
	private final AstNode left, right;
	private Class<?> resultType;

	public AstBinaryOperation(int position, AstNode left, AstNode right, Operator operator)
	{
		super(position);
		this.left = left;
		this.right = right;
		this.operator = operator;
	}

	public boolean compile(ITypeConverter converter, ICompilationContext ctx, ICompilationResultWriter messages)
	{
		this.converter = converter;
		
		// 1: compile children
		boolean compiled = left.compile(converter, ctx, messages);
		compiled = right.compile(converter, ctx, messages) && compiled;
		
		if(!compiled){
			return false;
		}
		
		// 2: compile this (check type)
		if(!operator.checkTypes(converter, left.getType(), right.getType()))
		{
			// TODO: more appropriate message
			messages.addMessage(this, ICompilationMessage.ERROR_LEVEL, "Invalid operand types");
			return false;
		}
		resultType = operator.getType(left.getType(), right.getType());		
		return true;
	}

	public Class<?> getType()
	{
		return resultType;
	}

	public Type getGenericType()
	{
		return resultType;
	}

	public boolean isStaticValue()
	{
		return left.isStaticValue() && right.isStaticValue();
	}

	public Operator getOperator()
	{
		return operator;
	}

	public Object invoke(IEvaluationContext context) throws Exception
	{
		try
		{
			return operator.apply(converter, left.invoke(context), right.invoke(context));
		}
		catch (Exception e)
		{
			throw new ExpressionEvaluationException(this, e);
		}
	}

	@Override
	protected void appendExpressionString(StringBuilder b)
	{
		left.appendExpressionString(b);
		b.append(' ');
		b.append(operator);
		b.append(' ');
		right.appendExpressionString(b);
	}

	@Override
	protected void appendValuePattern(StringBuilder builder)
	{
		RegExpBuilder.appendTypeExpr(builder, getGenericType());
	}
	
	// ====================================================================================
	// === Operator implementations
	// ====================================================================================
	public interface Operator
	{
		public boolean checkTypes(ITypeConverter converter, Class<?> t1, Class<?> t2);
		
		public Class<?> getType(Class<?> t1, Class<?> t2);

		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception;
	}

	public static abstract class BooleanOperator implements Operator
	{
		/*
		public boolean checkTypes(Class t1, Class t2)
		{
			return TypeConverter.DEFAULT.isConvertible(t1, Boolean.class) && TypeConverter.DEFAULT.isConvertible(t2, Boolean.class);
		}
		*/
		public Class<?> getType(Class<?> arg0, Class<?> arg1)
		{
			return Boolean.TYPE;
		}
	}
	public static abstract class ComparisionOperator implements Operator
	{
		public boolean checkTypes(ITypeConverter converter, Class<?> t1, Class<?> t2)
		{
			return BooleanOperations.checkComparisonTypes(converter, t1, t2);
		}
		public Class<?> getType(Class<?> arg0, Class<?> arg1)
		{
			return Boolean.TYPE;
		}
	}
	
	public static abstract class NumberOperator implements Operator
	{
		public boolean checkTypes(ITypeConverter converter, Class<?> t1, Class<?> t2)
		{
			return NumberOperations.checkTypes(converter, t1, t2);
		}
	}

	public static final Operator ADD = new NumberOperator()
	{
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return NumberOperations.add(converter, o1, o2);
		}

		public Class<?> getType(Class<?> t1, Class<?> t2)
		{
			return NumberOperations.getAddReturnType(t1, t2);
		}

		@Override
		public String toString()
		{
			return "+";
		}
	};

	public static final Operator AND = new BooleanOperator()
	{
		public boolean checkTypes(ITypeConverter converter, Class<?> t1, Class<?> t2)
		{
			return BooleanOperations.checkBooleanTypes(converter, t1, t2);
		}
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return BooleanOperations.and(converter, o1, o2);
		}

		@Override
		public String toString()
		{
			return "&&";
		}
	};

	public static final Operator DIV = new NumberOperator()
	{
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return NumberOperations.div(converter, o1, o2);
		}

		public Class<?> getType(Class<?> t1, Class<?> t2)
		{
			return NumberOperations.getDivReturnType(t1, t2);
		}

		@Override
		public String toString()
		{
			return "/";
		}
	};

	public static final Operator EQ = new BooleanOperator()
	{
		public boolean checkTypes(ITypeConverter converter, Class<?> t1, Class<?> t2)
		{
			return true;
		}
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return BooleanOperations.eq(converter, o1, o2);
		}

		@Override
		public String toString()
		{
			return "==";
		}
	};

	public static final Operator GE = new ComparisionOperator()
	{
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return BooleanOperations.ge(converter, o1, o2);
		}

		@Override
		public String toString()
		{
			return ">=";
		}
	};

	public static final Operator GT = new ComparisionOperator()
	{
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return BooleanOperations.gt(converter, o1, o2);
		}

		@Override
		public String toString()
		{
			return ">";
		}
	};

	public static final Operator LE = new ComparisionOperator()
	{
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return BooleanOperations.le(converter, o1, o2);
		}

		@Override
		public String toString()
		{
			return "<=";
		}
	};

	public static final Operator LT = new ComparisionOperator()
	{
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return BooleanOperations.lt(converter, o1, o2);
		}

		@Override
		public String toString()
		{
			return "<";
		}
	};

	public static final Operator MOD = new NumberOperator()
	{
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return NumberOperations.mod(converter, o1, o2);
		}

		public Class<?> getType(Class<?> t1, Class<?> t2)
		{
			return NumberOperations.getModReturnType(t1, t2);
		}

		@Override
		public String toString()
		{
			return "%";
		}
	};

	public static final Operator MUL = new NumberOperator()
	{
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return NumberOperations.mul(converter, o1, o2);
		}

		public Class<?> getType(Class<?> t1, Class<?> t2)
		{
			return NumberOperations.getMulReturnType(t1, t2);
		}

		@Override
		public String toString()
		{
			return "*";
		}
	};

	public static final Operator NE = new BooleanOperator()
	{
		public boolean checkTypes(ITypeConverter converter, Class<?> t1, Class<?> t2)
		{
			return true;
		}
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return BooleanOperations.ne(converter, o1, o2);
		}

		@Override
		public String toString()
		{
			return "!=";
		}
	};

	public static final Operator OR = new BooleanOperator()
	{
		public boolean checkTypes(ITypeConverter converter, Class<?> t1, Class<?> t2)
		{
			return BooleanOperations.checkBooleanTypes(converter, t1, t2);
		}
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return BooleanOperations.or(converter, o1, o2);
		}

		@Override
		public String toString()
		{
			return "||";
		}
	};

	public static final Operator SUB = new NumberOperator()
	{
		public Object apply(ITypeConverter converter, Object o1, Object o2) throws Exception
		{
			return NumberOperations.sub(converter, o1, o2);
		}

		public Class<?> getType(Class<?> t1, Class<?> t2)
		{
			return NumberOperations.getSubReturnType(t1, t2);
		}

		@Override
		public String toString()
		{
			return "-";
		}
	};
	
}
